/******************************************************************************
 The computer software and associated documentation called DOMAK hereinafter
 referred to as the WORK which is more particularly identified and described in
 Appendix A of the file LICENSE.  Conditions and restrictions for use of
 this package are also in this file.

 This routine was written by Asim S. Siddiqui

 The WORK was developed by:
        Asim S. Siddiqui and Geoffrey J. Barton
        Laboratory of Molecular Biophysics
        University of Oxford
        Rex Richards Building
        South Parks Road
        Oxford OX1 3QU U.K.
        Tel:  (+44) 865-275379
        FAX:  (+44) 865-510454
        INTERNET: as@bioch.ox.ac.uk
        JANET:    as@uk.ac.ox.bioch

 The WORK is Copyright (1995) University of Oxford
        Administrative Offices
        Wellington Square
        Oxford OX1 2JD U.K.

 All use of the WORK must cite:
 Siddiqui, A. S. and Barton, G. J., "Continuous and Discontinuous Domains: An
 Algorithm for the Automatic Generation of Reliable Protein Domain Definitions" 
 PROTEIN SCIENCE, 4:872-884 (1995).
*****************************************************************************/

/*
 * Title
 *    asd_make_domains.c
 * Purpose
 *    contains the algorithm to work out domains
 * Author
 *    Asim Siddiqui
 * SccsId
 *    %W%   %U%   %E%
 */

#include <rdssp.h>
#include <stdio.h>
#include <local.h>
#include <asd_structs.h>
#include <asm_mop.h>
#include <ase_error.h>
#include <asstr_util.h>
#include <ass_stamp_utils.h>
#include <asd_make_domains.h>
#include <asd_make_domains_2.h>
#include <asd_glob_interface.h>
#include <asd_mds_utils1.h>
#include <asd_utils.h>
#include <asd_value.h>
#include <asd_check_utils.h>

extern Asd_Parameters params;

/*
 * Name:
 *    asd_make_multiple_domains
 * Purpose
 *    acts as a top level for CONTINUOUS segements
 */
Asd_Domain_List *
asd_make_multiple_domains(Asd_Domain *d_by_choice, int *num_d_by_choice,
                          Asd_Domain *d_to_split, int num_atoms,
                          Asd_Contact_Info **contact_info,
                          Asd_Contact_Info **contact_rev_info,
                          struct brookn *bn, int start_chain, int end_chain,
                          char *ss)
{
    int increment;            /* testing increment            */
    int i;                    /* loop counter                 */
    int s1_s;                 /* start of segment 1           */
    int s1_e;                 /* end   of segment 1           */
    int s2_s;                 /* start of segment 2           */
    int s2_e;                 /* end   of segment 2           */
    int s1_sn;                /* start of segment 1           */
    int s1_en;                /* end   of segment 1           */
    int s2_sn;                /* start of segment 2           */
    int s2_en;                /* end   of segment 2           */
    int s1_s_lim;             /* s1 start limit               */
    int s1_e_lim;             /* s1 end limit                 */
    int s2_s_lim;             /* s2 start limit               */
    int s2_e_lim;             /* s2 end limit                 */
    int flag;                 /* boolean flag                 */
    int level_f;              /* boolean flag                 */
    int level_f2;             /* boolean flag                 */
    int total_contacts;       /* total number of contacts     */
    bool_t s_ok;              /* is single seg ok to split    */
    bool_t m_ok;              /* is multiple seg ok to split  */
    bool_t ss_only;           /* whether to use ss only       */
    Asd_Seg_Cont s_conts;     /* segment contacts             */
    Asd_Domain *max_domain_m; /* most likely multi domain     */
    Asd_Domain *max_domain_s; /* most likely single domain    */
    Asd_Domain *d_left_over;  /* d's left over to be analysed */
    Asd_Domain_List *d_list1; /* list of domains divided up   */ 
    Asd_Domain_List *d_list2; /* list of domains divided up   */ 
    Asd_Domain_List *d_list3; /* list of domains divided up   */ 
    Asd_Domain_List *d_list4; /* list of domains divided up   */ 
    Asd_Contact_List *c_list; /* list of contacts             */

    d_list1 = NULL;
    d_list2 = NULL;
    d_list3 = NULL;
    d_list4 = NULL;
    d_to_split->type = 1;
    d_to_split->ok = TRUE;
    asd_d_comments(6, d_to_split, bn);
    if (((params.MIN_SEGMENT_SIZE_MID * 2) + params.MIN_SEGMENT_SIZE_END) >
       (params.MIN_DOMAIN_SIZE * 2)) {
        ase_error_fatal("asd_make_multiple_domains", "error in #defines");
    } /*if*/
    if ((d_to_split->end1 - d_to_split->start1 + 1) < params.MIN_DOMAIN_SIZE) {
        d_list1 = (Asd_Domain_List *) asm_malloc(sizeof(Asd_Domain_List));
        d_list1->n_b_left_over = 1;
        d_list1->n_d_list = 0;
        d_list1->bits_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
        d_list1->bits_left_over->type = 1;
        d_list1->bits_left_over->d_assigned = -1;
        d_list1->bits_left_over->start1 = d_to_split->start1;
        d_list1->bits_left_over->end1 = d_to_split->end1;
        d_list1->bits_left_over->glob = LARGE_NEG_GLOB;
        d_list1->bits_left_over->id = asstr_save(d_to_split->id);
        d_list1->bits_left_over->ok = TRUE;
        d_list1->d_split_up = d_to_split;
        asd_d_comments(1, d_list1->bits_left_over, bn);
        return(d_list1);
    } /*if*/
    if ((d_to_split->end1 - d_to_split->start1 + 1) <
                                 (params.MIN_DOMAIN_SIZE * 2)) {
        d_list1 = (Asd_Domain_List *) asm_malloc(sizeof(Asd_Domain_List));
        d_list1->n_d_list = 1;
        d_list1->n_b_left_over = 0;
        d_list1->domains = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
        d_list1->domains->type = 1;
        d_list1->domains->start1 = d_to_split->start1;
        d_list1->domains->end1 = d_to_split->end1;
        d_list1->domains->glob = LARGE_NEG_GLOB;
        d_list1->d_split_up = d_to_split;
        d_list1->domains->id = asstr_save(d_to_split->id);
        d_list1->domains->ok = TRUE;
        asd_d_comments(2, d_list1->domains, bn);
        return(d_list1);
    } else if ((d_to_split->end1 - d_to_split->start1 + 1) < params.MIN_DOUBLE_SPLIT) {
        return(asd_make_single_split(d_by_choice, num_d_by_choice, d_to_split,
                                    num_atoms, contact_info, contact_rev_info,
                                     bn, start_chain, end_chain, ss, TRUE));
    } /*if*/


    ss_only = asd_ss_per(ss, d_to_split, &increment);
    c_list = asd_total_contacts(d_to_split, ss, ss_only, contact_info,
                                        contact_rev_info, bn, &total_contacts);
/* set up start and end positions and pointers to arrays */
    s1_s = d_to_split->start1;
    if (s1_s == start_chain) {
        s1_e = s1_s - 1 + params.MIN_SEGMENT_SIZE_END;
        s2_s = s1_e + params.MIN_SEGMENT_SIZE_MID + 1;
        if ((params.MIN_DOMAIN_SIZE - params.MIN_SEGMENT_SIZE_END) < params.MIN_SEGMENT_SIZE_MID) {
            s2_e = s2_s + params.MIN_SEGMENT_SIZE_MID - 1;
        } else {
            s2_e = s2_s + (params.MIN_DOMAIN_SIZE - params.MIN_SEGMENT_SIZE_END) - 1;
        } /*if*/
    } else {
        s1_e = params.MIN_SEGMENT_SIZE_MID + s1_s - 1;
        s2_s = s1_e + params.MIN_SEGMENT_SIZE_MID + 1;
        if ((params.MIN_SEGMENT_SIZE_MID * 2) > params.MIN_DOMAIN_SIZE) {
            s2_e = s2_s + params.MIN_SEGMENT_SIZE_MID - 1;
        } else {
            s2_e = s2_s + (params.MIN_DOMAIN_SIZE - params.MIN_SEGMENT_SIZE_MID) - 1;
        } /*if*/
    } /*if*/

    s1_sn = s1_s;
    s1_en = s1_e;
    s2_sn = s2_s;
    s2_en = s2_e;
    max_domain_m = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
    max_domain_m->value = -1.0;
    max_domain_m->type = 2;
    if (d_to_split->end1 == end_chain) {
        s1_s_lim = d_to_split->end1 + 1 - params.MIN_SEGMENT_SIZE_END -
                  (2 * params.MIN_SEGMENT_SIZE_MID);
        s1_e_lim = d_to_split->end1 - params.MIN_SEGMENT_SIZE_END -
                       params.MIN_SEGMENT_SIZE_MID;
        s2_s_lim = d_to_split->end1 + 1 - params.MIN_SEGMENT_SIZE_END;
    } else {
        s1_s_lim = d_to_split->end1 + 1 - params.MIN_SEGMENT_SIZE_MID -
                  (2 * params.MIN_SEGMENT_SIZE_MID);
        s1_e_lim = d_to_split->end1 - params.MIN_SEGMENT_SIZE_MID -
                       params.MIN_SEGMENT_SIZE_MID;
        s2_s_lim = d_to_split->end1 + 1 - params.MIN_SEGMENT_SIZE_MID;
    } /*if*/
    s2_e_lim = d_to_split->end1;
/* now add one to all the limits so only have to have < sign not <= */
    s1_s_lim++;
    s1_e_lim++;
    s2_s_lim++;
    s2_e_lim++;
       
    while (s1_sn < s1_s_lim) {
        level_f = asd_check_seg_a(c_list, ss_only, ss, d_to_split, s1_sn,
                                  params.MIN_SEGMENT_SIZE_MID, start_chain, 2);
        while (s1_en < s1_e_lim && level_f == 0) {
            level_f2 = asd_check_seg_1(c_list, ss_only, ss, d_to_split,
                                      s1_sn, s1_en);
            while (s2_sn < s2_s_lim && level_f2 == 0) {
                flag = asd_calc_res_from_scratch_m(contact_info,
                                        contact_rev_info,
                                        d_to_split, max_domain_m,
                                        &s_conts, ss_only, ss,
                                        start_chain, end_chain,
                                        s1_sn, s1_en, s2_sn, s2_en, bn, c_list);
 
                s1_s = s1_sn;
                s1_e = s1_en;
                s2_s = s2_sn;
                s2_e = s2_en;
                while (s2_en < s2_e_lim && flag !=2) {
                    flag = asd_calc_res_given_m(contact_info, contact_rev_info,
                                       d_to_split, max_domain_m, &s_conts,
                                       ss_only, ss, start_chain, end_chain,
                                       s1_sn,  s1_en,  s2_sn,  s2_en,
                                       s1_s,   s1_e,   s2_s,   s2_e,
                                       total_contacts, bn, c_list, increment);
                    if (flag == 0) {
                        s2_e = s2_en;
                    } /*if*/
                    if (s2_en == d_to_split->end1) {
                        s2_en = s2_en + increment;
                    } else {
                        if (ss_only) {
                            i = s2_en - 1;
                            i++;
                            while (ss[i] != 'H' && ss[i] != 'E' &&
                                   i < d_to_split->end1) {
                                i++;
                            } /*while*/
                            if (i == s2_en) {
                                s2_en = s2_en + increment;
                            } else {
                                s2_en = i;
                            } /*if*/
                        } else {
                            s2_en = s2_en + increment;
                        } /*if*/
                        if (s2_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_END) &&
                            d_to_split->end1 == end_chain) {
                            s2_en = d_to_split->end1;
                        } else if (s2_en >
                                  (d_to_split->end1 - params.MIN_SEGMENT_SIZE_MID) &&
                                   d_to_split->end1 != end_chain) {
                            s2_en = d_to_split->end1;
                        } /*if*/
                    } /*if*/
                } /*while*/
                s2_s = s2_sn;
                if (ss_only) {
                    i = s2_sn - 1;
                    i++;
                    while (ss[i] != 'H' && ss[i] != 'E' &&
                           i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    if (i == s2_sn) {
                        s2_sn = s2_sn + increment;
                    } else {
                        s2_sn = i;
                    } /*if*/
                } else {
                    s2_sn = s2_sn + increment;
                } /*if*/
                s2_en = s2_sn + params.MIN_SEGMENT_SIZE_MID - 1;
                if (s2_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_END) &&
                             d_to_split->end1 == end_chain) {
                    s2_en = d_to_split->end1;
                } else if (s2_en >
                          (d_to_split->end1 - params.MIN_SEGMENT_SIZE_MID) &&
                           d_to_split->end1 != end_chain) {
                    s2_en = d_to_split->end1;
                } /*if*/
            } /*while*/
            s1_e = s1_en;
            if (ss_only) {
                i = s1_en - 1;
                i++;
                while (ss[i] != 'H' && ss[i] != 'E' &&
                       i < d_to_split->end1) {
                    i++;
                } /*while*/
                if (i == s1_en) {
                    s1_en = s1_en + increment;
                } else {
                    s1_en = i;
                } /*if*/
            } else {
                s1_en = s1_en + increment;
            } /*if*/
            s2_sn = s1_en + params.MIN_SEGMENT_SIZE_MID + 1;
            s2_en = s2_sn + params.MIN_SEGMENT_SIZE_MID - 1;
            if (s2_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_END) &&
                         d_to_split->end1 == end_chain) {
                s2_en = d_to_split->end1;
            } else if (s2_en >
                      (d_to_split->end1 - params.MIN_SEGMENT_SIZE_MID) &&
                       d_to_split->end1 != end_chain) {
                s2_en = d_to_split->end1;
            } /*if*/
        } /*while*/
        s1_s = s1_sn;
        if (ss_only) {
            i = s1_sn - 1;
            i++;
            while (ss[i] != 'H' && ss[i] != 'E' &&
                   i < d_to_split->end1) {
                i++;
            } /*while*/
            if (i == s1_sn) {
                s1_sn = s1_sn + increment;
            } else {
                s1_sn = i;
            } /*if*/
        } else {
            s1_sn = s1_sn + increment;
        } /*if*/
        if (d_to_split->start1 == start_chain &&
            s1_sn < (start_chain + params.MIN_SEGMENT_SIZE_END)) {
            s1_sn = start_chain + params.MIN_SEGMENT_SIZE_END;
        } else if (d_to_split->start1 != start_chain &&
                   s1_sn < (d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID)) {
            s1_sn = d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID;
        } /*if*/
        s1_en = s1_sn + params.MIN_SEGMENT_SIZE_MID - 1;
        s2_sn = s1_en + params.MIN_SEGMENT_SIZE_MID + 1;
        s2_en = s2_sn + params.MIN_SEGMENT_SIZE_MID - 1;
        if (s2_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_END) &&
                     d_to_split->end1 == end_chain) {
            s2_en = d_to_split->end1;
        } else if (s2_en >
                  (d_to_split->end1 - params.MIN_SEGMENT_SIZE_MID) &&
                   d_to_split->end1 != end_chain) {
            s2_en = d_to_split->end1;
        } /*if*/
    } /*while*/
/* now do single splits */
    d_list1 = asd_make_single_split(d_by_choice, num_d_by_choice, d_to_split,
                              num_atoms, contact_info, contact_rev_info, bn,
                              start_chain, end_chain, ss, FALSE);
    max_domain_s = d_list1->domains;
/* now examaine results */

    if ((!ss_only && max_domain_s->value < params.MIN_PEAK_C) ||
         (ss_only && max_domain_s->value < params.MIN_PEAK_SS_ONLY_C)) {
        s_ok = 0;
    } else {
        s_ok = 1;
    } /*if*/
    if ((!ss_only && max_domain_m->value < params.MIN_PEAK_MC) ||
         (ss_only && max_domain_m->value < params.MIN_PEAK_SS_ONLY_MC)) {
        m_ok = 0;
    } else {
        m_ok = 1;
    } /*if*/
    /* remove inaccuarcies due to increments */
    if (max_domain_s->value > 0.0) {
        if (s_ok && (!m_ok ||
                     (m_ok && max_domain_s->value >= max_domain_m->value))) {
            max_domain_s->ok = TRUE;
        } else {
            max_domain_s->ok = FALSE;
        } /*if*/
        asd_focus_range(max_domain_s, d_to_split, contact_info,
                        contact_rev_info, bn, start_chain, end_chain, ss);
        max_domain_s->ok = TRUE;
    } else {
        max_domain_s->ok = FALSE;
    } /*if*/
    if (max_domain_m->value > 0.0) {
        if (m_ok && (!s_ok ||
                     (s_ok && max_domain_m->value >= max_domain_s->value))) {
            max_domain_m->ok = TRUE;
        } else {
            max_domain_m->ok = FALSE;
        } /*if*/
        asd_focus_range(max_domain_m, d_to_split, contact_info,
                        contact_rev_info, bn, start_chain, end_chain, ss);
        max_domain_m->ok = TRUE;
    } else {
        max_domain_m->ok = FALSE;
    } /*if*/
/* if peak was not big enough */
    if (s_ok == 0 && m_ok == 0) {
        d_list1 = (Asd_Domain_List *) asm_malloc(sizeof(Asd_Domain_List));
        d_list1->n_d_list = 1;
        d_list1->n_b_left_over = 0;
        d_list1->domains = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
        d_list1->domains->type = 1;
        d_list1->domains->start1 = d_to_split->start1;
        d_list1->domains->end1 = d_to_split->end1;
        d_list1->domains->glob = LARGE_NEG_GLOB;
        d_list1->d_split_up = d_to_split;
        d_list1->domains->id = asstr_save(d_to_split->id);
        if (max_domain_m->ok == FALSE && max_domain_s->ok == FALSE) {
            d_list1->domains->ok = FALSE;
        } else {
            d_list1->domains->ok = TRUE;
        } /*if*/
        asd_d_comments(3, d_list1->domains, bn);
        d_list1->domains->ok = TRUE;
        max_domain_s->id = asstr_cat_safe(asstr_save(d_to_split->id), ".1");
        max_domain_m->id = asstr_cat_safe(asstr_save(d_to_split->id), ".1");
        asd_d_comments(4, max_domain_s, bn);
        asd_d_comments(4, max_domain_m, bn);
        asd_free_c_list(d_to_split, c_list);
        return(d_list1);
    } /*if*/

    if (s_ok == 1 && m_ok == 1) {
        if (max_domain_s->value > max_domain_m->value) {
            m_ok = 0;
        } else {
            s_ok = 0;
        } /*if*/
    } /*if*/

    if (s_ok) {
        max_domain_s->id = asstr_cat_safe(asstr_save(d_to_split->id), ".1");
        asd_d_comments(5, max_domain_s, bn);
/* if 2 parts */
        if (max_domain_s->start1 != d_to_split->start1 &&
            max_domain_s->end1 != d_to_split->end1) {
            d_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain) * 2);
            d_left_over[0].start1 = d_to_split->start1;
            d_left_over[0].end1 = max_domain_s->start1 - 1;
            d_left_over[1].start1 =  max_domain_s->end1 + 1;
            d_left_over[1].end1 = d_to_split->end1;
/* if pair correlated analyse together else analyse seperately */
            if (asd_check_correlation(contact_info, contact_rev_info, bn,
                                      d_left_over, ss, 2, c_list)) {
                asd_free_c_list(d_to_split, c_list);
                d_list1 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  max_domain_s, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_list1->d_split_up = max_domain_s;
                d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                                 ".2");
                d_list2 = asd_multi_split_sep(contact_info, contact_rev_info,
                                bn, d_left_over, d_by_choice, num_d_by_choice,
                                start_chain, end_chain, ss);
                d_list2->d_split_up = d_left_over;
            } else {
                asd_free_c_list(d_to_split, c_list);
                d_list1 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  max_domain_s, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_list1->d_split_up = max_domain_s;
                d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                                 ".2");
                d_list2 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  d_left_over, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_list2->d_split_up = d_left_over;
                d_left_over[1].id = asstr_cat_safe(asstr_save(d_to_split->id),
                                                 ".3");
                d_list3 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  &(d_left_over[1]), num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_list3->d_split_up = &(d_left_over[1]);
            } /*if*/
        } else {
/* analyse both parts seperately */
            d_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
            if (max_domain_s->start1 == d_to_split->start1) {
                d_left_over->start1 = max_domain_s->end1 + 1;
                d_left_over->end1 = d_to_split->end1;
            } else if (max_domain_s->end1 == d_to_split->end1) {
                d_left_over->start1 = d_to_split->start1;
                d_left_over->end1 = max_domain_s->start1 - 1;
            } /*if*/
            asd_free_c_list(d_to_split, c_list);
            d_list1 = asd_make_multiple_domains(d_by_choice, num_d_by_choice,
                                  max_domain_s, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
            d_list1->d_split_up = max_domain_s;
            d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                             ".2");
            d_list2 = asd_make_multiple_domains(d_by_choice, num_d_by_choice,
                                  d_left_over, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
            d_list2->d_split_up = d_left_over;
        } /*if*/
            
    } else {
/* now analyse the multi domain that was picked out */
        max_domain_m->id = asstr_cat_safe(asstr_save(d_to_split->id), ".1");
        asd_d_comments(5, max_domain_m, bn);
/* if one part left over */
        if (max_domain_m->start1 == d_to_split->start1 && max_domain_m->end2 ==
            d_to_split->end1) {
            d_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
            d_left_over->start1 = max_domain_m->end1 + 1;
            d_left_over->end1 = max_domain_m->start2 - 1;
            asd_free_c_list(d_to_split, c_list);
            d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                              ".2");
            d_list1 = asd_make_multiple_domains(d_by_choice, num_d_by_choice,
                                  d_left_over, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
            d_list1->d_split_up = d_left_over;
/* if two parts left over */
        } else if (max_domain_m->start1 == d_to_split->start1) {
            d_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain) * 2);
            d_left_over->start1 = max_domain_m->end1 + 1;
            d_left_over->end1 = max_domain_m->start2 - 1;
            d_left_over[1].start1 = max_domain_m->end2 + 1;
            d_left_over[1].end1 = d_to_split->end1;
            if (asd_check_correlation(contact_info, contact_rev_info, bn,
                                      d_left_over, ss, 2, c_list)) {
                asd_free_c_list(d_to_split, c_list);
                d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                              ".2");
                d_list1 = asd_multi_split_sep(contact_info, contact_rev_info,
                                    bn,
                                    d_left_over, d_by_choice, num_d_by_choice,
                                    start_chain, end_chain, ss);
                d_list1->d_split_up = d_left_over;
            } else {
                asd_free_c_list(d_to_split, c_list);
                d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                              ".2");
                d_list1 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  d_left_over, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_list1->d_split_up = d_left_over;
                d_left_over[1].id = asstr_cat_safe(asstr_save(d_to_split->id),
                                              ".3");
                d_list2 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  &(d_left_over[1]), num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_list2->d_split_up = &(d_left_over[1]);
            } /*if*/
        } else if (max_domain_m->end2 == d_to_split->end1) {
            d_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain) * 2);
            d_left_over->start1 = d_to_split->start1;
            d_left_over->end1 = max_domain_m->start1 - 1;
            d_left_over[1].start1 = max_domain_m->end1 + 1;
            d_left_over[1].end1 = max_domain_m->start2 - 1;
            if (asd_check_correlation(contact_info, contact_rev_info, bn,
                                      d_left_over, ss, 2, c_list)) {
                asd_free_c_list(d_to_split, c_list);
                d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                              ".2");
                d_list1 = asd_multi_split_sep(contact_info, contact_rev_info,
                                    bn,
                                    d_left_over, d_by_choice, num_d_by_choice,
                                    start_chain, end_chain, ss);
                d_list1->d_split_up = d_left_over;
            } else {
                asd_free_c_list(d_to_split, c_list);
                d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                              ".2");
                d_list1 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  d_left_over, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_list1->d_split_up = d_left_over;
                d_left_over[1].id = asstr_cat_safe(asstr_save(d_to_split->id),
                                              ".3");
                d_list2 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  &(d_left_over[1]), num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_list2->d_split_up = &(d_left_over[1]);
            } /*if*/

/* else if three parts left over */
        } else {
            d_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain) * 3);
            d_left_over->start1 = d_to_split->start1;
            d_left_over->end1 = max_domain_m->start1 - 1;
            d_left_over[1].start1 = max_domain_m->end1 + 1;
            d_left_over[1].end1 = max_domain_m->start2 - 1;
            d_left_over[2].start1 = max_domain_m->end2 + 1;
            d_left_over[2].end1 = d_to_split->end1;
            if (asd_check_correlation(contact_info, contact_rev_info, bn,
                                      d_left_over, ss, 3, c_list)) {
                asd_free_c_list(d_to_split, c_list);
                d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                              ".2");
                d_list1 = asd_multi_split_sep(contact_info, contact_rev_info,
                                    bn,
                                    d_left_over, d_by_choice, num_d_by_choice,
                                    start_chain, end_chain, ss);
                d_list1->d_split_up = d_left_over;
                d_left_over[1].id = asstr_cat_safe(asstr_save(d_to_split->id),
                                              ".3");
                d_list2 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  &(d_left_over[1]), num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_list2->d_split_up = &(d_left_over[1]);
            } else {
                asd_free_c_list(d_to_split, c_list);
                d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                              ".2");
                d_list1 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  d_left_over, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_list1->d_split_up = d_left_over;
                d_left_over[1].id = asstr_cat_safe(asstr_save(d_to_split->id),
                                              ".3");
                d_list2 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  &(d_left_over[1]), num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_list2->d_split_up = &(d_left_over[1]);
                d_left_over[2].id = asstr_cat_safe(asstr_save(d_to_split->id),
                                              ".4");
                d_list3 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  &(d_left_over[2]), num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_list3->d_split_up = &(d_left_over[2]);
            } /*if*/
        } /*if*/
        d_list4 = asd_multi_split_sep(contact_info, contact_rev_info, bn,
                            max_domain_m, d_by_choice, num_d_by_choice,
                            start_chain, end_chain, ss);
        d_list4->d_split_up = max_domain_m;
    } /*if*/

    return(asd_check_globularity(d_list1, d_list2, d_list3, d_list4, bn,
                                 d_to_split));
} /*asd_make_multiple_domains*/


Asd_Domain_List *
asd_work_out_domains(int num_atoms, Asd_Contact_Info **contact_info,
                     Asd_Contact_Info **contact_rev_info, struct brookn *bn,
                     char cid_g, char *ss)
{
    int num_chains;             /* number of chains in molecule */
    int num_d_by_choice;        /* number of chosen domains     */
    int i;                      /* loop counter                 */
    char cid;                   /* chain indentifier            */
    Asd_Chain_Info *chain_list; /* information on chains        */
    Asd_Domain *d_by_choice;    /* list of chosen domains       */
    Asd_Domain *d_to_split;     /* list of domains by default   */
    Asd_Domain_List *d_list;    /* final list of domains        */

/*
 * first count the number of chains in the molecule, assumes that the chains
 * are contiguous and there are no gaps in the numbering
 */
    num_chains = 0;
    chain_list = NULL;
    num_d_by_choice = 0;
    d_by_choice = 0;

    i = 0;
    while (i < num_atoms) {
        if (i != 0) {
            if (bn[i].cid != cid) {
                chain_list[num_chains - 1].end = i;
                cid = bn[i].cid;
                num_chains++;
                chain_list = (Asd_Chain_Info *) asm_realloc(chain_list, 
                             sizeof(Asd_Chain_Info) * num_chains);
                chain_list[num_chains - 1].cid = cid;
                chain_list[num_chains - 1].start = i + 1;
            } /*if*/
        } else {
            cid = bn[i].cid;
            chain_list = (Asd_Chain_Info *) asm_malloc(sizeof(Asd_Chain_Info));
            chain_list[0].cid = cid;
            chain_list[0].start = i + 1;
            num_chains++;
        } /*if*/
        i++;
    } /*while*/
    chain_list[num_chains - 1].end = i;
/*
 * now figure out if chains are distinct - duplicated ???
 * align sequences and check to see if chains are duplicated
 */

/*
 * now examine each chain or combinations of chains
 */
    d_to_split = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
    if (cid_g == '*') {
        d_to_split->start1 = 1;
        d_to_split->end1 = num_atoms;
    } else {
        i = 0;
        while (i < num_chains && chain_list[i].cid != cid_g) {
            i++;
        } /*while*/
        if (i == num_chains) {
            ase_error_fatal("asd_work_out_domains",
            "could not find chain");
        } /*if*/
        d_to_split->start1 = chain_list[i].start;
        d_to_split->end1 = chain_list[i].end;
    } /*if*/

    asd_sort_helices(d_to_split->start1, d_to_split->end1, contact_info, ss);
    d_to_split->type = 1;
    d_to_split->start2 = 0;
    d_to_split->end2 = 0;
    d_to_split->id = asstr_save("1");
    d_list = asd_make_multiple_domains(d_by_choice, &num_d_by_choice, 
                                      d_to_split, num_atoms, 
                                      contact_info, contact_rev_info, bn,
                                      d_to_split->start1, d_to_split->end1, ss);
/* analyse bits left over to see if they belong anywhere */
    i = 0;
    while (i < d_list->n_d_list) {
        d_list->domains[i].d_assigned = -2;
        i++;
    } /*while*/
    while (i < d_list->n_b_left_over) {
        d_list->bits_left_over[i].d_assigned = -1;
        i++;
    } /*while*/

    asd_analyse_bits_left_over(d_to_split, d_list, contact_info,
                               contact_rev_info, bn, ss);
    asd_check_glob_end(d_to_split, d_list, contact_info,
                               contact_rev_info, bn, ss);

    return(d_list);
} /*asd_work_out_domains*/
