/******************************************************************************
 The computer software and associated documentation called DOMAK hereinafter
 referred to as the WORK which is more particularly identified and described in
 Appendix A of the file LICENSE.  Conditions and restrictions for use of
 this package are also in this file.

 This routine was written by Geoffrey J. Barton

 The WORK was developed by:
        Asim S. Siddiqui and Geoffrey J. Barton
        Laboratory of Molecular Biophysics
        University of Oxford
        Rex Richards Building
        South Parks Road
        Oxford OX1 3QU U.K.
        Tel:  (+44) 865-275379
        FAX:  (+44) 865-510454
        INTERNET: as@bioch.ox.ac.uk
        JANET:    as@uk.ac.ox.bioch

 The WORK is Copyright (1995) University of Oxford
        Administrative Offices
        Wellington Square
        Oxford OX1 2JD U.K.

 All use of the WORK must cite:
 Siddiqui, A. S. and Barton, G. J., "Continuous and Discontinuous Domains: An
 Algorithm for the Automatic Generation of Reliable Protein Domain Definitions" 
 PROTEIN SCIENCE, 4:872-884 (1995).
*****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <rdssp.h>

#define MAXL 200    /* max length for input line */
#define MAXR 16	    /* max number of ranges */
#define MAXFLEN 50   /* max length for a filename */
/*
#define FILEHEADER  "/data/dssp/"
#define FILETAIL    ".ALL"
*/

/* rdssp:  Routines to access DSSP .ALL file format.  The routines cope with
   the vagaries of the format, including the problem of numbers < -999 shifting
   all following columns...  All routines return a pointer to an array of the
   appropriate type for the data being requested.  The space for the array is
   allocated in the routine (using malloc()).  The first argument of all
   routines is the name of the DSSP .ALL file from which the data is to be
   extracted.  The second is the address of an integer to store the number of
   elements in the returned array.  All arrays are stored from the 0th element,
   thus the first value is array[0], not array[1].  Note that the value of
   count for character arrays does not include the terminating '\0'.

   The main programme commented out below, illustrates the useage of the
   functions

   Routines starting with 'n' return ranges of information.  e.g.  nget_ss()
   will return a range of secondary structure summaries.  See examples in the
   following main programme.  Note that this is not done efficiently - the
   complete dataset is first selected from the file, then the range selected
   from the dataset (!).  This should be fast enough for most purposes....

   See "rdssp.h" for definitions of structures returned by these functions.

   If you find bugs in this code, or feel that there are functions that could
   be usefully added - then please let me know.

   Author:  Geoff Barton April 1991.

   Version 1.0:  April 29th 1991.
           1.1:  April 30th 1991 - add disul_s();
	   2.0   August 18th 1993 - ANSI-fy the code


main(int argc, char *argv[])

{
    char *aa,*ss,*ao;
    int *n,*acc;
    struct brookn *bn,*bo;
    int i;
    int count;
    struct hbond *nho1,*onh1,*nho2,*onh2;
    float *tco,*kappa,*alpha;
    struct ps *phipsi;
    struct threed *xyz;
    struct br *bridge;
    char *sheet;
    struct range *r;
    char *clist;
    int s,e;
    char *fname;

    fname = (char *) malloc(sizeof(char)*MAXFLEN);
    s=2;
    e=3;

    fname = strcpy(fname,FILEHEADER);
    fname = strcat(fname,argv[1]);
    fname = strcat(fname,FILETAIL);

    aa = get_aa(fname,&count);
    aa = disul_s(aa,&count);
    ao = nget_aa(fname,&count,s,e);
    bn = get_bn(fname,&count);
    bo = nget_bn(fname,&count,s,e);

    ss = get_ss(fname,&count);
    n = get_n(fname,&count);

    acc = nget_acc(fname,&count,s,e);
    nho1 = nget_hb(fname,&count,"NHO1",s,e);
    onh1 = nget_hb(fname,&count,"ONH1",s,e);
    nho2 = nget_hb(fname,&count,"NHO2",s,e);
    onh2 = nget_hb(fname,&count,"ONH2",s,e);
    tco = nget_tco(fname,&count,s,e);
    kappa = nget_kappa(fname,&count,s,e);
    alpha = nget_alpha(fname,&count,s,e);
    phipsi = nget_phipsi(fname,&count,s,e);
    bridge = nget_bridge(fname,&count,s,e);
    sheet = nget_sheet(fname,&count,s,e);
    xyz = nget_ca(fname,&count,s,e);
    r = chain_range(fname,&count,'H');
    clist = chain_list(fname,&count);



    for(i=0;i<count;++i){
	printf("%d %c %c %c%d%c %d",n[i],aa[i],ss[i],bn[i].cid,bn[i].n,bn[i].in,acc[i]);
	printf(" %d %f %d %f %d %f %d %f ",nho1[i].pos,nho1[i].energy,
					   onh1[i].pos,onh1[i].energy,
					   nho2[i].pos,nho2[i].energy,
					   onh2[i].pos,onh2[i].energy);
	printf("%f ",tco[i]);
	printf("%f ",kappa[i]);
	printf("%f ",alpha[i]);
	printf("%f %f ",phipsi[i].phi,phipsi[i].psi);
	printf("%d %d ",bridge[i].one,bridge[i].two);
	printf("%c ",sheet[i]);
	printf("%f %f %f ",xyz[i].x,xyz[i].y,xyz[i].z);
	printf("\n");
    }

}
*/
/*************************************************************
check_dssp:  check if dssp has been run to completion on this
file.  ie. just look for the # symbol

G. J. Barton 1993
--------------------------------------------------------------*/
int check_dssp(char *fname)


{
    FILE *fp;
    char *line;

    line = (char *) malloc(sizeof(char)*(MAXL+1));

    fp = fopen(fname,"r");
    if(fp == NULL) return 0;

    while(fgets(line,MAXL,fp) != NULL){
	if(line[2] == '#'){
	  free((char *) line);
	  fclose(fp);
	  return 1;
	}
    }
    free((char *) line);
    fclose(fp);
    return 0;
}

/*************************************************************
get_aa:  return the amino acid sequence column from a dssp file
G. J. Barton 1991
--------------------------------------------------------------*/
char *get_aa(char *fname,int *count)


{
    FILE *fp;
    char *line;
    char start;

    char *aa;

    line = (char *) malloc(sizeof(char)*(MAXL+1));
    start=0;
    *count=0;

    fp = fopen(fname,"r");
    aa = (char *) malloc(sizeof(char));

    while(fgets(line,MAXL,fp) != NULL){
	if(start){
	    aa = (char *) realloc(aa,sizeof(char) * (*count+1));
	    aa[(*count)++] = line[13];
	}
	if(line[2] == '#')start=1;
    }
    aa = (char *) realloc(aa,sizeof(char) * (*count+1));    
    aa[*count]='\0';
    free((char *) line);
    fclose(fp);
    return aa;
}
/*************************************************************
get_ss:  return the secondary structure summary
	 from a dssp file
--------------------------------------------------------------*/
char *get_ss(char *fname,int *count)
{
    FILE *fp;
    char *line;
    char start;

    char *aa;
    
    start=0;
    *count=0;

    line = (char *) malloc(sizeof(char)*(MAXL + 1));

    fp = fopen(fname,"r");
    aa = (char *) malloc(sizeof(char));

    while(fgets(line,MAXL,fp) != NULL){
	if(start){
	    aa = (char *) realloc(aa,sizeof(char) * (*count+1));
	    aa[(*count)++] = line[16];
	}
	if(line[2] == '#')start=1;
    }
    free((char *) line);
    aa = (char *) realloc(aa,sizeof(char) * (*count+1));    
    aa[*count]='\0';
    fclose(fp);
    return aa;
}
/*************************************************************
get_n:  return the numbering scheme
	 from a dssp file
--------------------------------------------------------------*/
int *get_n(char *fname,int *count)

{
    FILE *fp;
    char *line;
    char start;

    int *aa;
    char *bit;  /* bit to read from */
    
    start=0;
    *count=0;

    line = (char *) malloc(sizeof(char)*(MAXL + 1));

    fp = fopen(fname,"r");
    aa = (int *) malloc(sizeof(int));

    while(fgets(line,MAXL,fp) != NULL){
	if(start){
	    aa = (int *) realloc(aa,sizeof(int) * (*count+1));
	    bit = strbit(0,4,line);
	    sscanf(bit,"%d",&aa[(*count)++]);
	}
	if(line[2] == '#')start=1;
    }
    free((char *) line);
    free((char *) bit);
    fclose(fp);
    return aa;
}
/*************************************************************
get_bn:  return the brookhaven numbering scheme
	 from a dssp file
NOTE:  If an error occurs when translating the numerical part of 
the residue number (e.g. when it is absent), the bn.n value is
set to 0 and an '!' character stored as the bn.cid character.
--------------------------------------------------------------*/
struct brookn *get_bn(char *fname,int *count)

{
    FILE *fp;
    char *line;
    char start;

    struct brookn *aa;
    char *bit;  /* bit to read from */
    
    start=0;
    *count=0;

    line = (char *) malloc(sizeof(char)*(MAXL + 1));

    fp = fopen(fname,"r");
    aa = (struct brookn *) malloc(sizeof(struct brookn));

    while(fgets(line,MAXL,fp) != NULL){
	if(start){
	    aa = (struct brookn *) realloc(aa,sizeof(struct brookn) * (*count+1));
	    bit = strbit(5,9,line);
	    if(sscanf(bit,"%d",&aa[*count].n)==EOF){
		aa[*count].n=0;		/* this is a position that is not numbered */
		aa[*count].cid='!';	/* e.g. a chain break */
		aa[*count].in=' ';
	    }else{
		aa[*count].cid = line[11];
		aa[*count].in = line[10];
	    }
	    ++(*count);
	}
	if(line[2] == '#')start=1;
    }
    free((char *) line);
    free((char *) bit);
    fclose(fp);
    return aa;
}
/*************************************************************
get_acc:  return the accessibility from a dssp file

--------------------------------------------------------------*/
int *get_acc(char *fname,int *count)

{
    FILE *fp;
    char *line;
    char start;

    int *aa;
    char *bit;  /* bit to read from */
    
    start=0;
    *count=0;

    line = (char *) malloc(sizeof(char)*(MAXL + 1));

    fp = fopen(fname,"r");
    aa = (int *) malloc(sizeof(int));

    while(fgets(line,MAXL,fp) != NULL){
	if(start){
	    aa = (int *) realloc(aa,sizeof(int) * (*count+1));
	    bit = strbit(35,37,line);
	    sscanf(bit,"%d",&aa[*count]);
	    ++(*count);
	}
	if(line[2] == '#')start=1;
    }
    free((char *) line);
    free((char *) bit);
    fclose(fp);
    return aa;
}
/*************************************************************
get_hb:  return pointer to  hbond structure from a dssp file
G. J. Barton 1991
--------------------------------------------------------------*/
struct hbond *get_hb(char *fname,int *count,char *type)

{
    FILE *fp;
    char *line;
    char start;

    struct hbond *aa;
    struct range *tr;
    char *bit;  /* bit to read from */
    char *bit1;
    
    start=0;
    *count=0;

    line = (char *) malloc(sizeof(char)*(MAXL + 1));

    fp = fopen(fname,"r");
    aa = (struct hbond *) malloc(sizeof(struct hbond));

    while(fgets(line,MAXL,fp) != NULL){
	if(start){
	    tr = i_fields(line);
	    aa = (struct hbond *) realloc(aa,sizeof(struct hbond) * (*count+1));
	    if(strcmp(type,"NHO1")==0){
		bit = strbit(tr[0].s,tr[0].e,line);
		bit1= strbit(tr[1].s,tr[1].e,line);
	    }else if(strcmp(type,"ONH1")==0){
		bit = strbit(tr[2].s,tr[2].e,line);
		bit1 = strbit(tr[3].s,tr[3].e,line);
	    }else if(strcmp(type,"NHO2")==0){
		bit = strbit(tr[4].s,tr[4].e,line);
		bit1 = strbit(tr[5].s,tr[5].e,line);
	    }else if(strcmp(type,"ONH2")==0){
		bit = strbit(tr[6].s,tr[6].e,line);
		bit1 = strbit(tr[7].s,tr[7].e,line);
	    }else{
		fprintf(stderr,"ERROR: Unrecognised hbond type: %s\n",type);
		exit(1);
	    }
	    sscanf(bit,"%d",&aa[*count].pos);
	    sscanf(bit1,"%f",&aa[*count].energy);
	    ++(*count);
	}
	if(line[2] == '#')start=1;
    }
    free((char *) line);
    free((char *) bit);
    free((char *) bit1);
    fclose(fp);
    return aa;
}
/*************************************************************
get_tco:  return pointer to  array of tco values from dssp output
G. J. Barton 1991
--------------------------------------------------------------*/
float *get_tco(char *fname,int *count)

{
    FILE *fp;
    char *line;
    char start;

    float *aa;
    struct range *tr;
    char *bit;  /* bit to read from */
    
    start=0;
    *count=0;

    line = (char *) malloc(sizeof(char)*(MAXL + 1));

    fp = fopen(fname,"r");
    aa = (float *) malloc(sizeof(float));

    while(fgets(line,MAXL,fp) != NULL){
	if(start){
	    tr = i_fields(line);
	    aa = (float *) realloc(aa,sizeof(float) * (*count+1));
	    bit = strbit(tr[8].s,tr[8].e,line);
	    sscanf(bit,"%f",&aa[*count]);
	    ++(*count);
	}
	if(line[2] == '#')start=1;
    }
    free((char *) line);
    free((char *) bit);
    fclose(fp);
    return aa;
}
/*************************************************************
get_kappa:  return pointer to  array of kappa values from dssp output
G. J. Barton 1991
--------------------------------------------------------------*/
float *get_kappa(char *fname,int *count)

{
    FILE *fp;
    char *line;
    char start;

    float *aa;
    struct range *tr;
    char *bit;  /* bit to read from */
    
    start=0;
    *count=0;

    line = (char *) malloc(sizeof(char)*(MAXL + 1));

    fp = fopen(fname,"r");
    aa = (float *) malloc(sizeof(float));

    while(fgets(line,MAXL,fp) != NULL){
	if(start){
	    tr = i_fields(line);
	    aa = (float *) realloc(aa,sizeof(float) * (*count+1));
	    bit = strbit(tr[9].s,tr[9].e,line);
	    sscanf(bit,"%f",&aa[*count]);
	    ++(*count);
	}
	if(line[2] == '#')start=1;
    }
    free((char *) line);
    free((char *) bit);
    fclose(fp);
    return aa;
}
/*************************************************************
get_alpha:  return pointer to  array of alpha values from dssp output
G. J. Barton 1991
--------------------------------------------------------------*/
float *get_alpha(char *fname,int *count)

{
    FILE *fp;
    char *line;
    char start;

    float *aa;
    struct range *tr;
    char *bit;  /* bit to read from */
    
    start=0;
    *count=0;

    line = (char *) malloc(sizeof(char)*(MAXL + 1));

    fp = fopen(fname,"r");
    aa = (float *) malloc(sizeof(float));

    while(fgets(line,MAXL,fp) != NULL){
	if(start){
	    tr = i_fields(line);
	    aa = (float *) realloc(aa,sizeof(float) * (*count+1));
	    bit = strbit(tr[10].s,tr[10].e,line);
	    sscanf(bit,"%f",&aa[*count]);
	    ++(*count);
	}
	if(line[2] == '#')start=1;
    }
    free((char *) line);
    free((char *) bit);
    fclose(fp);
    return aa;
}
/*************************************************************
get_phipsi:  return pointer to phipsi structure
G. J. Barton 1991
--------------------------------------------------------------*/
struct ps *get_phipsi(char *fname,int *count)

{
    FILE *fp;
    char *line;
    char start;

    struct ps *aa;
    struct range *tr;
    char *bit;  /* bit to read from */
    char *bit1;  /* bit to read from */
    
    start=0;
    *count=0;

    line = (char *) malloc(sizeof(char)*(MAXL + 1));

    fp = fopen(fname,"r");
    aa = (struct ps *) malloc(sizeof(struct ps));

    while(fgets(line,MAXL,fp) != NULL){
	if(start){
	    tr = i_fields(line);
	    aa = (struct ps *) realloc(aa,sizeof(struct ps) * (*count+1));
	    bit = strbit(tr[11].s,tr[11].e,line);
	    bit1 = strbit(tr[12].s,tr[12].e,line);
	    sscanf(bit,"%f",&aa[*count].phi);
	    sscanf(bit1,"%f",&aa[*count].psi);
	    ++(*count);
	}
	if(line[2] == '#')start=1;
    }
    free((char *) line);
    free((char *) bit);
    free((char *) bit1);
    fclose(fp);
    return aa;
}
/*************************************************************
get_ca:  return pointer to coords of ca atoms using threed
structure 
G. J. Barton 1991
--------------------------------------------------------------*/
struct threed *get_ca(char *fname,int *count)

{
    FILE *fp;
    char *line;
    char start;

    struct threed *aa;
    struct range *tr;
    char *bit;  /* bit to read from */
    char *bit1;  /* bit to read from */
    char *bit2;  /* bit to read from */
    
    start=0;
    *count=0;

    line = (char *) malloc(sizeof(char)*(MAXL + 1));

    fp = fopen(fname,"r");
    aa = (struct threed *) malloc(sizeof(struct threed));

    while(fgets(line,MAXL,fp) != NULL){
	if(start){
	    tr = i_fields(line);
	    aa = (struct threed *) realloc(aa,sizeof(struct threed) * (*count+1));
	    bit = strbit(tr[13].s,tr[13].e,line);
	    bit1 = strbit(tr[14].s,tr[14].e,line);
	    bit2 = strbit(tr[15].s,tr[15].e,line);
	    sscanf(bit,"%f",&aa[*count].x);
	    sscanf(bit1,"%f",&aa[*count].y);
	    sscanf(bit2,"%f",&aa[*count].z);
	    ++(*count);
	}
	if(line[2] == '#')start=1;
    }
    free((char *) line);
    free((char *) bit);
    free((char *) bit1);
    free((char *) bit2);
    fclose(fp);
    return aa;
}

/*************************************************************
get_bridge:  return pointer to structure showing beta bridge
partners of each residue in a dssp file
G. J. Barton 1991
--------------------------------------------------------------*/
struct br *get_bridge(char *fname,int *count)
{
    FILE *fp;
    char *line;
    char start;

    struct br *aa;
    char *bit;  /* bit to read from */
    char *bit1;  /* bit to read from */
    
    start=0;
    *count=0;

    line = (char *) malloc(sizeof(char)*(MAXL + 1));

    fp = fopen(fname,"r");
    aa = (struct br *) malloc(sizeof(struct br));

    while(fgets(line,MAXL,fp) != NULL){
	if(start){
	    aa = (struct br *) realloc(aa,sizeof(struct br) * (*count+1));
	    bit = strbit(25,28,line);
	    bit1 = strbit(29,32,line);
	    sscanf(bit,"%d",&aa[*count].one);
	    sscanf(bit1,"%d",&aa[*count].two);
	    ++(*count);
	}
	if(line[2] == '#')start=1;
    }
    free((char *) line);
    free((char *) bit);
    free((char *) bit1);
    fclose(fp);
    return aa;
}
/*************************************************************
get_sheet:  return the sheet id codes for each residue
G. J. Barton 1991
--------------------------------------------------------------*/
char *get_sheet(char *fname,int *count)

{
    FILE *fp;
    char *line;
    char start;

    char *aa;

    line = (char *) malloc(sizeof(char)*(MAXL + 1));
    start=0;
    *count=0;

    fp = fopen(fname,"r");
    aa = (char *) malloc(sizeof(char));

    while(fgets(line,MAXL,fp) != NULL){
	if(start){
	    aa = (char *) realloc(aa,sizeof(char) * (*count+1));
	    aa[(*count)++] = line[33];
	}
	if(line[2] == '#')start=1;
    }
    aa = (char *) realloc(aa,sizeof(char) * (*count+1));
    aa[*count]='\0';
    free((char *) line);
    fclose(fp);
    return aa;
}

/*******************************************************************
 given string s, start and end character numbers, return a string
 containing the characters between start and end inclusive
*/
char *strbit(int start,int end,char *s)

{
    int len;
    char *bit;
    int i,j;
    bit = (char *) malloc(sizeof(char)*(end-start+2));
    for(i=start,j=0;i<(end+1);++i,++j){
	bit[j]=s[i];
    }
    bit[j]='\0';
    return bit;
}
/********************************************************************************
i_fields:

Takes line and returns a pointer to an array of start,end positions 
in line that correspond to the numerical fields in a dssp file.
The fields start with the H-bonding information - character 38 of line there are
16 fields to the right of this character.
This routine is necessary because values below -999 are not printed in the same 
columns as all other values thus lengthening the lines.
*/
struct range rcol[]={
    38,42,
    44,47,
    48,51,
    53,56,
    57,60,
    62,65,
    66,69,
    71,74,
    75,82,
    83,88,
    89,94,
    95,100,
    101,106,
    107,113,
    114,120,
    121,127
};  /* the standard positions */
struct range cran[MAXR];    /* array to hold non-standard positions */

struct range *i_fields(char *line)

{
    int nc,nd,i;
    int *commas,*dots;

    if(strlen(line) == 129){	/* normal length line */
	return rcol;
    }
/* abnormal length line */
    commas = (int *) malloc(sizeof(int)*MAXR);
    dots = (int *) malloc(sizeof(int)*MAXR);
    nc =0;
    nd =0;
    for(i=38;i<strlen(line);++i){
	if(line[i] == ',')commas[nc++]=i;
	if(line[i] == '.')dots[nd++]=i;
    }
    if(nc != 4) {
	fprintf(stderr,"ERROR: incorrect number of commas in DSSP file\n");
	fprintf(stderr,"%s",line);
    }
    if(nd != 12) {
	fprintf(stderr,"ERROR: incorrect number of dots in DSSP file\n");
	fprintf(stderr,"%s",line);
    }
    cran[0].s = 38;
    cran[0].e = commas[0]-1;
    cran[1].s = commas[0]+1;
    cran[1].e = dots[0]+1;
    cran[2].s = dots[0]+2;
    cran[2].e = commas[1]-1;
    cran[3].s = commas[1]+1;
    cran[3].e = dots[1]+1;
    cran[4].s = dots[1]+2;
    cran[4].e = commas[2]-1;
    cran[5].s = commas[2]+1;
    cran[5].e = dots[2]+1;
    cran[6].s = dots[2]+2;
    cran[6].e = commas[3]-1;
    cran[7].s = commas[3]+1;
    cran[7].e = dots[3]+1;
    cran[8].s = dots[3]+2;
    cran[8].e = dots[4]+3;
    cran[9].s = dots[4]+4;
    cran[9].e = dots[5]+1;
    cran[10].s = dots[5]+2;
    cran[10].e = dots[6]+1;
    cran[11].s = dots[6]+2;
    cran[11].e = dots[7]+1;
    cran[12].s = dots[7]+2;
    cran[12].e = dots[8]+1;
    cran[13].s = dots[8]+2;
    cran[13].e = dots[9]+1;
    cran[14].s = dots[9]+2;
    cran[14].e = dots[10]+1;
    cran[15].s = dots[10]+2;
    cran[15].e = dots[11]+1;

    free((char *) commas);
    free((char *) dots);


    return cran;
}

/* return the range of the chain identified by cid */
/* 19/8/93 - add check for chain breaks */
struct range *chain_range(char *fname,int *count,char cid)

{
    int i;
    char fs,fe;
    struct brookn *b;
    struct range *r;
/*    extern FILE *std_err;

    fprintf(std_err,"Getting range of chain %c\n",cid); */
    r = (struct range *) malloc(sizeof(struct range));
    fs = 0;
    fe = 0;
    b = get_bn(fname,count);
    for(i=0;i<*count;++i){
	if(fe){
	    /* end of chain found so return range */
	    free((char *)b);
	    return r;
	}
	if(!fs && b[i].cid == cid){
	    /* start not found but chain id = query chain id so set start */
	    fs = 1;
	    r->s = i+1;
	}
	if(fs && (b[i].cid != cid) && (b[i].cid != '!')){
	    /* start found cid not the same as query cid and this is not a chain break */
	    /* so this is the end */
	    fe = 1;
	    r->e = i;
	}
	if(*count-1 == i){
	    /* found the end since this is the end of the array */
	    fe = 1;
	    r->e = i+1;
	}
    }
    free((char *)b);
    return r;
}
/* chain_list:  return a string containing the chain identifiers
that are present in the file */

char *chain_list(char *fname,int *count)

{
    int i,j;
    struct brookn *b;
    char *l,c;
    l = (char *) malloc(sizeof(char));
    b = get_bn(fname,count);
    c = '*';
    j = 0;
    for(i=0;i<*count;++i){
	if(c != b[i].cid && b[i].cid != '!'){
	    l = (char *) realloc(l,sizeof(char)*(j+1));
	    l[j]=b[i].cid;
	    c=l[j];
	    ++j;
	}
    }
    l = (char *) realloc(l,sizeof(char)*(j+1));
    l[j] = '\0';
    return l;
}
/*******************************************************************
nget_aa:  get a specific range using get_aa()
(Note:  The nget_() functions could be grouped into a single 
 generalised function by passing get_()  functions as arguments rather
 than having a separate nget function for each get_() function....

G. J. Barton 1991
-------------------------------------------------------------------*/
char *nget_aa(char *fname,int *count,int s,int e)

{
    int i,j;
    char *ao,*aa;
    aa = get_aa(fname,count);
    ao = (char *) malloc(sizeof(char)*(e-s+3));
    for(i=(s-1),j=0;i<e;++i,++j){
	ao[j]=aa[i];
    }
    ao[j]='\0';
    free((char *) aa);
    *count = j;
    return ao;
}
/*******************************************************************
nget_ss:  get a specific range using get_ss()
G. J. Barton 1991
-------------------------------------------------------------------*/
char *nget_ss(char *fname,int *count,int s,int e)

{
    int i,j;
    char *ao,*aa;
    aa = get_ss(fname,count);
    ao = (char *) malloc(sizeof(char)*(e-s+3));
    for(i=(s-1),j=0;i<e;++i,++j){
	ao[j]=aa[i];
    }
    ao[j]='\0';
    free((char *) aa);
    *count = j;
    return ao;
}
/*******************************************************************
nget_n:  get a specific range using get_n()
G. J. Barton 1991
-------------------------------------------------------------------*/
int *nget_n(char *fname,int *count,int s,int e)
{
    int i,j;
    int *ao,*aa;
    aa = get_n(fname,count);
    ao = (int *) malloc(sizeof(int)*(e-s+3));
    for(i=(s-1),j=0;i<e;++i,++j){
	ao[j]=aa[i];
    }
    free((char *) aa);
    *count = j;
    return ao;
}
/*******************************************************************
nget_bn:  get a specific range using get_bn()
G. J. Barton 1991
-------------------------------------------------------------------*/
struct brookn *nget_bn(char *fname,int *count,int s,int e)

{
    int i,j;
    struct brookn *ao,*aa;
    aa = get_bn(fname,count);
    ao = (struct brookn *) malloc(sizeof(struct brookn)*(e-s+3));
    for(i=(s-1),j=0;i<e;++i,++j){
	ao[j]=aa[i];
    }
    free((char *) aa);
    *count = j;
    return ao;
}
/*******************************************************************
nget_acc:  get a specific range using get_acc()
G. J. Barton 1991
-------------------------------------------------------------------*/
int *nget_acc(char *fname,int  *count,int s,int e)

{
    int i,j;
    int *ao,*aa;
    aa = get_acc(fname,count);
    ao = (int *) malloc(sizeof(int)*(e-s+3));
    for(i=(s-1),j=0;i<e;++i,++j){
	ao[j]=aa[i];
    }
    free((char *) aa);
    *count = j;
    return ao;
}
/*******************************************************************
nget_hb:  get a specific range using get_hbond()
G. J. Barton 1991
-------------------------------------------------------------------*/
struct hbond *nget_hb(char *fname,int *count,char *type,int s,int e)

{
    int i,j;
    struct hbond *ao,*aa;
    aa = get_hb(fname,count,type);
    ao = (struct hbond *) malloc(sizeof(struct hbond)*(e-s+3));
    for(i=(s-1),j=0;i<e;++i,++j){
	ao[j]=aa[i];
    }
    free((char *) aa);
    *count = j;
    return ao;
}
/*******************************************************************
nget_tco:  get a specific range using get_tco()
G. J. Barton 1991
-------------------------------------------------------------------*/
float *nget_tco(char *fname,int *count,int s,int e)

{
    int i,j;
    float *ao,*aa;
    aa = get_tco(fname,count);
    ao = (float *) malloc(sizeof(float)*(e-s+3));
    for(i=(s-1),j=0;i<e;++i,++j){
	ao[j]=aa[i];
    }
    free((char *) aa);
    *count = j;
    return ao;
}
/*******************************************************************
nget_kappa:  get a specific range using get_kappa()
G. J. Barton 1991
-------------------------------------------------------------------*/
float *nget_kappa(char *fname,int *count,int s,int e)

{
    int i,j;
    float *ao,*aa;
    aa = get_kappa(fname,count);
    ao = (float *) malloc(sizeof(float)*(e-s+3));
    for(i=(s-1),j=0;i<e;++i,++j){
	ao[j]=aa[i];
    }
    free((char *) aa);
    *count = j;
    return ao;
}
/*******************************************************************
nget_alpha:  get a specific range using get_alpha()
G. J. Barton 1991
-------------------------------------------------------------------*/
float *nget_alpha(char *fname,int *count,int s,int e)

{
    int i,j;
    float *ao,*aa;
    aa = get_alpha(fname,count);
    ao = (float *) malloc(sizeof(float)*(e-s+3));
    for(i=(s-1),j=0;i<e;++i,++j){
	ao[j]=aa[i];
    }
    free((char *) aa);
    *count = j;
    return ao;
}
/*******************************************************************
nget_phipsi:  get a specific range using get_phipsi()
G. J. Barton 1991
-------------------------------------------------------------------*/
struct ps *nget_phipsi(char *fname,int *count,int s,int e)

{
    int i,j;
    struct ps *ao,*aa;
    aa = get_phipsi(fname,count);
    ao = (struct ps *) malloc(sizeof(struct ps)*(e-s+3));
    for(i=(s-1),j=0;i<e;++i,++j){
	ao[j]=aa[i];
    }
    free((char *) aa);
    *count = j;
    return ao;
}
/*******************************************************************
nget_ca:  get a specific range using get_ca()
G. J. Barton 1991
-------------------------------------------------------------------*/
struct threed *nget_ca(char *fname,int *count,int s,int e)

{
    int i,j;
    struct threed *ao,*aa;
    aa = get_ca(fname,count);
    ao = (struct threed *) malloc(sizeof(struct threed)*(e-s+3));
    for(i=(s-1),j=0;i<e;++i,++j){
	ao[j]=aa[i];
    }
    free((char *) aa);
    *count = j;
    return ao;
}
/*******************************************************************
nget_bridge:  get a specific range using get_bridge()
G. J. Barton 1991
-------------------------------------------------------------------*/
struct br *nget_bridge(char *fname,int *count,int s,int e)

{
    int i,j;
    struct br *ao,*aa;
    aa = get_bridge(fname,count);
    ao = (struct br *) malloc(sizeof(struct br)*(e-s+3));
    for(i=(s-1),j=0;i<e;++i,++j){
	ao[j]=aa[i];
    }
    free((char *) aa);
    *count = j;
    return ao;
}

/*******************************************************************
nget_sheet:  get a specific range using get_sheet()
G. J. Barton 1991
-------------------------------------------------------------------*/
char *nget_sheet(char *fname,int *count,int s,int e)

{
    int i,j;
    char *ao,*aa;
    aa = get_sheet(fname,count);
    ao = (char *) malloc(sizeof(char)*(e-s+3));
    for(i=(s-1),j=0;i<e;++i,++j){
	ao[j]=aa[i];
    }
    ao[j]='\0';
    free((char *) aa);
    *count = j;
    return ao;
}
/********************************************************************
disul_s:  Takes the result of calling get_aa or nget_aa and converts
all lowercase letters to uppercase 'C'.  i.e.  removes the lowercase
s-s bridge information.
--------------------------------------------------------------------*/
char *disul_s(char *aa,int *count)

{
    int i;
    for(i=0;i<*count;++i){
	if(islower(aa[i]))aa[i]='C';
    }
    return aa;
}
/*******************************************************************
dblank_ss:  Takes a secondary structure string and substitutes all 
' ' for '-'
-------------------------------------------------------------------*/
char *dblank_ss(char *aa, int *count)
{
  int i;
  for(i=0;i<*count;++i){
    if(aa[i] == ' ')aa[i] = '-';
  }
  return aa;
}





















