/******************************************************************************
 The computer software and associated documentation called DOMAK hereinafter
 referred to as the WORK which is more particularly identified and described in
 Appendix A of the file LICENSE.  Conditions and restrictions for use of
 this package are also in this file.

 This routine was developed by Robert B. Russell

 The WORK was developed by:
        Asim S. Siddiqui and Geoffrey J. Barton
        Laboratory of Molecular Biophysics
        University of Oxford
        Rex Richards Building
        South Parks Road
        Oxford OX1 3QU U.K.
        Tel:  (+44) 865-275379
        FAX:  (+44) 865-510454
        INTERNET: as@bioch.ox.ac.uk
        JANET:    as@uk.ac.ox.bioch

 The WORK is Copyright (1995) University of Oxford
        Administrative Offices
        Wellington Square
        Oxford OX1 2JD U.K.

 All use of the WORK must cite:
 Siddiqui, A. S. and Barton, G. J., "Continuous and Discontinuous Domains: An
 Algorithm for the Automatic Generation of Reliable Protein Domain Definitions" 
 PROTEIN SCIENCE, 4:872-884 (1995).
*****************************************************************************/

#include <stdio.h>
#include "structs.h"

/* Determines whether two atoms (of a supplied type) are within
 *  a threshold distance of each other.  
 * 
 * We define a `contact' as an atom-atom separation less than
 *  VDW1+VDW2+0.5 A, where VDW1/2 are the Van Der Waal radii of
 *  the two atoms in question.
 *
 * We define a `potential S-S bond' as a CYS-S - CYS-S separation
 *  of less than 2.2 A.
 *
 * We define a `potential H-bond' as a N-O separation less than
 *   3.0 A.
 *
 * We define an `electrostatic contact' as any of the following
 *   contacts:
 *	O - O
 *	O - S
 *	O - P
 *	N - N
 *	N - S
 *	N - P
 * We define a `half electrostatic contact' as any of the following
 *  contacts
 *	C - O
 *	C - N
 *  	C - P
 * We define a `hydrophobic contact' as any C--C or C--S distance less
 *  than 5.0 angstroms.
 *
 * Unless it is a H-bond or a S-S, a contact is too close if it is less
 *  than two angtroms.  If it is either of these types, then it is
 *  too close if it is less than 1.2 angstroms.
 */

int check(name1,ats1,name2,ats2,sqD,hbond,elec,s_s,half_elec,hydrophobic,tooclose,SQHYDROPHOBIC,VDW,parms)
char *name1,*name2;
float *ats1, *ats2;
float *sqD;
int *hbond,*elec,*s_s,*half_elec,*hydrophobic,*tooclose;
float SQHYDROPHOBIC;
struct radii *VDW;
struct parameters parms;
{
	int type1,type2; /* types: 1=C, 2=N, 3=O, 4=S, 5=P, 6=I, 7=? */
	float threshold;
	float sqdist();
	threshold=0.0;

	if(name1[1]=='C' && name2[1]=='C') {
	   type1=type2=1;
	   threshold=SQHYDROPHOBIC;
	} else {
	   incthresh(&name1[1],&threshold,&type1,VDW);
	   incthresh(&name2[1],&threshold,&type2,VDW);
	   threshold+=parms.AA_DISTANCE;
	   threshold*=threshold;
	}

	(*half_elec)=(*hydrophobic)=(*hbond)=(*s_s)=(*elec)=(*tooclose)=0;
	(*sqD)=sqdist(ats1,ats2);
	if(*sqD<=threshold) { 
	  if ((type1==1 && type2==1) || (type1==1 && type2==4) || (type1==4 && type2==1))  {
		 (*hydrophobic)=1;
		 (*tooclose)=((*sqD)<=parms.SQTOOCLOSE);
	  } else if( (type1==4) && (type2==4) && ((*sqD)<parms.SQDISUL) ) {
		 (*s_s)=1;
		 (*tooclose)=((*sqD)<=parms.SQMUCHTOOCLOSE);
	  } else if( (type1==1 || type2==1) &&
		   (type1==2 || type2==2 || type1==3 || type2==3 ||
		    type1==5 || type2==5)
	         ) {
		 (*half_elec)=1;
		 (*tooclose)=((*sqD)<=parms.SQTOOCLOSE);
	  } else if( (((type1==2) && (type2==3)) || ((type2==2) && (type1==3))) && ((*sqD)<parms.SQHBOND) ) {
		 (*hbond)=1;
		 (*elec)=1;
		 (*tooclose)=((*sqD)<=parms.SQMUCHTOOCLOSE);
	  } else if( ( type1==2 || type1==3 || type1==4 || type1==5) &&
		   ( type2==2 || type2==3 || type2==4 || type2==5) ) {
		 (*elec)=1;
		 (*tooclose)=((*sqD)<=parms.SQTOOCLOSE);
	  } else {
		 (*tooclose)=((*sqD)<=parms.SQTOOCLOSE);
	  }
	  
	  return 1; 
	} else 
	  return 0;
}

int incthresh(atom,threshold,type,VDW)
/* returns the literature VDW radii */
char *atom;
float *threshold;
int *type;
struct radii *VDW;
{
	*threshold+=VDW[(int)((char)(*atom)-'A')].radius;
	switch((*atom)) {
	  case 'C': { *type=1; } break;
	  case 'N': { *type=2; } break;
	  case 'O': { *type=3; } break;
	  case 'S': { *type=4; } break;
	  case 'P': { *type=5; } break;
	  case 'I': { *type=6; } break; 
	  default:  { *type=7; }
	}
}
int old_incthresh(atom,threshold,type)
/* returns the literature VDW radii */
int atom;
float *threshold;
int *type;
{
	switch(atom) {
	case 'C': { *threshold+=1.7; *type=1; } break; /* From Schultz and Schirmer p 33 */
	case 'N': { *threshold+=1.6; *type=2; } break; /* " */
	case 'O': { *threshold+=1.5; *type=3; } break; /* " */  
	case 'S': { *threshold+=1.8; *type=4; } break; /* " */  
	case 'P': { *threshold+=1.9; *type=5; } break; /* Taken from Linus Pauling's */
	case 'I': { *threshold+=2.15; *type=6; } break; /* "The Nature of the Chemical Bond" p260 */
	default:  { *threshold+=2.00; *type=7; } break; /* Default value is 2.0 angstroms, be warned */
	}
}
