/******************************************************************************
 The computer software and associated documentation called DOMAK hereinafter
 referred to as the WORK which is more particularly identified and described in
 Appendix A of the file LICENSE.  Conditions and restrictions for use of
 this package are also in this file.

 This routine was written by Asim S. Siddiqui

 The WORK is Copyright (1995) A. S. Siddiqui and G. J. Barton

 All use of the WORK must cite:
 Siddiqui, A. S. and Barton, G. J., "Continuous and Discontinuous Domains: An
 Algorithm for the Automatic Generation of Reliable Protein Domain Definitions" 
 PROTEIN SCIENCE, 4:872-884 (1995).
*****************************************************************************/

/*
 * Title
 *    asd_make_misc_domains_2.c
 * Purpose
 *    contains the asd_make_single_split and asd_multi_split_sep routines
 * Author
 *    Asim Siddiqui
 * SccsId
 *    %W%   %U%   %E%
 */

#include <rdssp.h>
#include <local.h>
#include <stdio.h>
#include <asd_structs.h>
#include <asm_mop.h>
#include <ase_error.h>
#include <ass_stamp_utils.h>
#include <asd_make_domains.h>
#include <asd_make_domains_2.h>
#include <asd_mds_utils2.h>
#include <asd_mds_utils3.h>
#include <asstr_util.h>
#include <asd_utils.h>
#include <asd_check_utils.h>
#include <asd_glob_interface.h>

extern Asd_Parameters params;

/*
 * Name:
 *    asd_make_single_split
 * Puprose:
 *    extracts a single continuous domain from the protein
 */
Asd_Domain_List *
asd_make_single_split(Asd_Domain *d_by_choice, int *num_d_by_choice,
                      Asd_Domain *d_to_split, int num_atoms,
                      Asd_Contact_Info **contact_info,
                      Asd_Contact_Info **contact_rev_info,
                      struct brookn *bn, int start_chain, int end_chain,
                      char *ss, bool_t divide_further)
{
    int increment;                 /* testing increment               */
    int i;                         /* loop counter                    */
    int s1_s;                      /* start of segment 1              */
    int s1_e;                      /* end   of segment 1              */
    int s1_sn;                     /* start of segment 1              */
    int s1_en;                     /* end   of segment 1              */
    int total_contacts;            /* total number of contacts        */
    int flag;                      /* status of getting residues      */
    bool_t ss_only;                /* whether to use ss only          */
    bool_t s_ok;                   /* whether single split is ok      */
    Asd_Domain_List *d_list1;      /* list of domains divided up      */
    Asd_Domain_List *d_list2;      /* list of domains divided up      */
    Asd_Domain_List *d_list3;      /* list of domains divided up      */
    Asd_Domain *max_domain_s;      /* most likely single domain       */
    Asd_Domain *d_left_over;       /* d's left over to be analysed    */
    Asd_Seg_Cont s_conts;          /* segment contacts                */
    Asd_Contact_List *c_list; /* list of contacts             */

    d_list1 = NULL;
    d_list2 = NULL;
    d_list3 = NULL;

    d_to_split->type = 1;
    d_to_split->ok = TRUE;
    if (divide_further) {
        asd_d_comments(6, d_to_split, bn);
    } /*if*/
    if ((d_to_split->end1 - d_to_split->start1 + 1) < params.MIN_DOMAIN_SIZE) {
        d_list1 = (Asd_Domain_List *) asm_malloc(sizeof(Asd_Domain_List));
        d_list1->n_b_left_over = 1;
        d_list1->n_d_list = 0;
        d_list1->bits_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
        d_list1->bits_left_over->type = 1;
        d_list1->bits_left_over->start1 = d_to_split->start1;
        d_list1->bits_left_over->end1 = d_to_split->end1;
        d_list1->bits_left_over->d_assigned = -1;
        d_list1->bits_left_over->glob = LARGE_NEG_GLOB;
        d_list1->d_split_up = d_to_split;
        d_list1->bits_left_over->id = asstr_save(d_to_split->id);
        d_list1->bits_left_over->ok = TRUE;
        if (divide_further) {
            asd_d_comments(1, d_list1->bits_left_over, bn);
        } /*if*/
        return(d_list1);
    } /*if*/
    if ((d_to_split->end1 - d_to_split->start1 + 1) < (params.MIN_DOMAIN_SIZE * 2)) {
        d_list1 = (Asd_Domain_List *) asm_malloc(sizeof(Asd_Domain_List));
        d_list1->n_d_list = 1;
        d_list1->n_b_left_over = 0;
        d_list1->domains = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
        d_list1->domains->type = 1;
        d_list1->domains->start1 = d_to_split->start1;
        d_list1->domains->end1 = d_to_split->end1;
        d_list1->domains->glob = LARGE_NEG_GLOB;
        d_list1->d_split_up = d_to_split;
        d_list1->domains->id = asstr_save(d_to_split->id);
        d_list1->domains->ok = TRUE;
        if (divide_further) {
            asd_d_comments(2, d_list1->domains, bn);
        } /*if*/
        return(d_list1);
    } /*if*/

/* params.MIN_NO_CONTACT_CUTOFF_END * 2 <= MIN_DOMAIN_SIZE else problems */
    if ((params.MIN_NO_CONTACT_CUTOFF_END * 2) > params.MIN_DOMAIN_SIZE) {
        ase_error_fatal("asd_make_single_split", "error in #define's");
    } /*if*/

    ss_only = asd_ss_per(ss, d_to_split, &increment);
    c_list = asd_total_contacts(d_to_split, ss, ss_only, contact_info,
                                        contact_rev_info, bn, &total_contacts);

/* set up start and end positions */
    s1_s = d_to_split->start1;
    s1_e = s1_s + params.MIN_DOMAIN_SIZE - 1;
    s1_sn = s1_s;
    s1_en = s1_e;

    max_domain_s = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
    max_domain_s->value = -1.0;
    max_domain_s->type = 1;

    while (s1_sn <= (d_to_split->end1 + 1 - params.MIN_DOMAIN_SIZE)) {
        flag = asd_calc_res_from_scratch_s(contact_info, contact_rev_info,
                                    d_to_split,
                                    max_domain_s, &s_conts, ss_only, ss,
                                    start_chain, end_chain,
                                    s1_sn, s1_en, c_list);
        s1_s = s1_sn;
        s1_e = s1_en;
            
        while (s1_en <= d_to_split->end1 && flag !=2) {
            flag = asd_calc_res_given_s(contact_info, contact_rev_info,
                                       d_to_split, max_domain_s, &s_conts,
                                       ss_only, ss, start_chain, end_chain,
                                       s1_sn,  s1_en, s1_s, s1_e,
                                       total_contacts, c_list, increment, bn);
            if (flag == 0) {
                s1_e = s1_en;
            } /*if*/
            if (s1_en == d_to_split->end1) {
                s1_en = s1_en + increment;
            } else {
                if (ss_only) {
                    i = s1_en - 1;
                    i++;
                    while (ss[i] != 'H' && ss[i] != 'E' &&
                           i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    if (i == s1_en) {
                        s1_en = s1_en + increment;
                    } else {
                        s1_en = i;
                    } /*if*/
                } else {
                    s1_en = s1_en + increment;
                } /*if*/
                if (s1_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_END) &&
                             d_to_split->end1 == end_chain) {
                    s1_en = d_to_split->end1;
                } else if (s1_en > (d_to_split->end1 -
                           params.MIN_SEGMENT_SIZE_MID) &&
                           d_to_split->end1 != end_chain) {
                    s1_en = d_to_split->end1;
                } /*if*/
            } /*if*/
            if (s1_sn == d_to_split->start1 && s1_en == d_to_split->end1) {
                flag = 2;
            } /*if*/ 
        } /*while*/
        s1_s = s1_sn;
        if (ss_only) {
            i = s1_sn - 1;
            i++;
            while (ss[i] != 'H' && ss[i] != 'E' &&
                   i < d_to_split->end1) {
                i++;
            } /*while*/
            if (i == s1_sn) {
                s1_sn = s1_sn + increment;
            } else {
                s1_sn = i;
            } /*if*/
        } else {
            s1_sn = s1_sn + increment;
        } /*if*/
        if (d_to_split->start1 == start_chain &&
                   s1_sn < (start_chain + params.MIN_SEGMENT_SIZE_END)) {
            s1_sn = start_chain + params.MIN_SEGMENT_SIZE_END;
        } else if (d_to_split->start1 != start_chain &&
                   s1_sn < (d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID)) {
            s1_sn = d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID;
        } /*if*/
        s1_en = s1_sn + params.MIN_DOMAIN_SIZE - 1;
        if ((s1_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_END) &&
             d_to_split->end1 == end_chain) ||
            (s1_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_MID) &&
             d_to_split->end1 != end_chain)) {
            s1_en = d_to_split->end1;
        } /*if*/
            
    } /*while*/

    if ((ss_only && max_domain_s->value < params.MIN_PEAK_SS_ONLY_C) ||
       (!ss_only && max_domain_s->value < params.MIN_PEAK_C)) {
        s_ok = FALSE;
    } else {
        s_ok = TRUE;
    } /*if*/

    if (max_domain_s->value > 0.0) {
        if (s_ok) {
            max_domain_s->ok = TRUE;
        } else {
            max_domain_s->ok = FALSE;
        } /*if*/
        asd_focus_range(max_domain_s, d_to_split, contact_info,
                        contact_rev_info, bn, start_chain, end_chain, ss);
        max_domain_s->ok = TRUE;
    } else {
        max_domain_s->ok = FALSE;
    } /*if*/

    if (s_ok == 0 && divide_further) {
        d_list1 = (Asd_Domain_List *) asm_malloc(sizeof(Asd_Domain_List));
        d_list1->n_d_list = 1;
        d_list1->n_b_left_over = 0;
        d_list1->domains = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
        d_list1->domains->value = max_domain_s->value;
        d_list1->domains->type = 1;
        d_list1->domains->start1 = d_to_split->start1;
        d_list1->domains->end1 = d_to_split->end1;
        d_list1->domains->glob = LARGE_NEG_GLOB;
        d_list1->d_split_up = d_to_split;
        d_list1->domains->id = asstr_save(d_to_split->id);
        if (max_domain_s->ok == FALSE) {
            d_list1->domains->ok = FALSE;
        } else {
            d_list1->domains->ok = TRUE;
        } /*if*/
        if (divide_further) {
            asd_d_comments(3, d_list1->domains, bn);
            max_domain_s->id = asstr_cat_safe(asstr_save(d_to_split->id), ".1");
            asd_d_comments(4, max_domain_s, bn);
        } /*if*/
        d_list1->domains->ok = TRUE;
        asd_free_c_list(d_to_split, c_list);
        return(d_list1);
    } /*if*/

    if (divide_further) {
        max_domain_s->id = asstr_cat_safe(asstr_save(d_to_split->id), ".1");
        asd_d_comments(5, max_domain_s, bn);
/* if 2 parts */
        if (max_domain_s->start1 != d_to_split->start1 &&
            max_domain_s->end1 != d_to_split->end1) {
            d_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain) * 2);
            d_left_over[0].start1 = d_to_split->start1;
            d_left_over[0].end1 = max_domain_s->start1 - 1;
            d_left_over[1].start1 =  max_domain_s->end1 + 1;
            d_left_over[1].end1 = d_to_split->end1;
/* if pair correlated analyse together else analyse seperately */
            if (asd_check_correlation(contact_info, contact_rev_info, bn,
                                      d_left_over, ss, 2, c_list)) {
                asd_free_c_list(d_to_split, c_list);
                d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                                 ".2");
                d_list1 = asd_multi_split_sep(contact_info, contact_rev_info,
                                    bn, d_left_over, d_by_choice,
                                    num_d_by_choice,
                                    start_chain, end_chain, ss);
                d_list2 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  max_domain_s, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
            } else {
                asd_free_c_list(d_to_split, c_list);
                d_list1 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  max_domain_s, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                                 ".2");
                d_list2 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  d_left_over, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_left_over[1].id = asstr_cat_safe(asstr_save(d_to_split->id),
                                                 ".3");
                d_list3 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  &(d_left_over[1]), num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
            } /*if*/
        } else {
/* analyse both parts seperately */
            d_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
            if (max_domain_s->start1 == d_to_split->start1) {
                d_left_over->start1 = max_domain_s->end1 + 1;
                d_left_over->end1 = d_to_split->end1;
            } else if (max_domain_s->end1 == d_to_split->end1) {
                d_left_over->start1 = d_to_split->start1;
                d_left_over->end1 = max_domain_s->start1 - 1;
            } /*if*/
            asd_free_c_list(d_to_split, c_list);
            d_list1 = asd_make_multiple_domains(d_by_choice, num_d_by_choice,
                                  max_domain_s, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
            d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                             ".2");
            d_list2 = asd_make_multiple_domains(d_by_choice, num_d_by_choice,
                                  d_left_over, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
        } /*if*/
    } else {
        d_list1 = (Asd_Domain_List *) asm_malloc(sizeof(Asd_Domain_List));
        d_list1->n_d_list = 1;
        d_list1->n_b_left_over = 0;
        d_list1->domains = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
        d_list1->domains = max_domain_s;
        d_list1->domains->glob = LARGE_NEG_GLOB;
        d_list1->d_split_up = d_to_split;
        asd_free_c_list(d_to_split, c_list);
        return(d_list1);
    } /*if*/

    return(asd_check_globularity(d_list1, d_list2, d_list3, NULL, bn,
                                 d_to_split));

} /*asd_make_single_split*/

/* 
 * Name
 *   asd_multi_split_sep
 * Purpose
 *   to further subdivide a domain that is made up of two seperate segements
 */
Asd_Domain_List *
asd_multi_split_sep(Asd_Contact_Info **contact_info,
                    Asd_Contact_Info **contact_rev_info,
                    struct brookn *bn, Asd_Domain *d_to_split,
                    Asd_Domain *d_by_choice, int *num_d_by_choice,
                    int start_chain, int end_chain, char *ss)
{
    int increment;            /* testing increment            */
    int num_atoms;            /* dummy variable               */
    int i;                    /* loop counter                 */
    int s1_s;                 /* start of segment 1           */
    int s1_e;                 /* end   of segment 1           */
    int s2_s;                 /* start of segment 2           */
    int s2_e;                 /* end   of segment 2           */
    int s1_sn;                /* start of segment 1           */
    int s1_en;                /* end   of segment 1           */
    int s2_sn;                /* start of segment 2           */
    int s2_en;                /* end   of segment 2           */
    int flag;                 /* boolean flag                 */
    int level_f;              /* boolean flag                 */
    int level_e;              /* boolean flag                 */
    int split1;               /* whether can split segment1   */
    int split2;               /* whether can split segment1   */
    int total_contacts;       /* total number of contacts     */
    bool_t m_ok;              /* whether split was ok         */
    bool_t ss_only;           /* whether to use ss only       */
    Asd_Seg_Cont s_conts;     /* segment contacts             */
    Asd_Domain_List *d_list1; /* list of domains divided up   */
    Asd_Domain_List *d_list2; /* list of domains divided up   */
    Asd_Domain_List *d_list3; /* list of domains divided up   */
    Asd_Domain *max_domain_m; /* most likely multi domain     */
    Asd_Domain *d_left_over;  /* d's left over to be analysed */
    Asd_Domain d_temp;        /* temporary domain             */
    Asd_Contact_List *c_list; /* list of contacts             */

/* first check to see which if any segment needs splitting */
    num_atoms = 0;
    d_list1 = NULL;
    d_list2 = NULL;
    d_list3 = NULL;
    d_to_split->type = 2;
    d_to_split->ok = TRUE;
    asd_d_comments(6, d_to_split, bn);
    split1 = 1;
    split2 = 1;
    if (d_to_split->start1 == start_chain) {
        if ((d_to_split->end1 - d_to_split->start1 + 1) < (params.MIN_SEGMENT_SIZE_MID
             + params.MIN_SEGMENT_SIZE_END)) {
            split1 = 0;
        } /*if*/
    } else {
        if ((d_to_split->end1 - d_to_split->start1 + 1) < (params.MIN_SEGMENT_SIZE_MID
             * 2)) {
            split1 = 0;
        } /*if*/
    } /*if*/
    if (d_to_split->end2 == end_chain) {
        if ((d_to_split->end2 - d_to_split->start2 + 1) < (params.MIN_SEGMENT_SIZE_MID             + params.MIN_SEGMENT_SIZE_END)) {
            split2 = 0;
        } /*if*/
    } else {
        if ((d_to_split->end2 - d_to_split->start2 + 1) < (params.MIN_SEGMENT_SIZE_MID             * 2)) {
            split2 = 0;
        } /*if*/
    } /*if*/

/* sort out the various cases of splits */
    if (split1 == 0 && split2 == 0) {
        d_list1 = (Asd_Domain_List *) asm_malloc(sizeof(Asd_Domain_List));
        d_list1->n_d_list = 1;
        d_list1->n_b_left_over = 0;
        d_list1->domains = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
        d_list1->domains->type = 2;
        d_list1->domains->start1 = d_to_split->start1;
        d_list1->domains->end1 = d_to_split->end1;
        d_list1->domains->start2 = d_to_split->start2;
        d_list1->domains->end2 = d_to_split->end2;
        d_list1->domains->glob = LARGE_NEG_GLOB;
        d_list1->d_split_up = d_to_split;
        d_list1->domains->id = asstr_save(d_to_split->id);
        d_list1->domains->ok = TRUE;
        asd_d_comments(2, d_list1->domains, bn);
        return(d_list1);
    } /*if*/

    ss_only = asd_ss_per(ss, d_to_split, &increment);
        
    c_list = asd_total_contacts(d_to_split, ss, ss_only, contact_info,
                                contact_rev_info, bn, &total_contacts);

    max_domain_m = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
    max_domain_m->value = -1.0;
    max_domain_m->type = 2;

    if (split1 == 0) {
        s1_s = d_to_split->start1;
        s1_e = d_to_split->end1;
        s2_s = d_to_split->start2;
        s2_e = d_to_split->start2 + params.MIN_SEGMENT_SIZE_MID - 1;
        if (s2_e >= d_to_split->end2) {
            s2_s = d_to_split->end2;
        } /*if*/
        s1_sn = s1_s;
        s1_en = s1_e;
        s2_sn = s2_s;
        s2_en = s2_e;

        while ((s2_sn <= (d_to_split->end2 + 1 - params.MIN_SEGMENT_SIZE_END) &&
               d_to_split->end2 == end_chain) ||
              (s2_sn <= (d_to_split->end2 + 1 - params.MIN_SEGMENT_SIZE_MID) &&
               d_to_split->end2 != end_chain)) {

            flag = asd_calc_res_from_scratch_sep_m(contact_info,
                                            contact_rev_info,
                                            d_to_split, max_domain_m,
                                            &s_conts, ss_only, ss,
                                            start_chain, end_chain,
                                            s1_sn, s1_en, s2_sn, s2_en, 1, bn,
                                            c_list);
    
            s1_s = s1_sn;
            s1_e = s1_en;
            s2_s = s2_sn;
            s2_e = s2_en;
            while (s2_en <= d_to_split->end2 && flag !=2) {
                flag = asd_calc_res_given_sep_m(contact_info, contact_rev_info,
                                       d_to_split, max_domain_m, &s_conts,
                                       ss_only, ss, start_chain, end_chain,
                                       s1_sn,  s1_en,  s2_sn,  s2_en,
                                       s1_s,   s1_e,   s2_s,   s2_e, 1,
                                       total_contacts, bn, c_list, increment);
                if (flag == 0) {
                    s2_e = s2_en;
                } /*if*/
                if (s2_en == d_to_split->end2) {
                    s2_en = s2_en + increment;
                } else {
                    if (ss_only) {
                        i = s2_en - 1;
                        i++;
                        while (ss[i] != 'H' && ss[i] != 'E' &&
                               i < d_to_split->end1) {
                            i++;
                        } /*while*/
                        if (i == s2_en) {
                            s2_en = s2_en + increment;
                        } else {
                            s2_en = i;
                        } /*if*/
                    } else {
                        s2_en = s2_en + increment;
                    } /*if*/

                    if (s2_en > (d_to_split->end2 -
                        params.MIN_NO_CONTACT_CUTOFF_END) &&
                        d_to_split->end2 == end_chain) {
                        s2_en = d_to_split->end2;
                    } else if (s2_en >
                              (d_to_split->end2 - params.MIN_NO_CONTACT_CUTOFF_MID) &&
                               d_to_split->end2 != end_chain) {
                        s2_en = d_to_split->end2;
                    } /*if*/
                } /*if*/
                if (s2_sn == d_to_split->start2 && s2_en == d_to_split->end2) {
                    flag = 2;
                } /*if*/ 
            } /*while*/
            s2_s = s2_sn;
            if (s2_sn == d_to_split->start2) {
                s2_sn = d_to_split->start2 + params.MIN_NO_CONTACT_CUTOFF_MID;
            } else {
                if (ss_only) {
                    i = s2_sn - 1;
                    i++;
                    while (ss[i] != 'H' && ss[i] != 'E' &&
                           i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    if (i == s2_sn) {
                        s2_sn = s2_sn + increment;
                    } else {
                        s2_sn = i;
                    } /*if*/
                } else {
                    s2_sn = s2_sn + increment;
                } /*if*/
            } /*if*/
            s2_en = d_to_split->end2;
        } /*while*/

    } else if (split2 == 0) {
        s1_s = d_to_split->start1;
        if (s1_s == start_chain) {
            s1_e = d_to_split->start1 + params.MIN_SEGMENT_SIZE_END - 1;
        } else {
            s1_e = d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID - 1;
        } /*if*/
        if (s1_e >= d_to_split->end1) {
            s1_s = d_to_split->end1;
        } /*if*/
        s2_s = d_to_split->start2;
        s2_e = d_to_split->end2;
        s1_sn = s1_s;
        s1_en = s1_e;
        s2_sn = s2_s;
        s2_en = s2_e;

        while (s1_sn <= (d_to_split->end1 + 1 - params.MIN_SEGMENT_SIZE_MID)) {
            flag = asd_calc_res_from_scratch_sep_m(contact_info,
                                            contact_rev_info,
                                            d_to_split, max_domain_m,
                                            &s_conts, ss_only, ss,
                                            start_chain, end_chain,
                                            s1_sn, s1_en, s2_sn, s2_en, 2, bn,
                                            c_list);
    
            s1_s = s1_sn;
            s1_e = s1_en;
            s2_s = s2_sn;
            s2_e = s2_en;
            while (s1_en <= d_to_split->end1 && flag !=2) {
                flag = asd_calc_res_given_sep_2m(contact_info, contact_rev_info,
                                       d_to_split, max_domain_m, &s_conts,
                                       ss_only, ss, start_chain, end_chain,
                                       s1_sn,  s1_en,  s2_sn,  s2_en,
                                       s1_s,   s1_e,   s2_s,   s2_e, 2,
                                       total_contacts, bn, c_list, increment);
                if (flag == 0) {
                    s1_e = s1_en;
                } /*if*/
                if (s1_en == d_to_split->end1) {
                    s1_en = s1_en + increment;
                } else {
                    if (ss_only) {
                        i = s1_en - 1;
                        i++;
                        while (ss[i] != 'H' && ss[i] != 'E' &&
                               i < d_to_split->end1) {
                            i++;
                        } /*while*/
                        if (i == s1_en) {
                            s1_en = s1_en + increment;
                        } else {
                            s1_en = i;
                        } /*if*/
                    } else {
                        s1_en = s1_en + increment;
                    } /*if*/
                    if (s1_en > (d_to_split->end1 -
                                                   params.MIN_NO_CONTACT_CUTOFF_MID)) {
                        s1_en = d_to_split->end1;
                    } /*if*/
                } /*if*/
                if (s1_sn == d_to_split->start1 && s1_en == d_to_split->end1) {
                    flag = 2;
                } /*if*/ 
            } /*while*/
            s1_s = s1_sn;
            if (s1_sn == d_to_split->start1 &&
                        d_to_split->start1 == start_chain) {
                s1_sn = d_to_split->start1 + params.MIN_NO_CONTACT_CUTOFF_END;
            } else if (s1_sn == d_to_split->start1 &&
                        d_to_split->start1 != start_chain) {
                s1_sn = d_to_split->start1 + params.MIN_NO_CONTACT_CUTOFF_MID;
            } else {
                if (ss_only) {
                    i = s1_sn - 1;
                    i++;
                    while (ss[i] != 'H' && ss[i] != 'E' &&
                           i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    if (i == s1_sn) {
                        s1_sn = s1_sn + increment;
                    } else {
                        s1_sn = i;
                    } /*if*/
                } else {
                    s1_sn = s1_sn + increment;
                } /*if*/
            } /*if*/
            s1_en = d_to_split->end1;
        } /*while*/

    } else {
        s1_s = d_to_split->start1;
        if (s1_s == start_chain) {
            s1_e = d_to_split->start1 + params.MIN_SEGMENT_SIZE_END - 1;
        } else {
            s1_e = d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID - 1;
        } /*if*/
        s2_s = d_to_split->start2;
        s2_e = d_to_split->start2 + params.MIN_SEGMENT_SIZE_MID - 1;
        s1_sn = s1_s;
        s1_en = s1_e;
        s2_sn = s2_s;
        s2_en = s2_e;
        level_e = 0;
        while (s1_sn <= (d_to_split->end1 + 1 - params.MIN_SEGMENT_SIZE_MID)) {
            level_f = asd_check_seg_a(c_list, ss_only, ss, d_to_split, s1_sn,
                                      params.MIN_SEGMENT_SIZE_MID, start_chain, 1);
            while (s1_en <= d_to_split->end1 && level_f == 0 && level_e == 0) {
                while ((s2_sn <= (d_to_split->end2 + 1 - params.MIN_SEGMENT_SIZE_END)
                       && d_to_split->end2 == end_chain) ||
                      (s2_sn <= (d_to_split->end2 + 1 - params.MIN_SEGMENT_SIZE_MID) &&
                       d_to_split->end2 != end_chain)) {
                    flag = asd_calc_res_from_scratch_sep_m(contact_info,
                                        contact_rev_info, d_to_split,
                                        max_domain_m,
                                        &s_conts, ss_only, ss,
                                        start_chain, end_chain,
                                        s1_sn, s1_en, s2_sn, s2_en, 3, bn,
                                        c_list);
    
                    s1_s = s1_sn;
                    s1_e = s1_en;
                    s2_s = s2_sn;
                    s2_e = s2_en;
                    while (s2_en <= d_to_split->end2 && flag !=2) {
                        flag = asd_calc_res_given_sep_m(contact_info,
                                       contact_rev_info, d_to_split,
                                       max_domain_m, &s_conts, ss_only, ss,
                                       start_chain, end_chain,
                                       s1_sn,  s1_en,  s2_sn,  s2_en,
                                       s1_s,   s1_e,   s2_s,   s2_e, 3,
                                       total_contacts, bn, c_list, increment);
                        if (flag == 0) {
                            s2_e = s2_en;
                        } /*if*/
                        if (s2_en == d_to_split->end2) {
                            s2_en = s2_en + increment;
                        } else {
                            if (ss_only) {
                                i = s2_en - 1;
                                i++;
                                while (ss[i] != 'H' && ss[i] != 'E' &&
                                       i < d_to_split->end1) {
                                    i++;
                                } /*while*/
                                if (i == s2_en) {
                                    s2_en = s2_en + increment;
                                } else {
                                    s2_en = i;
                                } /*if*/
                            } else {
                                s2_en = s2_en + increment;
                            } /*if*/
                            if (s1_sn == d_to_split->start1 &&
                                s1_en == d_to_split->end1) {
                                if (s2_en > (d_to_split->end2 -
                                    params.MIN_NO_CONTACT_CUTOFF_END) &&
                                    d_to_split->end2 == end_chain) {
                                    s2_en = d_to_split->end2;
                                } else if (s2_en >
                                          (d_to_split->end2 -
                                           params.MIN_NO_CONTACT_CUTOFF_MID) &&
                                           d_to_split->end2 != end_chain) {
                                    s2_en = d_to_split->end2;
                                } /*if*/
                            } else {
                                if (s2_en > (d_to_split->end2 -
                                    params.MIN_SEGMENT_SIZE_END) &&
                                    d_to_split->end2 == end_chain) {
                                    s2_en = d_to_split->end2;
                                } else if (s2_en >
                                          (d_to_split->end2 -
                                           params.MIN_SEGMENT_SIZE_MID) &&
                                           d_to_split->end2 != end_chain) {
                                    s2_en = d_to_split->end2;
                                } /*if*/
                            } /*if*/
                        } /*if*/
                        if (s2_sn == d_to_split->start2 &&
                            s2_en == d_to_split->end2 &&
                            s1_sn == d_to_split->start1 &&
                            s1_en == d_to_split->end1) {
                            flag = 2;
                        } /*if*/ 
                    } /*while*/
                    s2_s = s2_sn;
                    if (ss_only) {
                        i = s2_sn - 1;
                        i++;
                        while (ss[i] != 'H' && ss[i] != 'E' &&
                               i < d_to_split->end1) {
                            i++;
                        } /*while*/
                        if (i == s2_sn) {
                            s2_sn = s2_sn + increment;
                        } else {
                            s2_sn = i;
                        } /*if*/
                    } else {
                        s2_sn = s2_sn + increment;
                    } /*if*/
                    if (s1_sn == d_to_split->start1 &&
                        s1_en == d_to_split->end1) {
                        if (s2_sn < (d_to_split->start2 +
                            params.MIN_NO_CONTACT_CUTOFF_MID)) {
                            s2_sn = d_to_split->start2 +
                                    params.MIN_NO_CONTACT_CUTOFF_MID;
                        } /*if*/
                    } else {
                        if (s2_sn < (d_to_split->start2 +
                                    params.MIN_SEGMENT_SIZE_MID)) {
                            s2_sn = d_to_split->start2 +
                                    params.MIN_SEGMENT_SIZE_MID;
                        } /*if*/
                    } /*if*/
                    s2_en = d_to_split->end2;
                } /*while*/
                s1_e = s1_en;
                if (s1_en == d_to_split->end1) {
                    level_e = 1;
                } /*if*/
                if (ss_only) {
                    i = s1_en - 1;
                    i++;
                    while (ss[i] != 'H' && ss[i] != 'E' &&
                           i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    if (i == s1_en) {
                        s1_en = s1_en + increment;
                    } else {
                        s1_en = i;
                    } /*if*/
                } else {
                    s1_en = s1_en + increment;
                } /*if*/
                if (s1_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_MID)) {
                    s1_en = d_to_split->end1;
                } /*if*/
                s2_sn = d_to_split->start2;
                s2_en = d_to_split->start2 + params.MIN_SEGMENT_SIZE_MID - 1;
            } /*while*/
            s1_s = s1_sn;
            level_e = 0;
            if (ss_only) {
                i = s1_sn - 1;
                i++;
                while (ss[i] != 'H' && ss[i] != 'E' &&
                       i < d_to_split->end1) {
                    i++;
                } /*while*/
                if (i == s1_sn) {
                    s1_sn = s1_sn + increment;
                } else {
                    s1_sn = i;
                } /*if*/
            } else {
                s1_sn = s1_sn + increment;
            } /*if*/
            if (d_to_split->start1 == start_chain &&
                s1_sn < (start_chain + params.MIN_SEGMENT_SIZE_END)) {
                s1_sn = start_chain + params.MIN_SEGMENT_SIZE_END;
            } else if (d_to_split->start1 != start_chain &&
                       s1_sn < (d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID)) {
                s1_sn = d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID;
            } /*if*/
            s1_en = d_to_split->end1;
            s2_sn = d_to_split->start2;
            s2_en = d_to_split->start2 + params.MIN_SEGMENT_SIZE_MID - 1;
            if (s2_en > (d_to_split->end2 - params.MIN_SEGMENT_SIZE_MID) &&
                d_to_split->end2 != end_chain) {
                s2_en = d_to_split->end2;
            } else if (s2_en > (d_to_split->end2 - params.MIN_SEGMENT_SIZE_END) &&
                       d_to_split->end2 == end_chain) {
                s2_en = d_to_split->end2;
            } /*if*/
        } /*while*/
    } /*if*/

/* now analyse results */
/* if peak not big enough */
    if ((ss_only && max_domain_m->value < params.MIN_PEAK_SS_ONLY_C) ||
       (!ss_only && max_domain_m->value < params.MIN_PEAK_C)) {
        m_ok = 0;
    } else {
        m_ok = 1;
    } /*if*/

    if (max_domain_m->value > 0.0) {
        if (m_ok) {
            max_domain_m->ok = TRUE;
        } else {
            max_domain_m->ok = FALSE;
        } /*if*/
        asd_focus_range(max_domain_m, d_to_split, contact_info,
                        contact_rev_info, bn, start_chain, end_chain, ss);
        max_domain_m->ok = TRUE;
    } else {
        max_domain_m->ok = FALSE;
    } /*if*/
    if ((ss_only && max_domain_m->value < params.MIN_PEAK_SS_ONLY_C) ||
       (!ss_only && max_domain_m->value < params.MIN_PEAK_C)) {
        d_list1 = (Asd_Domain_List *) asm_malloc(sizeof(Asd_Domain_List));
        d_list1->n_d_list = 1;
        d_list1->n_b_left_over = 0;
        d_list1->domains = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
        d_list1->domains->type = 2;
        d_list1->domains->start1 = d_to_split->start1;
        d_list1->domains->end1 = d_to_split->end1;
        d_list1->domains->start2 = d_to_split->start2;
        d_list1->domains->end2 = d_to_split->end2;
        d_list1->domains->glob = LARGE_NEG_GLOB;
        d_list1->domains->id = asstr_save(d_to_split->id);
        if (max_domain_m->ok == FALSE) {
            d_list1->domains->ok = FALSE;
        } else {
            d_list1->domains->ok = TRUE;
        } /*if*/
        d_list1->d_split_up = d_to_split;
        asd_d_comments(3, d_list1->domains, bn);
        d_list1->domains->ok = TRUE;
        max_domain_m->id = asstr_cat_safe(asstr_save(d_to_split->id), ".1");
        asd_d_comments(4, max_domain_m, bn);
        asd_free_c_list(d_to_split, c_list);
        return(d_list1);
    } /*if*/

    if (1) {
        max_domain_m->id = asstr_cat_safe(asstr_save(d_to_split->id), ".1");
        asd_d_comments(5, max_domain_m, bn);
        if (split1 == 0) {
            d_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
            if (max_domain_m->start2 == d_to_split->start2) {
                d_left_over->start1 = max_domain_m->end2 + 1;
                d_left_over->end1 = d_to_split->end2;
            } else {
                d_left_over->start1 = d_to_split->start2;
                d_left_over->end1 = max_domain_m->start2 - 1;
            } /*if*/
            asd_free_c_list(d_to_split, c_list);
            d_list1 = asd_multi_split_sep(contact_info, contact_rev_info, bn,
                                max_domain_m, d_by_choice, num_d_by_choice,
                                start_chain, end_chain, ss);
            d_list1->d_split_up = max_domain_m;
            d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                             ".2");
            d_list2 = asd_make_multiple_domains(d_by_choice, num_d_by_choice,
                                  d_left_over, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
            d_list2->d_split_up = d_left_over;
        } else if (split2 == 0) {
            d_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
            if (max_domain_m->start1 == d_to_split->start1) {
                d_left_over->start1 = max_domain_m->end1 + 1;
                d_left_over->end1 = d_to_split->end1;
            } else {
                d_left_over->start1 = d_to_split->start1;
                d_left_over->end1 = max_domain_m->start1 - 1;
            } /*if*/
            asd_free_c_list(d_to_split, c_list);
            d_list1 = asd_multi_split_sep(contact_info, contact_rev_info, bn,
                                max_domain_m, d_by_choice, num_d_by_choice,
                                start_chain, end_chain, ss);
            d_list1->d_split_up = max_domain_m;
            d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                             ".2");
            d_list2 = asd_make_multiple_domains(d_by_choice, num_d_by_choice,
                                  d_left_over, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
            d_list2->d_split_up = d_left_over;
        } else {
            d_list1 = asd_multi_split_sep(contact_info, contact_rev_info, bn,
                                max_domain_m, d_by_choice, num_d_by_choice,
                                start_chain, end_chain, ss);
            d_list1->d_split_up = max_domain_m;
            if (max_domain_m->start1 == d_to_split->start1 &&
                max_domain_m->end1 == d_to_split->end1) {
                d_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
                if (max_domain_m->start2 == d_to_split->start2) {
                    d_left_over->start1 = max_domain_m->end2 + 1;
                    d_left_over->end1 = d_to_split->end2;
                } else {
                    d_left_over->start1 = d_to_split->start2;
                    d_left_over->end1 = max_domain_m->start2 - 1;
                } /*if*/
                asd_free_c_list(d_to_split, c_list);
                d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                                 ".2");
                d_list2 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  d_left_over, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_list2->d_split_up = d_left_over;
            } else if (max_domain_m->start2 == d_to_split->start2 &&
                       max_domain_m->end2 == d_to_split->end2) {
                d_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
                if (max_domain_m->start1 == d_to_split->start1) {
                    d_left_over->start1 = max_domain_m->end1 + 1;
                    d_left_over->end1 = d_to_split->end1;
                } else {
                    d_left_over->start1 = d_to_split->start1;
                    d_left_over->end1 = max_domain_m->start1 - 1;
                } /*if*/
                asd_free_c_list(d_to_split, c_list);
                d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                                 ".2");
                d_list2 = asd_make_multiple_domains(d_by_choice,
                                  num_d_by_choice,
                                  d_left_over, num_atoms, contact_info,
                                  contact_rev_info, bn, start_chain, end_chain,
                                  ss);
                d_list2->d_split_up = d_left_over;
            } else {
                d_left_over = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain) * 2);
                if (max_domain_m->start1 == d_to_split->start1) {
                    d_left_over->start1 = max_domain_m->end1 + 1;
                    d_left_over->end1 = d_to_split->end1;
                } else {
                    d_left_over->start1 = d_to_split->start1;
                    d_left_over->end1 = max_domain_m->start1 - 1;
                } /*if*/
                if (max_domain_m->start2 == d_to_split->start2) {
                    d_left_over[1].start1 = max_domain_m->end2 + 1;
                    d_left_over[1].end1 = d_to_split->end2;
                } else {
                    d_left_over[1].start1 = d_to_split->start2;
                    d_left_over[1].end1 = max_domain_m->start2 - 1;
                } /*if*/
                if (asd_check_correlation(contact_info, contact_rev_info, bn,
                                          d_left_over, ss, 2, c_list)) {
                    asd_free_c_list(d_to_split, c_list);
                    d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                                     ".2");
                    d_list2 = asd_multi_split_sep(contact_info,
                                      contact_rev_info, bn,
                                      d_left_over, d_by_choice, num_d_by_choice,
                                      start_chain, end_chain, ss);
                    d_list2->d_split_up = d_left_over;
                } else {
                    asd_free_c_list(d_to_split, c_list);
                    d_left_over->id = asstr_cat_safe(asstr_save(d_to_split->id),
                                                     ".2");
                    d_list2 = asd_make_multiple_domains(d_by_choice,
                                   num_d_by_choice,
                                   d_left_over, num_atoms, contact_info,
                                   contact_rev_info, bn, start_chain, end_chain,
                                      ss);
                    d_list2->d_split_up = d_left_over;
                    d_left_over[1].id =
                               asstr_cat_safe(asstr_save(d_to_split->id), ".3");
                    d_list3 = asd_make_multiple_domains(d_by_choice,
                                   num_d_by_choice, &(d_left_over[1]),
                                   num_atoms, contact_info, contact_rev_info,
                                   bn, start_chain, end_chain, ss);
                    d_list3->d_split_up = &(d_left_over[1]);
                } /*if*/
            } /*if*/
        } /*if*/
    } /*if*/

    return(asd_check_globularity(d_list1, d_list2, d_list3, NULL, bn,
                                 d_to_split));

} /*asd_multi_split_sep*/ 
