/*
 * Decompiled with CFR 0.152.
 */
package org.biojava3.ronn;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.Callable;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.biojava3.data.sequence.FastaSequence;
import org.biojava3.data.sequence.SequenceUtil;
import org.biojava3.ronn.InputParameters;
import org.biojava3.ronn.ModelLoader;
import org.biojava3.ronn.NullOutputStream;
import org.biojava3.ronn.ORonnModel;
import org.biojava3.ronn.Timer;

public final class ORonn
implements Callable<ORonn> {
    private static final DateFormat DATE_FORMAT = DateFormat.getDateTimeInstance(1, 1, Locale.US);
    private static final NumberFormat nformat = NumberFormat.getInstance();
    static final byte NUMBER_OF_MODELS = 10;
    private final FastaSequence sequence;
    private final ModelLoader mloader;
    private final PrintWriter out;
    private final ResultLayout layout;
    private final PrintWriter stat;
    private final Timer timer;
    private final float disorder;
    private float[] cummulativeScore;

    static {
        nformat.setMaximumFractionDigits(2);
    }

    ORonn(FastaSequence sequence, ModelLoader mloader, InputParameters params) throws NumberFormatException, IOException {
        this.sequence = sequence;
        this.mloader = mloader;
        this.out = params.getOutputWriter();
        assert (this.out != null);
        this.layout = params.getFormat();
        this.stat = params.getStatWriter();
        this.disorder = params.getDisorder();
        this.timer = new Timer(TimeUnit.MILLISECONDS);
    }

    ORonn(FastaSequence sequence, ModelLoader mloader) throws NumberFormatException, IOException {
        this.sequence = sequence;
        this.mloader = mloader;
        this.out = new PrintWriter(new NullOutputStream());
        this.layout = ResultLayout.HORIZONTAL;
        this.stat = new PrintWriter(new NullOutputStream());
        this.disorder = 0.53f;
        this.timer = new Timer(TimeUnit.MILLISECONDS);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void writeResults(float[] meanScores, char[] seqs) {
        PrintWriter printWriter = this.out;
        synchronized (printWriter) {
            this.out.println(">" + this.sequence.getId());
            if (this.layout == ResultLayout.VERTICAL) {
                int i = 0;
                while (i < meanScores.length) {
                    this.out.printf("%c\t%.2f%n", Character.valueOf(seqs[i]), Float.valueOf(meanScores[i]));
                    ++i;
                }
            } else {
                StringBuilder seqLine = new StringBuilder();
                StringBuilder resultLine = new StringBuilder();
                String spacer = "\t";
                int i = 0;
                while (i < meanScores.length) {
                    seqLine.append(seqs[i]);
                    seqLine.append("\t");
                    resultLine.append(nformat.format(meanScores[i]));
                    resultLine.append("\t");
                    ++i;
                }
                this.out.println(seqLine.toString());
                this.out.println(resultLine.toString());
            }
            this.out.println();
            this.out.flush();
        }
    }

    static boolean isValidSequence(FastaSequence fsequence) {
        assert (fsequence != null);
        return fsequence.getLength() > 19;
    }

    @Override
    public ORonn call() throws NumberFormatException, IOException {
        String seq = this.sequence.getSequence();
        int m = 0;
        while (m < 10) {
            ModelLoader.Model model = this.mloader.getModel(m);
            ORonnModel rmodel = new ORonnModel(seq, model, this.disorder);
            float[] scores = rmodel.detect();
            this.addScore(scores);
            ++m;
        }
        char[] ch = seq.toCharArray();
        float[] meanScores = this.getMeanScores();
        assert (meanScores.length == seq.length()) : "Scores are not calculated for all residues!";
        this.writeResults(meanScores, ch);
        this.stat.println(String.valueOf(this.timer.getTotalTime()) + "ms prediction completed for " + this.sequence.getId());
        return this;
    }

    private void addScore(float[] scores) {
        if (this.cummulativeScore == null) {
            this.cummulativeScore = scores;
            return;
        }
        if (this.cummulativeScore.length != scores.length) {
            throw new IllegalArgumentException("Expected " + this.cummulativeScore.length + " but get " + scores.length);
        }
        int i = 0;
        while (i < scores.length) {
            int n = i;
            this.cummulativeScore[n] = this.cummulativeScore[n] + scores[i];
            ++i;
        }
    }

    float[] getMeanScores() {
        float[] meanScores = new float[this.cummulativeScore.length];
        int i = 0;
        while (i < this.cummulativeScore.length) {
            meanScores[i] = this.cummulativeScore[i] / 10.0f;
            ++i;
        }
        return meanScores;
    }

    static void printUsage() {
        System.out.println(" \r\nJRONN version 3.1b usage 1 August 2011:\r\njava -jar JRONN_JAR_NAME -i=inputfile <OPTIONS>\r\n\r\nWhere -i=input file \r\n\tInput file can contain one or more FASTA formatted sequences.\r\n\r\nAll OPTIONS are optional\r\nSupported OPTIONS are: \r\n\t-o=output file\r\n\t-d=disorder value\r\n\t-f=V or H \r\n\t-s=statistics file\r\n\t-n=number of threads to use\r\nOPTION DETAILED DESCRIPTION:\r\n\t-o full path to the output file, if not specified \r\n\tstandard out is used\r\n\r\n\t-d the value of disorder, defaults to 0.5\r\n\r\n\t-f output format, V for vertical, where the letters \r\n\tof the sequence and corresponding disorder values are \r\n\toutput in two column layout. H for horizontal, where the\r\n\tdisorder values are provided under the letters of the \r\n\tsequence. Letters and values separated by tabulation in\r\n\tthis case. Defaults to V.\r\n\r\n\t-s the file name to write execution statistics to.\r\n\r\n\t-n the number of threads to use. Defaults to the number of \r\n\tcores available on the computer. n=1 mean sequential \r\n\tprocessing. Valid values are 1 < n < (2 x num_of_cores)\r\n\tDefault value will give the best performance.\r\n\t\r\nEXAMPLES: \r\n\r\n\tPredict disorder values for sequences from input file /home/input.fasta\r\n\toutput the results to the standard out. Use default disorder value\r\n\tand utilise all cpus available on the computer.\r\n\r\n\tjava -jar JRONN.JAR -i=/home/input.fasta\r\n\t\r\n\tPredict disorder values for sequences from input file /home/input.fasta\r\n\toutput the results in horizontal layout to the /home/jronn.out, collect \r\n\texecution statistics to /home/jronn.stat.txt file and limit the number \r\n\tof threads to two. \r\n\t\r\n\tjava -jar JRONN.JAR -i=/home/input.fasta -o=/home/jronn.out -d=0.6 -n=2 -f=H\r\n\t \r\n\tThe arguments can be provided in any order.\r\n\r\nABOUT THE PROGRAM: \t\r\n\t\r\n\tJRONN is a Java implementation of RONN. JRONN is based on RONN and uses the \r\n\tsame model data, therefore gives the same predictions. Main motivation \r\n\tbehind JRONN development was providing an implementation of RONN more \r\n\tsuitable to use by the automated analysis pipelines and web services.  \r\n\t\r\n\tOriginal version of RONN is described in Yang,Z.R., Thomson,R., \r\n\tMcMeil,P. and Esnouf,R.M. (2005) RONN: the bio-basis function neural network\r\n\ttechnique applied to the detection of natively disordered regions in proteins  \r\n\tBioinformatics 21: 3369-3376\r\n\tSee also http://www.strubi.ox.ac.uk/RONN\r\n\t\r\n\tAuthor: Peter Troshin \r\n\temail: to.petr AT gmail DOT com\r\n\t\r\n\tThis is a free software which comes with no guarantees.\r\n\tJRONN is distributed under Apache Licence version 2. The full version of \r\n\tlicence\tcan be obtained from http://www.apache.org/licenses/LICENSE-2.0\r\n\t");
    }

    static boolean isValidSequenceForRonn(FastaSequence fsequence, PrintWriter stat) {
        String sequence;
        boolean valid = true;
        String message = "";
        if (!ORonn.isValidSequence(fsequence)) {
            message = "IGNORING sequence " + fsequence.getId() + " as its too short. Minimum sequence length for disorder prediction is " + 20 + " characters!";
            stat.println(message);
            System.err.println(message);
            valid = false;
        }
        if (!SequenceUtil.isProteinSequence(sequence = fsequence.getSequence()) && !SequenceUtil.isAmbiguosProtein(sequence)) {
            message = "IGNORING sequence " + fsequence.getId() + " as it is not a protein sequence!";
            stat.println(message);
            System.err.println(message);
            valid = false;
        }
        return valid;
    }

    static void validateSequenceForRonn(FastaSequence fsequence) {
        String message = "";
        if (!ORonn.isValidSequence(fsequence)) {
            message = "IGNORING sequence " + fsequence.getId() + " as its too short. Minimum sequence length for disorder prediction is " + 20 + " characters!";
            throw new IllegalArgumentException(message);
        }
        String sequence = fsequence.getSequence();
        if (!SequenceUtil.isProteinSequence(sequence) && !SequenceUtil.isAmbiguosProtein(sequence)) {
            message = "IGNORING sequence " + fsequence.getId() + " as it is not a protein sequence!";
            throw new IllegalArgumentException(message);
        }
    }

    private static InputParameters parseArguments(String[] args) throws IOException {
        InputParameters prms = new InputParameters();
        int i = 0;
        while (i < args.length) {
            String prm = args[i].trim().toLowerCase();
            if (prm.startsWith("-i=")) {
                prms.setFilePrm(args[i], "-i=");
            }
            if (prm.startsWith("-o=")) {
                prms.setFilePrm(args[i], "-o=");
            }
            if (prm.startsWith("-d=")) {
                prms.setDisorder(prm);
            }
            if (prm.startsWith("-f=")) {
                prms.setFormat(prm);
            }
            if (prm.startsWith("-s=")) {
                prms.setFilePrm(args[i], "-s=");
            }
            if (prm.startsWith("-n=")) {
                prms.setThreadNum(prm);
            }
            ++i;
        }
        return prms;
    }

    public static void main(String[] args) throws NumberFormatException, IOException {
        if (args.length == 0 || args.length > 5) {
            ORonn.printUsage();
            System.exit(1);
        }
        InputParameters prms = ORonn.parseArguments(args);
        PrintWriter stat = prms.getStatWriter();
        stat.println("Using parameters: \n[" + prms + "]");
        if (prms.getInput() == null) {
            System.err.println("Input is not defined! ");
            ORonn.printUsage();
            System.exit(1);
        }
        stat.println("Calculation started: " + DATE_FORMAT.format(new Date()));
        Timer timer = new Timer();
        List<FastaSequence> sequences = SequenceUtil.readFasta(new FileInputStream(prms.getInput()));
        stat.println(String.valueOf(timer.getStepTime(TimeUnit.MILLISECONDS)) + "ms input file loaded");
        stat.println("Input file has " + sequences.size() + " sequences");
        ModelLoader mloader = new ModelLoader();
        mloader.loadModels();
        PrintWriter out = prms.getOutputWriter();
        assert (out != null);
        if (prms.getThreadNum() == 1) {
            stat.println("Running predictions serially");
            ORonn.predictSerial(sequences, prms, mloader);
        } else {
            stat.print("Running preditions in parallel - ");
            stat.println("Using " + prms.getThreadNum() + " threads");
            ORonn.predictParallel(sequences, prms, mloader);
        }
        stat.println("Total calculation time: " + timer.getTotalTime() + "s ");
        stat.println("Calculation completed: " + DATE_FORMAT.format(new Date()));
        stat.close();
        out.flush();
        out.close();
    }

    static void predictSerial(List<FastaSequence> fsequences, InputParameters prms, ModelLoader mloader) throws NumberFormatException, IOException {
        for (FastaSequence sequence : fsequences) {
            if (!ORonn.isValidSequenceForRonn(sequence, prms.getStatWriter())) continue;
            ORonn ronn = new ORonn(sequence, mloader, prms);
            ronn.call();
        }
    }

    static void predictParallel(List<FastaSequence> fsequences, InputParameters prms, ModelLoader mloader) throws NumberFormatException, IOException {
        PrintWriter stat = prms.getStatWriter();
        ThreadPoolExecutor executor = new ThreadPoolExecutor(prms.getThreadNum(), prms.getThreadNum(), 0L, TimeUnit.SECONDS, new SynchronousQueue<Runnable>(), new ThreadPoolExecutor.CallerRunsPolicy());
        try {
            try {
                for (FastaSequence sequence : fsequences) {
                    if (!ORonn.isValidSequenceForRonn(sequence, stat)) continue;
                    ORonn ronn = new ORonn(sequence, mloader, prms);
                    executor.submit(ronn);
                }
                executor.shutdown();
                int timeOut = fsequences.size() < 60 ? 60 : fsequences.size();
                stat.println("All task submitted. Waiting for complition for maximum of " + timeOut + " minutes");
                executor.awaitTermination(timeOut, TimeUnit.MINUTES);
            }
            catch (InterruptedException e) {
                System.err.println("Execution is terminated! Terminated by either by the system or the timeout. Maximum of 1 minute is allowed for one sequence analisys! If it took longer to complite this analysis the program is terminated.");
                e.printStackTrace();
                executor.shutdownNow();
            }
        }
        finally {
            executor.shutdownNow();
        }
    }

    static enum ResultLayout {
        VERTICAL,
        HORIZONTAL;

    }
}

