/*
 * Jalview - A Sequence Alignment Editor and Viewer (Version 2.6)
 * Copyright (C) 2010 J Procter, AM Waterhouse, G Barton, M Clamp, S Searle
 * 
 * This file is part of Jalview.
 * 
 * Jalview is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * 
 * Jalview is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with Jalview.  If not, see <http://www.gnu.org/licenses/>.
 */
package jalview.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.ContainerEvent;
import java.awt.event.ContainerListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.awt.event.WindowStateListener;
import java.net.URL;
import java.util.ArrayList;
import java.util.EventObject;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.swing.InputVerifier;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSlider;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.border.TitledBorder;
import javax.swing.event.CellEditorListener;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.table.*;

import compbio.metadata.Argument;
import compbio.metadata.Option;
import compbio.metadata.Parameter;
import compbio.metadata.Preset;
import compbio.metadata.PresetManager;
import compbio.metadata.RunnerConfig;
import compbio.metadata.ValueConstrain;
import compbio.metadata.WrongParameterException;
import compbio.metadata.ValueConstrain.Type;

import jalview.util.jarInputStreamProvider;
import jalview.ws.jws2.JabaParamStore;
import jalview.ws.jws2.JabaPreset;
import jalview.ws.jws2.Jws2Discoverer;
import jalview.ws.jws2.ParameterUtils;
import jalview.ws.jws2.Jws2Discoverer.Jws2Instance;
import jalview.ws.jws2.dm.JabaOption;
import jalview.ws.jws2.dm.JabaParameter;
import jalview.ws.params.ArgumentI;
import jalview.ws.params.OptionI;
import jalview.ws.params.ParamDatastoreI;
import jalview.ws.params.ParameterI;
import jalview.ws.params.ValueConstrainI;
import jalview.ws.params.WsParamSetI;

/**
 * job parameter editing/browsing dialog box. User can browse existing settings
 * (user + presets + Defaults), and any changes to parameters creates a modified
 * user parameter set. LOGIC: If the parameter set is modified, and its name is
 * a valid, non-existant user parameter set, then a save button is shown. If the
 * parameter set is modified and its name is a valid, extant user parameter set,
 * then an update button is shown. If user parameter set's name is edited, and
 * old name exists as a writable user parameter set, then rename button is
 * shown. If current parameter set is associated with a user defined parameter
 * set, then : if set is modifed, a 'revert' button is shown. if set is not
 * modified, a 'delete' button is shown.
 * 
 * @author JimP
 * 
 */
public class WsJobParameters extends JPanel implements ItemListener,
        ActionListener, DocumentListener
{
  URL linkImageURL = getClass().getResource("/images/link.gif");

  private static final String SVC_DEF = "Defaults"; // this is the null
                                                    // parameter set as shown to
                                                    // user

  private static final int PARAM_WIDTH = 340, PARAM_HEIGHT = 150,
          PARAM_CLOSEDHEIGHT = 80;

  private static final int OPTSET_HEIGHT = 30;

  JPanel SetNamePanel = new JPanel();

  JPanel setDetails = new JPanel();

  JSplitPane settingsPanel = new JSplitPane();

  JSplitPane jobPanel = new JSplitPane();

  boolean split = true;

  JTabbedPane tabpanels = new JTabbedPane();

  JPanel jobOptions = new JPanel();

  JScrollPane jobOptionsPane = new JScrollPane();

  JPanel jobParameters = new JPanel();

  JButton createpref = new JButton();

  JButton deletepref = new JButton();

  JButton revertpref = new JButton();

  JButton updatepref = new JButton();

  JButton startjob = new JButton();

  JButton canceljob = new JButton();

  JComboBox setName = new JComboBox();

  JTextArea setDescr = new JTextArea();

  JScrollPane paramPane = new JScrollPane();

  JPanel paramList = new JPanel();

  RunnerConfig serviceOptions;

  ParamDatastoreI paramStore;

  WsJobParameters(Jws2Instance service)
  {
    this(service, null);
  }

  public WsJobParameters(Jws2Instance service, WsParamSetI preset)
  {
    this(null, service, preset, null);
  }

  /**
   * 
   * @param desktop
   *          - if null, create new JFrame outside of desktop
   * @param service
   * @param preset
   */
  public WsJobParameters(JFrame parent, Jws2Instance service,
          WsParamSetI preset, List<Argument> jobArgset)
  {
    this(parent, null, service, preset, jobArgset);
  }

  /**
   * 
   * @param parent
   * @param paramStorei
   * @param service
   * @param preset
   * @param jobArgset
   */
  public WsJobParameters(JFrame parent, ParamDatastoreI paramStorei,
          Jws2Instance service, WsParamSetI preset, List<Argument> jobArgset)
  {
    super();
    jbInit();
    this.paramStore = paramStorei;
    if (paramStore == null)
    {
      paramStore = service.getParamStore();
    }
    this.service = service;
    // argSetModified(false);
    // populate parameter table
    initForService(service, preset, jobArgset);
    // display in new JFrame attached to parent.
    validate();
  }

  int response = -1;

  JDialog frame = null;

  /**
   * shows a modal dialog containing the parameters.
   * 
   * @return
   */
  public boolean showRunDialog()
  {

    frame = new JDialog(Desktop.instance, true);

    frame.setTitle("Edit parameters for " + service.getActionText());
    Rectangle deskr = Desktop.instance.getBounds();
    frame.setBounds(new Rectangle((int) (deskr.getCenterX() - 240),
            (int) (deskr.getCenterY() - 250), 480, 500));
    frame.setContentPane(this);
    // should recover defaults from user prefs.
    // settingsPanel.setDividerLocation(0.4);
    // jobPanel.setDividerLocation(0.5);
    frame.validate();
    if (split)
    {
      javax.swing.SwingUtilities.invokeLater(new Runnable()
      {
        public void run()
        {
          settingsPanel.setDividerLocation(0.5);
          jobPanel.setDividerLocation(setDescr.getLineCount() > 3 ? 0.35
                  : 0.16);
        }
      });
    }
    else
    {
      tabpanels.setSelectedComponent(jobOptionsPane);
    }
    frame.setVisible(true);

    if (response > 0)
    {
      return true;
    }
    return false;
  }

  private void jbInit()
  {
    updatepref = JvSwingUtils.makeButton("Update",
            "Update this existing user parameter set.",
            new ActionListener()
            {

              public void actionPerformed(ActionEvent e)
              {
                update_actionPerformed(e);
              }
            });
    deletepref = JvSwingUtils.makeButton("Delete",
            "Delete the currently selected user parameter set.",
            new ActionListener()
            {

              public void actionPerformed(ActionEvent e)
              {
                delete_actionPerformed(e);
              }
            });
    createpref = JvSwingUtils.makeButton("Create",
            "Create a new parameter set with the current settings.",
            new ActionListener()
            {

              public void actionPerformed(ActionEvent e)
              {
                create_actionPerformed(e);
              }
            });
    revertpref = JvSwingUtils.makeButton("Revert",
            "Undo all changes to the current parameter set",
            new ActionListener()
            {

              public void actionPerformed(ActionEvent e)
              {
                revert_actionPerformed(e);
              }
            });
    startjob = JvSwingUtils.makeButton("Start Job",
            "Start Job with current settings.", new ActionListener()
            {
              public void actionPerformed(ActionEvent e)
              {
                startjob_actionPerformed(e);
              }
            });
    canceljob = JvSwingUtils.makeButton("Cancel Job",
            "Close this dialog and cancel job.", new ActionListener()
            {
              public void actionPerformed(ActionEvent e)
              {
                canceljob_actionPerformed(e);
              }
            });

    setDetails.setBorder(new TitledBorder("Details"));
    setDetails.setLayout(new BorderLayout());
    setDescr.setColumns(40);
    setDescr.setWrapStyleWord(true);
    setDescr.setLineWrap(true);
    setDescr.setBackground(getBackground());
    setDescr.setEditable(true);
    setDescr.getDocument().addDocumentListener(this);
    setDescr.setToolTipText("Click to edit the notes for this parameter set.");
    JScrollPane setDescrView = new JScrollPane();
    // setDescrView.setPreferredSize(new Dimension(350, 200));
    setDescrView.getViewport().setView(setDescr);
    setName.setEditable(true);
    setName.addItemListener(this);
    setName.getEditor().addActionListener(this);
    JPanel setNameInfo = new JPanel(new FlowLayout(FlowLayout.LEFT));
    GridBagLayout gbl = new GridBagLayout();
    SetNamePanel.setLayout(gbl);
    SetNamePanel.setMinimumSize(new Dimension(300, 60));
    JLabel setNameLabel = new JLabel("Current parameter set name :");
    setNameLabel.setFont(new java.awt.Font("Verdana", Font.PLAIN, 10));
    // setNameLabel.setHorizontalAlignment(FlowLayout.LEFT);
    setNameInfo.add(setNameLabel);
    setNameInfo.add(setName);
    // initial button visibility
    updatepref.setVisible(false);
    deletepref.setVisible(false);
    revertpref.setVisible(false);
    createpref.setVisible(false);
    JPanel setsavebuts = new JPanel();
    setsavebuts.setLayout(new FlowLayout(FlowLayout.LEFT)); // GridLayout(1,2));
    ((FlowLayout) setsavebuts.getLayout()).setHgap(10);
    ((FlowLayout) setsavebuts.getLayout()).setVgap(0);
    setsavebuts.add(deletepref);
    setsavebuts.add(revertpref);
    setsavebuts.add(createpref);
    setsavebuts.add(updatepref);
    setsavebuts.setSize(new Dimension(150, 20));
    JPanel buttonArea = new JPanel(new GridLayout(1, 1));
    buttonArea.add(setsavebuts);
    SetNamePanel.add(setNameInfo);
    GridBagConstraints gbc = new GridBagConstraints();
    gbc.gridheight = 2;
    gbl.setConstraints(setNameInfo, gbc);
    SetNamePanel.add(buttonArea);
    gbc = new GridBagConstraints();
    gbc.gridx=0;
    gbc.gridy=2;
    gbc.gridheight = 1;
    gbl.setConstraints(buttonArea, gbc);
    setDetails.add(setDescrView, BorderLayout.CENTER);
    // setDetails.setPreferredSize(new Dimension(360, 100));
    jobParameters.setBorder(new TitledBorder("Parameters"));
    jobParameters.setLayout(new BorderLayout());
    paramPane.setPreferredSize(new Dimension(360, 300));
    paramPane.getVerticalScrollBar().setUnitIncrement(20);
    // paramPanel.setPreferredSize(new Dimension(360, 300));
    // TODO: relayout buttons nicely
    paramPane.getViewport().setView(paramList);
    jobParameters.add(paramPane, BorderLayout.CENTER);
    JPanel jobOptionsPanel = new JPanel();
    jobOptionsPanel.setLayout(new BorderLayout());
    jobOptionsPanel.setBorder(new TitledBorder("Options"));
    jobOptionsPane.getViewport().setView(jobOptions);
    jobOptionsPanel.add(jobOptionsPane, BorderLayout.CENTER);

    setLayout(new BorderLayout());
    jobPanel.setLeftComponent(setDetails);
    if (split)
    {
      settingsPanel.setLeftComponent(jobOptionsPanel);
      settingsPanel.setRightComponent(jobParameters);
      settingsPanel.setOrientation(JSplitPane.VERTICAL_SPLIT);
      jobPanel.setRightComponent(settingsPanel);
    }
    else
    {
      jobPanel.setRightComponent(tabpanels);
      jobOptionsPane.setName("Options");
      tabpanels.add(jobOptionsPane);
      paramPane.setName("Parameters");
      tabpanels.add(paramPane);
    }
    jobPanel.setOrientation(JSplitPane.VERTICAL_SPLIT);
    add(SetNamePanel, BorderLayout.NORTH);
    add(jobPanel, BorderLayout.CENTER);
    JPanel dialogpanel = new JPanel();
    dialogpanel.add(startjob);
    dialogpanel.add(canceljob);
    add(dialogpanel, BorderLayout.SOUTH);
  }

  protected void revert_actionPerformed(ActionEvent e)
  {
    reInitDialog(lastParmSet);

  }

  protected void update_actionPerformed(ActionEvent e)
  {
    if (isUserPreset)
    {
      String curname = ((String) setName.getSelectedItem()).trim();
      _updatePreset(lastParmSet, curname);
      lastParmSet = curname;
      isUserPreset = true;
      initArgSetModified();
      syncSetNamesWithStore();
    }
  }

  private void _deleteUserPreset(String lastParmSet2)
  {
    paramStore.deletePreset(lastParmSet2);
  }

  protected void delete_actionPerformed(ActionEvent e)
  {
    if (isUserPreset)
    {
      // delete current preset's saved entry
      _deleteUserPreset(lastParmSet);
    }
    reInitDialog(null); // service default
  }

  protected void create_actionPerformed(ActionEvent e)
  {
    String curname = ((String) setName.getSelectedItem()).trim();
    if (curname.length() > 0)
    {
      _storeCurrentPreset(curname);
      lastParmSet = curname;
      isUserPreset = true;
      initArgSetModified();
    }
    else
    {
      // TODO: show warning
      System.err.println("Invalid name. Not saved.");
    }
  }

  protected void canceljob_actionPerformed(ActionEvent e)
  {
    response = 0;
    if (frame != null)
    {
      frame.setVisible(false);
    }
  }

  protected void startjob_actionPerformed(ActionEvent e)
  {
    response = 1;
    if (frame != null)
    {
      frame.setVisible(false);
    }
  }

  Jws2Instance service;

  /**
   * list of service presets in the gui
   */
  Hashtable servicePresets = null;

  /**
   * set if dialog is being set - so handlers will avoid spurious events
   */
  boolean settingDialog = false;

  void initForService(Jws2Instance service, WsParamSetI jabap,
          List<Argument> jabajobArgset)
  {
    WsParamSetI p = null;
    List<ArgumentI> jobArgset = null;
    settingDialog = true;
    { // instantiate the abstract proxy for Jaba objects
      jobArgset = jabajobArgset == null ? null : JabaParamStore
              .getJwsArgsfromJaba(jabajobArgset);
      p = jabap; // (jabap != null) ? paramStore.getPreset(jabap.getName()) :
                 // null;
    }
    // TODO: Recover window geometry prefs for this service
    // jobPanel.setDividerLocation(proportionalLocation)
    // settingsPanel.setDividerLocation(proportionalLocation)
    Hashtable exnames = new Hashtable();
    for (int i = 0, iSize = setName.getItemCount(); i < iSize; i++)
    {
      exnames.put((String) setName.getItemAt(i), setName.getItemAt(i));
    }
    servicePresets = new Hashtable();
    // Add the default entry - if not present already.
    if (!exnames.contains(SVC_DEF))
    {
      setName.addItem(SVC_DEF);
      exnames.put(SVC_DEF, SVC_DEF);
      servicePresets.put(SVC_DEF, SVC_DEF);
    }
    String curname = (p == null ? "" : p.getName());
    for (WsParamSetI pr : paramStore.getPresets())
    {
      if (!pr.isModifiable())
      {
        servicePresets.put(pr.getName(), "preset");
      }
      else
      {
      }
      if (!exnames.contains(pr.getName()))
      {
        setName.addItem(pr.getName());
      }
    }
    // TODO: if initial jobArgset matches a given user setting or preset then
    // should recover setting accordingly
    // updateTable(p, jobArgset);
    if (p != null)
    {
      reInitDialog(p.getName());
      initArgSetModified();
    }
    else
    {
      if (jobArgset != null && jobArgset.size() > 0)
      {
        curSetName = "Supplied Settings";
        updateTable(p, jobArgset);
      }
      else
      {
        curSetName = null;
        reInitDialog(null);
      }
    }
    settingDialog = false;

  }

  @SuppressWarnings("unchecked")
  private void updateTable(WsParamSetI p, List<ArgumentI> jobArgset)
  {
    // populate table from default parameter set.
    List<ArgumentI> args = paramStore.getServiceParameters();

    // split to params and required arguments
    {
      for (ArgumentI myarg : args)
      {
        // Ideally, Argument would implement isRequired !
        if (myarg instanceof ParameterI)
        {
          ParameterI parm = (ParameterI) myarg;
          addParameter(parm);
        }
        else
        {
          if (myarg instanceof OptionI)
          {
            OptionI opt = (OptionI) myarg;
            addOption(opt).resetToDefault();
          }
          else
          {
            System.err.println("Ignoring unknown service argument type "
                    + myarg.getClass().getName());
          }
        }
      }
      args = null; // no more args to process.
    }
    if (p != null)
    {
      isUserPreset = false;
      // initialise setname
      setName.setSelectedItem(lastSetName = p.getName());
      setDescr.setText(lastDescrText = p.getDescription());
      // TODO - URL link
      try
      {
        args = p.getArguments();
      } catch (Exception e)
      {
        e.printStackTrace();
      }
      // TODO: check if args should be unselected prior to resetting using the
      // preset
    }
    else
    {
      if (lastParmSet == null)
      {
        isUserPreset = false;
        // first call - so create a dummy name

        setName.setSelectedItem(lastSetName = SVC_DEF);
      }
    }

    if (jobArgset != null)
    {
      argSetModified(jobArgset, true);
      args = jobArgset;
    }
    // get setargs from current object
    if (args != null)
    {
      for (ArgumentI arg : args)
      {
        if (arg instanceof ParameterI)
        {
          setParameter((ParameterI) arg);
        }
        else
        {
          if (arg instanceof OptionI)
          {
            // System.out.println("Setting option "
            // + System.identityHashCode(arg) + ":" + arg.getName()
            // + " with " + arg.getDefaultValue());
            selectOption((OptionI) arg, arg.getDefaultValue());
          }
        }

      }
    }

    jobOptions.setPreferredSize(new Dimension(PARAM_WIDTH, optSet.size()
            * OPTSET_HEIGHT));
    FlowLayout fl;
    jobOptions.setLayout(fl=new FlowLayout(FlowLayout.LEFT));
    
    refreshParamLayout();
    paramPane.validate();
    revalidate();
  }

  private boolean isModified()
  {
    return modifiedElements.size() > 0;
  }

  private Hashtable modifiedElements = new Hashtable();

  /**
   * reset gui and modification state settings
   */
  private void initArgSetModified()
  {
    curSetName = null;
    modifiedElements.clear();
    updateButtonDisplay();
  }

  private void updateButtonDisplay()
  {
    boolean _update = false, _create = false, _delete = false, _revert = false;
    if (modifiedElements.size() > 0)
    {
      // set modified
      _revert = true;
      _update = isUserPreset; // can only update user presets
      if (!isUserPreset || modifiedElements.containsKey(setName))
      {
        // name modified - can create new preset
        _create = true;
      }
    }
    else
    {
      // set unmodified
    }
    // can still delete a user preset
    _delete = isUserPreset;

    createpref.setVisible(_create);
    updatepref.setVisible(_update);
    deletepref.setVisible(_delete);
    revertpref.setVisible(_revert);
    validate();
  }

  private void argSetModified(Object modifiedElement, boolean b)
  {
    if (settingDialog)
    {
      return;
    }
    if (!b)
    {
      modifiedElements.remove(modifiedElement);
    }
    else
    {
      if (b && modifiedElement == setName
              && modifiedElements.contains(modifiedElement))
      {
        // HACK! prevents iteration on makeSetNameValid
        b = false;
      }
      modifiedElements.put(modifiedElement, modifiedElement);
    }
    // set mod status based on presence of elements in table
    if (b && modifiedElements.size() > 0)
    {
      makeSetNameValid(!isUserPreset);
      SetNamePanel.revalidate();
    }
    updateButtonDisplay();
  }

  private boolean isServicePreset(String selectedItem)
  {
    return selectedItem.equals(SVC_DEF)
            || servicePresets.containsKey(selectedItem);
  }

  /**
   * check if the current set name is a valid set name for saving, if not, then
   * fix it.
   */
  private void makeSetNameValid(boolean newuserset)
  {
    boolean stn = settingDialog;
    boolean renamed = false;
    settingDialog = true;
    String nm = (curSetName != null ? curSetName : (String) setName
            .getSelectedItem());
    // check if the name is reserved - if it is, rename it.
    if (isServicePreset(nm))
    {
      nm = "User " + nm;
      renamed = true;
    }
    String tnm = nm;
    if (newuserset)
    {
      int i = 0;
      while (paramStore.getPreset(tnm) != null)
      {
        tnm = nm + " (" + (++i) + ")";
        renamed = true;
      }
      if (i > 0)
      {
        nm = tnm;
      }
    }

    boolean makeupdate = false;
    // sync the gui with the preset database
    for (int i = 0, iS = setName.getItemCount(); i < iS; i++)
    {
      String snm = (String) setName.getItemAt(i);
      if (snm.equals(nm))
      {
        makeupdate = true;
        // setName.setSelectedIndex(i);
      }
    }
    if (!makeupdate)
    {
      setName.addItem(curSetName = nm);
      setName.setSelectedItem(curSetName);
    }
    if (renamed)
    {
      settingDialog = false; // we need this name change to be registered.
      argSetModified(setName, renamed);
    }
    settingDialog = stn;
  }

  private void addParameter(ParameterI arg)
  {
    ParamBox pb = paramSet.get(arg.getName());
    if (pb == null)
    {
      pb = new ParamBox(this, arg);
      paramSet.put(arg.getName(), pb);
      paramList.add(pb);
    }
    pb.init();
    // take the defaults from the parameter
    pb.updateControls(arg);
  }

  private void setParameter(ParameterI arg)
  {
    ParamBox pb = paramSet.get(arg.getName());
    if (pb == null)
    {
      addParameter(arg);
    }
    else
    {
      pb.updateControls(arg);
    }

  }

  private void selectOption(OptionI option, String string)
  {
    OptionBox cb = optSet.get(option.getName());
    if (cb == null)
    {
      cb = addOption(option);
    }
    cb.enabled.setSelected(string != null); // initial state for an option.
    if (string != null)
    {
      if (option.getPossibleValues().contains(string))
      {
        cb.val.setSelectedItem(string);
      }
      else
      {
        throw new Error("Invalid value " + string + " for option " + option);
      }

    }
    if (option.isRequired() && !cb.enabled.isSelected())
    {
      // TODO: indicate paramset is not valid.. option needs to be selected!
    }
    cb.setInitialValue();
  }

  Map<String, ParamBox> paramSet = new Hashtable<String, ParamBox>();

  public class ParamBox extends JPanel implements ChangeListener,
          ActionListener, MouseListener
  {
    JButton showDesc = new JButton();

    JTextArea string = new JTextArea();

    JScrollPane descPanel = new JScrollPane();

    JSlider slider = null;

    JTextField valueField = null;

    ValueConstrainI validator = null;

    JPanel settingPanel = new JPanel();

    JPanel controlPanel = new JPanel();

    boolean integ = false;

    boolean choice = false;

    boolean descisvisible = false;

    final WsJobParameters pmdialogbox;

    final URL finfo;

    public ParamBox(final WsJobParameters pmlayout, ParameterI parm)
    {
      pmdialogbox = pmlayout;
      setPreferredSize(new Dimension(PARAM_WIDTH, PARAM_CLOSEDHEIGHT));
      setBorder(new TitledBorder(parm.getName()));
      setLayout(null);
      showDesc.setFont(new Font("Verdana", Font.PLAIN, 6));
      showDesc.setText("+");
      string.setFont(new Font("Verdana", Font.PLAIN, 11));
      string.setBackground(getBackground());
      // string.setSize(new Dimension(PARAM_WIDTH, 80));
      string.setEditable(false);
      descPanel.getViewport().setView(string);
      // descPanel.setLocation(2,17);
      descPanel.setVisible(false);
      // string.setMinimumSize(new Dimension(140,80));
      // string.setMaximumSize(new Dimension(280,80));
      final ParamBox me = this;
      finfo = parm.getFurtherDetails();
      if (finfo != null)
      {
        showDesc.setToolTipText("<html>"
                + JvSwingUtils
                        .wrapTooltip("Click to show brief description<br><img src=\""
                                + linkImageURL
                                + "\"/> Right click for further information.")
                + "</html>");
        showDesc.addMouseListener(this);
      }
      else
      {
        showDesc.setToolTipText("<html>"
                + JvSwingUtils
                        .wrapTooltip("Click to show brief description.")
                + "</html>");
      }
      showDesc.addActionListener(new ActionListener()
      {

        public void actionPerformed(ActionEvent e)
        {
          descisvisible = !descisvisible;
          descPanel.setVisible(descisvisible);
          me.setPreferredSize(new Dimension(PARAM_WIDTH,
                  (descisvisible) ? PARAM_HEIGHT : PARAM_CLOSEDHEIGHT));
          me.validate();
          pmlayout.refreshParamLayout();
        }
      });
      string.setWrapStyleWord(true);
      string.setLineWrap(true);
      string.setColumns(32);
      string.setText(parm.getDescription());
      JPanel firstrow = new JPanel();
      firstrow.setLayout(null);
      controlPanel.setLayout(new BorderLayout());
      controlPanel.setBounds(new Rectangle(39, 10, PARAM_WIDTH - 70,
              PARAM_CLOSEDHEIGHT - 50));
      showDesc.setBounds(new Rectangle(10, 10, 16, 16));
      firstrow.add(showDesc);
      firstrow.add(controlPanel);
      firstrow.setBounds(new Rectangle(10, 20, PARAM_WIDTH - 30,
              PARAM_CLOSEDHEIGHT - 30));
      add(firstrow);
      validator = parm.getValidValue();
      parameter = parm;
      if (validator != null)
      {
        integ = validator.getType() == Type.Integer;
      }
      else
      {
        if (parameter.getPossibleValues() != null)
        {
          choice = true;
        }
      }
      updateControls(parm);
      descPanel.setBounds(new Rectangle(10, PARAM_CLOSEDHEIGHT,
              PARAM_WIDTH - 20, PARAM_HEIGHT - PARAM_CLOSEDHEIGHT - 5));
      add(descPanel);
      validate();
    }

    public void init()
    {
      // reset the widget's initial value.
      lastVal = null;
    }

    boolean adjusting = false;

    ParameterI parameter;

    JComboBox choicebox;

    public int getBoxHeight()
    {
      return (descisvisible ? PARAM_HEIGHT : PARAM_CLOSEDHEIGHT);
    }

    public void updateControls(ParameterI parm)
    {
      adjusting = true;
      boolean init = (choicebox == null && valueField == null);
      float fVal = 0f;
      int iVal = 0;
      if (init)
      {
        if (choice)
        {
          choicebox = new JComboBox();
          choicebox.addActionListener(this);
          controlPanel.add(choicebox, BorderLayout.CENTER);
        }
        else
        {
          slider = new JSlider();
          slider.addChangeListener(this);
          valueField = new JTextField();
          valueField.addActionListener(this);
          valueField.setPreferredSize(new Dimension(60, 25));
          controlPanel.add(slider, BorderLayout.WEST);
          controlPanel.add(valueField, BorderLayout.EAST);

        }
      }

      if (parm != null)
      {
        if (choice)
        {
          if (init)
          {
            List vals = parm.getPossibleValues();
            for (Object val : vals)
            {
              choicebox.addItem(val);
            }
          }

          if (parm.getDefaultValue() != null)
          {
            choicebox.setSelectedItem(parm.getDefaultValue());
          }
        }
        else
        {
          valueField.setText(parm.getDefaultValue());
        }
      }
      lastVal = updateSliderFromValueField();
      adjusting = false;
    }

    Object lastVal;

    public ParameterI getParameter()
    {
      ParameterI prm = parameter.copy();
      if (choice)
      {
        prm.setDefaultValue((String) choicebox.getSelectedItem());
      }
      else
      {
        prm.setDefaultValue(valueField.getText());
      }
      return prm;
    }

    public Object updateSliderFromValueField()
    {
      int iVal;
      float fVal;
      if (validator != null)
      {
        if (integ)
        {
          iVal = 0;
          try
          {
            valueField.setText(valueField.getText().trim());
            iVal = Integer.valueOf(valueField.getText());
            if (validator.getMin() != null
                    && validator.getMin().intValue() > iVal)
            {
              iVal = validator.getMin().intValue();
              // TODO: provide visual indication that hard limit was reached for
              // this parameter
            }
            if (validator.getMax() != null
                    && validator.getMax().intValue() < iVal)
            {
              iVal = validator.getMax().intValue();
              // TODO: provide visual indication that hard limit was reached for
              // this parameter
            }
          } catch (Exception e)
          {
          }
          ;
          if (validator.getMin() != null && validator.getMax() != null)
          {
            slider.getModel().setRangeProperties(iVal, 1,
                    validator.getMin().intValue(),
                    validator.getMax().intValue(), true);
          }
          else
          {
            slider.setVisible(false);
          }
          return new int[]
          { iVal };
        }
        else
        {
          fVal = 0f;
          try
          {
            fVal = Float.valueOf(valueField.getText());
            if (validator.getMin() != null
                    && validator.getMin().floatValue() > fVal)
            {
              fVal = validator.getMin().floatValue();
              // TODO: provide visual indication that hard limit was reached for
              // this parameter
            }
            if (validator.getMax() != null
                    && validator.getMax().floatValue() < fVal)
            {
              fVal = validator.getMax().floatValue();
              // TODO: provide visual indication that hard limit was reached for
              // this parameter
            }
          } catch (Exception e)
          {
          }
          ;
          if (validator.getMin() != null && validator.getMax() != null)
          {
            slider.getModel().setRangeProperties((int) fVal * 1000, 1,
                    (int) validator.getMin().floatValue() * 1000,
                    (int) validator.getMax().floatValue() * 1000, true);
          }
          else
          {
            slider.setVisible(false);
          }
          return new float[]
          { fVal };
        }
      }
      else
      {
        if (!choice)
        {
          slider.setVisible(false);
          return new String[]
          { valueField.getText().trim() };
        }
        else
        {
          return new String[]
          { (String) choicebox.getSelectedItem() };
        }
      }

    }

    public void stateChanged(ChangeEvent e)
    {
      if (!adjusting)
      {
        valueField.setText(""
                + ((integ) ? ("" + (int) slider.getValue())
                        : ("" + (float) (slider.getValue() / 1000f))));
        checkIfModified();
      }

    }

    public void actionPerformed(ActionEvent e)
    {
      if (adjusting)
      {
        return;
      }
      if (!choice)
      {
        updateSliderFromValueField();
      }
      checkIfModified();
    }

    private void checkIfModified()
    {
      Object cstate = updateSliderFromValueField();
      boolean notmod = false;
      if (cstate.getClass() == lastVal.getClass())
      {
        if (cstate instanceof int[])
        {
          notmod = (((int[]) cstate)[0] == ((int[]) lastVal)[0]);
        }
        else if (cstate instanceof float[])
        {
          notmod = (((float[]) cstate)[0] == ((float[]) lastVal)[0]);
        }
        else if (cstate instanceof String[])
        {
          notmod = (((String[]) cstate)[0].equals(((String[]) lastVal)[0]));
        }
      }
      pmdialogbox.argSetModified(this, !notmod);
    }

    public void mouseClicked(MouseEvent e)
    {
      if (javax.swing.SwingUtilities.isRightMouseButton(e))
      {
        showUrlPopUp(this, finfo.toString(), e.getX(), e.getY());
      }
    }

    public void mousePressed(MouseEvent e)
    {
      // TODO Auto-generated method stub

    }

    public void mouseReleased(MouseEvent e)
    {
      // TODO Auto-generated method stub

    }

    public void mouseEntered(MouseEvent e)
    {
      // TODO Auto-generated method stub

    }

    public void mouseExited(MouseEvent e)
    {
      // TODO Auto-generated method stub

    }

  }

  Map<String, OptionBox> optSet = new Hashtable<String, OptionBox>();

  public class OptionBox extends JPanel implements MouseListener,
          ActionListener
  {
    JComboBox val = new JComboBox();

    JCheckBox enabled = new JCheckBox();

    JLabel optlabel = new JLabel();

    final URL finfo;

    boolean hasLink = false;

    OptionI option;

    public OptionBox(OptionI opt)
    {
      option = opt;
      setLayout(new BorderLayout());
      enabled.setSelected(opt.isRequired()); // TODO: lock required options
      enabled.setFont(new Font("Verdana", Font.PLAIN, 11));
      enabled.setText("");
      enabled.setText(opt.getName());
      enabled.addActionListener(this);
      finfo = option.getFurtherDetails();
      if (finfo != null)
      {
        hasLink = true;
        // optlabel.setToolTipText("<html><p>"+opt.getDescription()+"</p><img src=\""+linkImageURL+"\"/></html>");
        enabled.setToolTipText("<html>"
                + JvSwingUtils.wrapTooltip(opt.getDescription()
                        + "<br><img src=\"" + linkImageURL + "\"/>")
                + "</html>");
        // optlabel.addMouseListener(this);
        enabled.addMouseListener(this);
      }
      else
      {
        // optlabel.setToolTipText(opt.getDescription());
        enabled.setToolTipText("<html>"
                + JvSwingUtils.wrapTooltip(opt.getDescription())
                + "</html>");
      }
      add(enabled, BorderLayout.NORTH);
      if (opt.getPossibleValues().size() > 1)
      {
        setLayout(new GridLayout(1, 2));
        for (Object str : opt.getPossibleValues())
        {
          val.addItem((String) str);
        }
        val.setSelectedItem((String) opt.getDefaultValue());
        val.addActionListener(this);
        add(val, BorderLayout.SOUTH);
      }
      // TODO: add actionListeners for popup (to open further info),
      // and to update list of parameters if an option is enabled
      // that takes a value.
      setInitialValue();
    }

    public void resetToDefault()
    {
      enabled.setSelected(false);
      if (option.isRequired())
      {
        // Apply default value
        selectOption(option, option.getDefaultValue());
      }
    }

    boolean initEnabled = false;

    String initVal = null;

    public void setInitialValue()
    {
      initEnabled = enabled.isSelected();
      if (option.getPossibleValues() != null
              && option.getPossibleValues().size() > 1)
      {
        initVal = (String) val.getSelectedItem();
      }
      else
      {
        initVal = (initEnabled) ? option.getDefaultValue() : null;
      }
    }

    public OptionI getOptionIfEnabled()
    {
      if (!enabled.isSelected())
      {
        return null;
      }
      OptionI opt = option.copy();

      if (val.getSelectedItem() != null)
      {
        opt.setDefaultValue((String) val.getSelectedItem());
      }
      return opt;
    }

    public void actionPerformed(ActionEvent e)
    {
      if (e.getSource() != enabled)
      {
        enabled.setSelected(true);
      }
      checkIfModified();
    }

    private void checkIfModified()
    {
      boolean notmod = (initEnabled == enabled.isSelected());
      if (enabled.isSelected())
      {
        if (initVal != null)
        {
          notmod &= initVal.equals(val.getSelectedItem());
        }
        else
        {
          // compare against default service setting
          notmod &= option.getDefaultValue() == null
                  || option.getDefaultValue().equals(val.getSelectedItem());
        }
      }
      else
      {
        notmod &= initVal == null;
      }
      argSetModified(this, !notmod);
    }

    public void mouseClicked(MouseEvent e)
    {
      if (javax.swing.SwingUtilities.isRightMouseButton(e))
      {
        showUrlPopUp(this, finfo.toString(), e.getX(), e.getY());
      }
    }

    public void mousePressed(MouseEvent e)
    {
      // TODO Auto-generated method stub

    }

    public void mouseReleased(MouseEvent e)
    {
      // TODO Auto-generated method stub

    }

    public void mouseEntered(MouseEvent e)
    {
      // TODO Auto-generated method stub

    }

    public void mouseExited(MouseEvent e)
    {
      // TODO Auto-generated method stub

    }

  }

  private OptionBox addOption(OptionI opt)
  {
    OptionBox cb = optSet.get(opt.getName());
    if (cb == null)
    {
      cb = new OptionBox(opt);
      optSet.put(opt.getName(), cb);
      jobOptions.add(cb);
    }
    return cb;
  }

  public static void showUrlPopUp(JComponent invoker, final String finfo,
          int x, int y)
  {

    JPopupMenu mnu = new JPopupMenu();
    JMenuItem mitem = new JMenuItem("View " + finfo);
    mitem.addActionListener(new ActionListener()
    {

      @Override
      public void actionPerformed(ActionEvent e)
      {
        Desktop.showUrl(finfo);

      }
    });
    mnu.add(mitem);
    mnu.show(invoker, x, y);
  }

  protected void refreshParamLayout()
  {
    FlowLayout fl = new FlowLayout();
    paramList.setLayout(fl);
    int s = 2 * fl.getVgap();
    for (ParamBox pbox : paramSet.values())
    {
      s += fl.getVgap() + pbox.getBoxHeight(); // getBoxHeight();
    }
    paramList.setPreferredSize(new Dimension(PARAM_WIDTH, s));
    validate();
  }

  /**
   * testing method - grab a service and parameter set and show the window
   * 
   * @param args
   */
  public static void main(String[] args)
  {
    jalview.ws.jws2.Jws2Discoverer disc = jalview.ws.jws2.Jws2Discoverer
            .getDiscoverer();
    int p = 0;
    if (args.length > 3)
    {
      Vector<String> services = new Vector<String>();
      services.addElement(args[p++]);
      Jws2Discoverer.setServiceUrls(services);
    }
    try
    {
      disc.run();
    } catch (Exception e)
    {
      System.err.println("Aborting. Problem discovering services.");
      e.printStackTrace();
      return;
    }
    Jws2Discoverer.Jws2Instance lastserv = null;
    for (Jws2Discoverer.Jws2Instance service : disc.getServices())
    {
      lastserv = service;
      if (p >= args.length || service.serviceType.equalsIgnoreCase(args[p]))
      {
        if (lastserv != null)
        {
          List<Preset> prl = null;
          Preset pr = null;
          if (++p < args.length)
          {
            PresetManager prman = lastserv.getPresets();
            if (prman != null)
            {
              pr = prman.getPresetByName(args[p]);
              if (pr == null)
              {
                // just grab the last preset.
                prl = prman.getPresets();
              }
            }
          }
          else
          {
            PresetManager prman = lastserv.getPresets();
            if (prman != null)
            {
              prl = prman.getPresets();
            }
          }
          Iterator<Preset> en = (prl == null) ? null : prl.iterator();
          while (en != null && en.hasNext())
          {
            if (en != null)
            {
              if (!en.hasNext())
              {
                en = prl.iterator();
              }
              pr = en.next();
            }
            {
              System.out.println("Testing opts dupes for "
                      + lastserv.getUri() + " : "
                      + lastserv.getActionText() + ":" + pr.getName());
              List<Option> rg = lastserv.getRunnerConfig().getOptions();
              for (Option o : rg)
              {
                try
                {
                  Option cpy = jalview.ws.jws2.ParameterUtils.copyOption(o);
                } catch (Exception e)
                {
                  System.err.println("Failed to copy " + o.getName());
                  e.printStackTrace();
                } catch (Error e)
                {
                  System.err.println("Failed to copy " + o.getName());
                  e.printStackTrace();
                }
              }
            }
            {
              System.out.println("Testing param dupes:");
              List<Parameter> rg = lastserv.getRunnerConfig()
                      .getParameters();
              for (Parameter o : rg)
              {
                try
                {
                  Parameter cpy = jalview.ws.jws2.ParameterUtils
                          .copyParameter(o);
                } catch (Exception e)
                {
                  System.err.println("Failed to copy " + o.getName());
                  e.printStackTrace();
                } catch (Error e)
                {
                  System.err.println("Failed to copy " + o.getName());
                  e.printStackTrace();
                }
              }
            }
            {
              System.out.println("Testing param write:");
              List<String> writeparam = null, readparam = null;
              try
              {
                writeparam = jalview.ws.jws2.ParameterUtils
                        .writeParameterSet(
                                pr.getArguments(lastserv.getRunnerConfig()),
                                " ");
                System.out.println("Testing param read :");
                List<Option> pset = jalview.ws.jws2.ParameterUtils
                        .processParameters(writeparam,
                                lastserv.getRunnerConfig(), " ");
                readparam = jalview.ws.jws2.ParameterUtils
                        .writeParameterSet(pset, " ");
                Iterator<String> o = pr.getOptions().iterator(), s = writeparam
                        .iterator(), t = readparam.iterator();
                boolean failed = false;
                while (s.hasNext() && t.hasNext())
                {
                  String on = o.next(), sn = s.next(), st = t.next();
                  if (!sn.equals(st))
                  {
                    System.out.println("Original was " + on
                            + " Phase 1 wrote " + sn + "\tPhase 2 wrote "
                            + st);
                    failed = true;
                  }
                }
                if (failed)
                {
                  System.out.println("Original parameters:\n"
                          + pr.getOptions());
                  System.out.println("Wrote parameters in first set:\n"
                          + writeparam);
                  System.out.println("Wrote parameters in second set:\n"
                          + readparam);

                }
              } catch (Exception e)
              {
                e.printStackTrace();
              }
            }
            WsJobParameters pgui = new WsJobParameters(lastserv,
                    new JabaPreset(lastserv, pr));
            JFrame jf = new JFrame("Parameters for "
                    + lastserv.getActionText());
            JPanel cont = new JPanel();
            // jf.setPreferredSize(new Dimension(600, 800));
            cont.add(pgui);
            jf.add(cont);
            final Thread thr = Thread.currentThread();
            jf.addWindowListener(new WindowListener()
            {

              public void windowActivated(WindowEvent e)
              {
                // TODO Auto-generated method stub

              }

              public void windowClosed(WindowEvent e)
              {
              }

              public void windowClosing(WindowEvent e)
              {
                thr.interrupt();

              }

              public void windowDeactivated(WindowEvent e)
              {
                // TODO Auto-generated method stub

              }

              public void windowDeiconified(WindowEvent e)
              {
                // TODO Auto-generated method stub

              }

              public void windowIconified(WindowEvent e)
              {
                // TODO Auto-generated method stub

              }

              public void windowOpened(WindowEvent e)
              {
                // TODO Auto-generated method stub

              }

            });
            jf.setVisible(true);
            boolean inter = false;
            while (!inter)
            {
              try
              {
                Thread.sleep(10000);
              } catch (Exception e)
              {
                inter = true;
              }
              ;
            }
            jf.dispose();
          }
        }
      }
    }
  }

  public List<ArgumentI> getJobParams()
  {
    List<ArgumentI> argSet = new ArrayList<ArgumentI>();
    // recover options and parameters from GUI
    for (OptionBox opts : optSet.values())
    {
      OptionI opt = opts.getOptionIfEnabled();
      if (opt != null)
      {
        argSet.add(opt);
      }
    }
    for (ParamBox parambox : paramSet.values())
    {
      ParameterI parm = parambox.getParameter();
      if (parm != null)
      {
        argSet.add(parm);
      }
    }

    return argSet;
  }

  String lastParmSet = null;

  /*
   * Hashtable<String, Object[]> editedParams = new Hashtable<String,
   * Object[]>();
   * 
   * store the given parameters in the user parameter set database.
   * 
   * @param storeSetName - lastParmSet
   * 
   * @param descr - setDescr.getText()
   * 
   * @param jobParams - getJobParams()
   * 
   * private void _storeUserPreset(String storeSetName, String descr,
   * List<ArgumentI> jobParams) { // this is a simple hash store. Object[] pset;
   * editedParams.put(storeSetName, pset = new Object[3]); pset[0] =
   * storeSetName; pset[1] = descr; pset[2] = jobParams; // writeParam("Saving "
   * + storeSetName + ": ", jobParams); }
   * 
   * private void writeParam(String nm, List<ArgumentI> params) { for (ArgumentI
   * p : params) { System.out.println(nm + ":" + System.identityHashCode(p) +
   * " Name: " + p.getName() + " Value: " + p.getDefaultValue()); } }
   * 
   * private Object[] _getUserPreset(String setName) { Object[] pset =
   * editedParams.get(setName); // if (pset != null) // writeParam("Retrieving "
   * + setName + ": ", (List<Argument>) pset[2]); return pset; }
   * 
   * * remove the given user preset from the preset stash
   * 
   * @param setName
   * 
   * private void _deleteUserPreset(String setName) {
   * editedParams.remove(setName); }
   */

  private void syncSetNamesWithStore()
  {
    int n = 0;
    // remove any set names in the drop down menu that aren't either a reserved
    // setting, or a user defined or service preset.
    Vector items = new Vector();
    while (n < setName.getItemCount())
    {
      String item = (String) setName.getItemAt(n);
      if (!item.equals(SVC_DEF) && !paramStore.presetExists(item))
      {
        setName.removeItemAt(n);
      }
      else
      {
        items.addElement(item);
        n++;
      }
    }
    if (!items.contains(SVC_DEF))
    {
      setName.addItem(SVC_DEF);
    }
    for (WsParamSetI upn : paramStore.getPresets())
    {
      if (!items.contains(upn.getName()))
      {
        setName.addItem(upn.getName());
      }
    }
  }

  /**
   * true if lastParmSet is a user preset
   */
  boolean isUserPreset = false;

  private void reInitDialog(String nextPreset)
  {
    settingDialog = true;
    // updateTable(null,null); // first reset to defaults
    WsParamSetI pset = null;
    if (nextPreset != null && nextPreset.length() > 0)
    {
      pset = paramStore.getPreset(nextPreset);
    }
    if (pset != null)
    {
      if (pset.isModifiable())
      {
        isUserPreset = true;
        setDescr.setText(pset.getDescription());
        updateTable(null, pset.getArguments());
        lastParmSet = nextPreset;
      }
      else
      {
        isUserPreset = false;
        setDescr.setText("");
        // must be a default preset from service
        updateTable(pset, null);
        lastParmSet = nextPreset;
      }
    }
    else
    {
      isUserPreset = false;
      // Service defaults
      setDescr.setText("");
      updateTable(null, null);
      lastParmSet = SVC_DEF;
    }

    initArgSetModified();
    syncSetNamesWithStore();
    setName.setSelectedItem(lastParmSet);
    validate();
    settingDialog = false;

  }

  String curSetName = null;

  public void itemStateChanged(ItemEvent e)
  {
    if (e.getSource() == setName && e.getStateChange() == e.SELECTED)
    {
      final String setname = (String) setName.getSelectedItem();
      System.out.println("Item state changed for " + setname
              + " (handling ? " + !settingDialog + ")");
      if (settingDialog)
      {
        // ignore event
        return;
      }
      if (setname == null)
      {
        return;
      }
      javax.swing.SwingUtilities.invokeLater(new Runnable()
      {
        public void run()
        {
          doPreferenceComboStateChange(setname);
        }
      });
    }
  }

  private void doPreferenceComboStateChange(String setname)
  {
    // user has selected a different item from combo-box
    if (isModified())
    {
      String lsetname = (curSetName != null) ? curSetName : lastParmSet;
      if (lsetname.equals(setname))
      {
        // setname was just edited - so ignore this event.
        return;
      }
      settingDialog = true;
      System.out.println("Prompting to save " + lsetname);
      if (javax.swing.JOptionPane
              .showConfirmDialog(
                      this,
                      "Parameter set '"
                              + lsetname
                              + "' is modifed, and your changes will be lost.\nReally change preset ?",
                      "Warning: Unsaved Changes",
                      javax.swing.JOptionPane.OK_CANCEL_OPTION) != JOptionPane.OK_OPTION)
      {
        // revert the combobox to the current item
        settingDialog = true;
        setName.setSelectedItem(lsetname);
        settingDialog = false;
        // and leave.
        return;
        // System.out.println("Saving for " + lsetname);
        // _storeCurrentPreset(lsetname);

      }
    }
    settingDialog = true;
    reInitDialog(setname);
    settingDialog = false;

  }

  private void _renameExistingPreset(String oldName, String curSetName2)
  {
    paramStore.updatePreset(oldName, curSetName2, setDescr.getText(),
            getJobParams());
  }

  /**
   * store current settings as given name. You should then reset gui.
   * 
   * @param curSetName2
   */
  private void _storeCurrentPreset(String curSetName2)
  {
    paramStore.storePreset(curSetName2, setDescr.getText(), getJobParams());
  }

  private void _updatePreset(String lastParmSet2, String curname)
  {
    paramStore.updatePreset(lastParmSet2, curname, setDescr.getText(),
            getJobParams());

  }

  /**
   * last saved name for this user preset
   */
  String lastSetName = null;

  /**
   * last saved value of the description text for this user preset
   */
  String lastDescrText = null;

  public void actionPerformed(ActionEvent e)
  {
    if (e.getSource() instanceof Component)
    {
      Component src = (Component) e.getSource();
      if (src.getParent() == setName)
      {
        // rename any existing records we know about for this set.
        String newname = (String) e.getActionCommand().trim();
        String msg = null;
        if (isServicePreset(newname))
        {
          final String oldname = curSetName != null ? curSetName
                  : lastParmSet;
          final Component ourframe = this;
          settingDialog = true;
          setName.getEditor().setItem(oldname);
          settingDialog = false;
          javax.swing.SwingUtilities.invokeLater(new Runnable()
          {
            public void run()
            {
              JOptionPane.showMessageDialog(ourframe,
                      "Invalid name - preset already exists.",
                      "Invalid name", JOptionPane.WARNING_MESSAGE);
            }
          });

          return;
        }
        curSetName = newname;
        System.err.println("New name for user setting " + curSetName
                + " (was " + setName.getSelectedItem() + ")");
        if (curSetName.equals(setName.getSelectedItem()))
        {
          curSetName = null;
        }
        if (curSetName != null)
        {
          argSetModified(setName, true);
          return;
        }

      }
    }
  }

  private void checkDescrModified()
  {
    if (!settingDialog)
    {

      argSetModified(
              setDescr,
              (lastDescrText == null ? setDescr.getText().trim().length() > 0
                      : !setDescr.getText().equals(lastDescrText)));

    }
  }

  public void insertUpdate(DocumentEvent e)
  {
    checkDescrModified();
  }

  public void removeUpdate(DocumentEvent e)
  {
    checkDescrModified();
  }

  public void changedUpdate(DocumentEvent e)
  {
    checkDescrModified();
  }

  /**
   * 
   * @return null or the service preset selected by the user
   */
  public WsParamSetI getPreset()
  {
    if (isUserPreset || isModified()
            || (lastParmSet != null && lastParmSet.equals(SVC_DEF)))
    {
      return null;
    }
    else
    {
      return paramStore.getPreset(lastParmSet);
    }
  }
}
