#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "array.h"
#include "gjutil.h"

/****************************************************************************

   ALPS - Main routines for ALSCRIPT alignment formatting tool

   Copyright:  University of Oxford (1992,1993)

TERMS OF USE:

The computer software and associated documentation called ALSCRIPT hereinafter
referred to as the WORK is more particularly identified and described in 
Appendix A.

The WORK was written and developed by: Geoffrey J. Barton

Laboratory of Molecular Biophysics
University of Oxford
Rex Richards Building
South Parks Road
Oxford OX1 3QU
U.K.

Tel:  (+44) 865-275368
Fax:  (+44) 865-510454

Internet: gjb@bioch.ox.ac.uk
Janet:    gjb@uk.ac.ox.bioch

The WORK is Copyright (1993) University of Oxford

Administrative Offices
Wellington Square
Oxford OX1 2JD
U.K.

CONDITIONS:

The WORK is made available for educational and non-commercial research 
purposes.

For commercial use, a commercial licence is required - contact the author
at the above address for details.

The WORK may be modified, however this text, all copyright notices and
the authors' address must be left unchanged on every copy, and all
changes must be documented after this notice.  A copy of the
modified WORK must be supplied to the author.

All use of the WORK must cite:  Barton, G. J. (1993), ALSCRIPT: A Tool to
Format Multiple Sequence Alignments, Protein Engineering, Volume 6, No. 1, 
pp. 37-40.

APPENDIX A:

The program package known as ALSCRIPT is made up of the following files:

README     	This copyright notice
EXAMPLE.BLC     Example block file
EXAMPLE.COM     Example command file
EXAMPLE1.BLC    Example block file
EXAMPLE1.COM    Example command file
EXAMPLE2.COM    Example command file
EXAMPLE3.COM    Example command file
ALSCRIPT.EXE    ALSCRIPT executable program
MSF2BLC.EXE     GCG MSF to BLOCK file conversion program
CLUS2BLC.EXE    CLUSTAL PIR file to BLOCK file conversion program
EXAMPLE1.PS     Example output of program
ALSCRIPT.DOC    Documentation
MAKEFILE        Makefile for programs
MAKEFILE.GCC    Makefile for GCC compiler
MAKEFILE.SGI    Makefile for Silicon Graphics 
MAKEFILE.WAT    Makefile for WATCOM 386 C compiler
AGETBLOC.C      Source code for block file reading routine
ALPS.C          Source code for main ALSCRIPT routines
ARRAY.H         Header file
CLUS2BLC.C      Source code for CLUSTAL PIR file to BLOCK file conversion program
DEFAULTS.H      Header file
GJUTIL.C        Source code for utility routines
GJUTIL.H        pHeader file for utility routines
MSF2BLC.C       Source code for GCG MSF to BLOCK file conversion program
MAKEFILE.SUN    Makefile for Sun acc compiler.
ALSCRIPT.C      Source code for ALSCRIPT main program

****************************************************************************

   Main routine for ALSCRIPT - formally called ALPS

   alps:  11th June 1992.
   Author: G. J. Barton

   This file contains a set of subroutines that provide for flexible
formatting of multiple sequence alignments in PostScript at
publication quality.

11-18th June 1992:

1.  Modified getbloc.c to Agetbloc.c.  Agetbloc reads all characters
(ie not just legal amino acid codes) from an AMPS blocfile format
multiple alignment file.

2.  Wrote basic page scaling features to allow arbitrary page
dimensions to be defined and to allow all x,y locations to be defined
in terms of characters of the chosen pointsize.  The number of
residues per line and hence number of pages is determined by the
pointsize.  The page is divided into four areas.  (a) The main region
for plotting out the amino acid sequences. (b) A region to the left of
a for plotting sequence identifier codes - the width of this region is
set by variable Cidwidth.  (c) A region above region a, and (d) a
region below a.  Regions c and d are initially set to zero size.
Routine ALReserveSpace() allows a selected number of lines to be
reserved above or below for plotting.

3.  Routines ALOutID and ALOutSeq written and multi-page plotting of
the sequences sorted out.  Still limited to one page in the vertical
direction - toyed with the idea of writing single page PostScript and
using PostScript to modify clipping boundaries like in the Poster demo
program in the CookBook.

4.  Experiment with simple boxing facility - ALDrawSimpleBox().  This
routine is incompatible with multi-page output since page breaks are
unpredictable.

5.  Develop sophisticated (!) boxing facility - defines boxes in terms
of TOP BOTTOM LEFT and RIGHT lines that may be defined around each
character routine ALLineRes() implements this feature.  The details of
TOP etc for each character are stored in the unsigned char array
"lines".  This uses the rightmost four bits to encode LEFT RIGHT etc.
Thus a line to the LEFT line is coded by decimal 1 (0001 binary), a
full box by decimal 15 (1111 binary).  ALBoxRes and ALBoxRegion
written to allow boxing of arbitrary regions.  

6.  The y coordinate sent to the low level ALLineRes routine 
refers directly to the index to the lines array.  If you want to work in 
residue coordinate space, you need to flip the y coordinate before 
writing to the lines array (also the fill array).  Routines ALSurroundChars
and ALFillChars do this directly.  Routines ALBoxRegion and ALFilRegion
make use of the Yindx array which does the flip in a much cleaner and more
elegant way.  If I'd thought of it at the
time, I would have used this in ALSurroundChars and ALFillChars as well!

7.  Now add capability to split alignments vertically as well as horizontally.
There is now no limit on pointsize or number/length of sequences 
(in principle!).  Next move on the page layout front will be to allow 
multiple blocks on a single page - this would permit alignments of a 
few sequences, or small fonts to make better use of paper!

8.  We now have most of the subroutines working OK.  So time to tidy up the 
input.  Add simple command parser to main programme - this is in two steps, 
step 1 reads the basic page layout commands, step 2 reads the boxing/shading
font changing commands - Oh yes, added the font changing option as well.

Current commands as follows:
Commands are delimited by space, comma, or tab. Newline signals the end of a 
command.  Unless otherwise stated, units are in characters of the default pointsize.
For a full list see the file "alscript.doc".

step 1 commands:

BLOCK_FILE <input filename>           %The name of the alignment file to format
OUTPUT_FILE <file>                    %must have this near the beginning
LANDSCAPE                             %Longest side horizontal
PORTRAIT                              %             vertical
POINTSIZE  <float>                    %Pointsize for page calculations and character spacing/default size.
IDENT_WIDTH <integer>                 %Width allocated to identifiers.
NUMBER_SEQS  1 || 0                   %1=number identifiers, 0= no numbers (default).
DEFINE_FONT <font number integer> <font name> <pointsize>  %define a font to be referred to as font number.
e.g.
  DEFINE_FONT 0 Helvetica 10          %font 0 is Helvetica at 10 point
  DEFINE_FONT 4 Times-Roman DEFAULT   %font 4 is Times-Roman at the default pointsize.
SETUP                                 %signals end of step 1 commands.

step 2 commands:

RESERVE_SPACE_TOP <integer>           %make space at the top of the alignment 
RESERVE_SPACE_BOTTOM <integer>        %                  bottom  DOES NOT WORK.
SURROUND_CHARS <chars> <start_res> <start_seq> <end_res> <end_seq> 
                                      %surround (box) the characters chars within the ranges specified 
e.g.
  SURROUND_CHARS PG 1 7 20 12         %surround P and G characters from residues 1 to 20 of sequences 7 to 12.
  SURROUND_CHARS DEKHR ALL            %surround D and E and K and R characters for the whole alignment.

SHADE_CHARS <chars> <start_res> <start_seq> <end_res> <end_seq> <grey>
                                      %similar syntax to SURROUND_CHARS with the addition of a grey value
                                      % 0 = black, 1 = white.  Note that shading is done before outputing the
                                      % alignment.
e.g.
  SHADE_CHARS PG 1 7 20 12 0.5        % shade P and G in the range shown with a grey value of 0.5
  SHADE_CHARS ILV ALL 0               % shade all  I and L and V in black.

FONT_CHARS <chars> <start_res> <start_seq> <end_res> <end_seq> <fontnumber>
                                      %same syntax as SHADE_CHARS, only give the font number in place of grey.
                                      %font numbers must first be defined with the DEFINE_FONT command above.
FONT_RESIDUE sx sy font               %allows setting of font for one residue (use with TEXT command)

BOX_REGION <start_res> <start_seq> <end_res> <end_seq>
                                      %draw a rectangular box around the characters within the defined range.
SHADE_REGION <start_res> <start_seq> <end_res> <end_seq> <grey>
                                      %shade the region by the chosed grey value.

9.  Add option to place arbitrary text string at a character location.
e.g
  TEXT 1 7 "SIGNAL PEPTIDE"           %puts this text at sequence position 1 7.
                                      %to change fonts use the FONT_RESIDUE or FONT_REGION commands.
10. Add option to change fonts of identifiers.
ID_FONT <sequence> <font>
e.g. 
  ID_FONT 3 1                         %sets font to 1 for sequence identifier 3
  ID_FONT ALL 7                       %uses font 7 for all identifiers

19th June 1992:
Still fiddling - must get back to some real work soon.
Split alps away from the scanps program.  Now have a separate main() for the routine.

Add options to redefine all defaults:  e.g. MAXslen (maximum sequence
length etc.) Add arbitrary line drawing options:  eg. LINE TOP 20 15 50
draws a line along the top of the characters from residue 20 to 50 or
sequence 15. Think about adding the option to change line thickness at
different parts of the plot. Decide not to implement that option right
now. Add option to include dummy sequences (ADD_SEQ)

Check the programme compiles on the 386 PC.  It does!
Write documentation and brief report of the programme.

22nd June 1992:  Get back to some real work!

19th October:  Add SILENT_MODE Command and carriage returns to all .PS print statements.
21st October:  Add VERTICAL_SPACING command and ability to output multiple blocks per page.
15th November: Add pipe options to alscript.c  Tidy up file handling.

6th December:  Add Colour options.
18th Jan 1993: Tidy up citations:  modify Inverse chars to allow reversion.
1 March 1993:  Fix bug in colour option.  Pass use_colour to ALDrawLines routine to 
               specify black for line colour.  Previously, this was drawn in white
               if coloured shading had been specified on the same characters.
2 March 1993:  Fix bug in colour option to output identifiers in black, not white!
9 March 1993:  Add NO_NUMBERS option to avoid printing residue numbers.
24 March 1993: Add mask features.
6 Sep   1993:  Add FIX_INVERSE option-this prevents INVERSE from toggling.


*/

#define TOKENS "\" ,\n\t"        /* standard command token delimiters */
#define POINT 72               /* Work in points throughout */
#define C_ID_WIDTH 10          /* Default width for writing id's (in characters) */
#define DEF_POINTSIZE 10.0      
#define X_OFFSET 40            /* offset for bottom lh corner of page (points) */
#define Y_OFFSET 30
#define DEF_YPOS 0             /* default first printinf position (character units) */
#define LINE_WIDTH_FACTOR 0.05 /* Multiply pointsize by this to get linewidth */
#define MAX_ADD_SEQ 10	       /* Maximum number of ADD_SEQ commands allowed */

/*#define MAX_S 10.5             max and min page side in inches */
/*#define MIN_S 6.5  */
/*#define MAXSIDE (MAX_S * POINT)   Default Max and min dimensions of paper */
/*#define MINSIDE (MIN_S * POINT) */


int width = 0;                 /* max width and height of drawing area */
int height = 0;
char *tokens = TOKENS;

float MAXside = 10.5 * POINT;     /* values to hold Max and min page side in points */
float MINside = 6.5 * POINT;

int xoff = X_OFFSET;
int yoff = Y_OFFSET;

int Cidwidth = C_ID_WIDTH;     /* width of id region in characters (left of sequence block)*/
int Cseqwidth;                 /* width of sequence region in characters */
int Cwidth;                    /* width/height of page in characters */
int Cheight;
int Xspace;                    /* spacing between characters in X */
int Yspace;                    /* spacing between characters in Y */
float XspaceFactor = 0.2;          /* extra bit to add to horizontal character spacing */
float YspaceFactor = 0.0;          /* extra bit to add to vertical character spacing */
float XshiftFactor = 0.12;          /* *pointsize gives the shift for vertical lines */
float YshiftFactor = 0.12;          /*                                horizontal lines */

int CseqTop = 0;
int CseqVoffset = 0;           /* vertical offset for sequence block (characters)*/
int CtopSpace =0;              /* extra space above sequence block (characters) */
int CbottomSpace = 0;          /*   "     "   below    "       "         "      */
int Csheet = 1;                /* number of vertical splits in sequence block */

int TVspace = 0;               /* total vertical space reserved (characters)*/

float Xshift = 0;     /* offsets for drawing box lines */
float Yshift = 0;
float LineWidthFactor = LINE_WIDTH_FACTOR;

float pointsize = DEF_POINTSIZE;
int npage;

int Vspacing = 0;               /* vertical spacing in characters between blocks on a page */


enum Orientations{
  LANDSCAPE,
  PORTRAIT
};

enum Spaces{
  TOP,
  BOTTOM
};

struct addseq {
	int pos;
	int num;
};


/* bit masks for the lines array */
unsigned char LINE_LEFT = 1; /* mask for line at left of char */
unsigned char LINE_RIGHT= 2; /*                  right        */
unsigned char LINE_TOP  = 4; /*                  top          */
unsigned char LINE_BOTTOM=8; /*                  bottom       */
 
void ALOutId(struct seqdat *,int,unsigned char *,int,int,int,int, FILE *);
void ALOutSeq(struct seqdat *,unsigned char **,char ***,unsigned char **,int ,unsigned char **,int ,int ,int ,int ,FILE *);
void PSPutText(int,int,char *,FILE *);
void PSPutChar(int,int,char,FILE *);
void PSShowPage(FILE *);
void ALSetFont(char *,int,FILE *);
void ALCheckSinglePage(int ,int ,int );
void ALSetPageLimits(void);
void PSSetOrientation(int, FILE *);
void PSPreamble(FILE *);
void PSLandscape(FILE *);
void PSPortrait(FILE *);
void PSSetupPage(int,int,FILE *);
void ALReserveSpace(int, int);
void ALReportInfo(int, FILE *);
char **ALCreateNumbers(int,int,int,int);
char *ALCreateTicks(int,int,int);
void ALOutTicks(char *,int,int,int,int,FILE *);
void ALOutNumbers(char **,int,int,int,int,FILE *);
void PSDrawBox(float,float,float,float,float,FILE *);
void ALSimpleBox(int,int,int,int,float,FILE *);
void PSline(float,float,float,float,FILE *);
void ALLineChar(unsigned char **,int,int,unsigned char);
void ALLineRes(unsigned char **,int,int,unsigned char);
void ALDrawLines(unsigned char **,int,int,int,int,int,int,int,FILE *);
void ALBoxRes(unsigned char **,int,int);
void ALDrawFills(signed char **,int ,unsigned char **, int,int ,int,int ,int ,int, FILE *);
void ALFilRes(signed char **,int,int,float);
void ALBoxRegion(unsigned char **,int,int,int,int,int *);
void ALFilRegion(signed char **,int ,int,int,int , int *,float);
void ALSurroundChars(struct seqdat *,unsigned char **,int,int,int,int,int,char *);
void ALShadeChars(struct seqdat *,signed char **,int ,int,int,int,int,char *,float);
void ALDefineFont(int,char *,float,FILE *);
void echo(char *);
void ALGetFourInt(int *,int *,int *,int *);
void ALGetThreeInt(int *,int *,int *);
void ALGetTwoInt(int *,int *);
void ALFontChars(struct seqdat *,unsigned char **,int,int,int,int,char *,unsigned char);
void ALFontRegion(unsigned char **,int,int,int,int,unsigned char);
void PSSetFont(unsigned char,FILE *);
void ALDoLine(unsigned char **,int,int,int,int,int *);
void ALSubChars(struct seqdat *,int,int,int,int,char,char);
char ALChekSpace(const char *);
void PSSetGrey(float,FILE *);
void ALInverseChars(struct seqdat *,unsigned char **,int,int,int,int,char *);
void ALFInverseChars(struct seqdat *,unsigned char **,int,int,int,int,char *);
int Agetbloc(FILE *,struct seqdat *,int *);
void ALDefineColour(int,float,float,float,FILE *);
void ALSColChars(struct seqdat *,unsigned char **,int,int,int,int,int,char *,unsigned char);
void ALColourChars(struct seqdat *,unsigned char **,int ,int,int,int,char *,unsigned char);
void ALColRes(unsigned char **,int ,int , int );
void PSSetColour(unsigned char,FILE *);
void ALColRegion(unsigned char **,int,int,int,int,int *,unsigned char);
void ckdd(struct seqdat *, int);
int Ifound(struct seqdat *,int,int,char *,int);
void ALSColMask(unsigned char **,unsigned char **,int, int, int, int,int, unsigned char);
void ALInverseMask(unsigned char **,unsigned char **,int, int, int, int);
void ALFontMask(unsigned char **,unsigned char **,int, int, int, int,unsigned char);
void ALShadeMask(unsigned char **,signed char **,int,int,int,int,int *, float);
void ALGetAllRange(int *, int *, int *, int *,int ,int );
void ALid_mask(unsigned char **,struct seqdat *,int, int, int, int, int, char *, char *);
void ALfre_mask(unsigned char **mask,struct seqdat *,int, int, int, int, char *, char *);
int Ifound(struct seqdat *,int,int,char *,int);
void ALagree_mask(unsigned char **,struct seqdat *,int,int,int,int,int);
void ALnot_mask(unsigned char **,int, int, int, int);
void ALsub_mask(unsigned char **,struct seqdat *,int, int, int, int, char);
void ALMask(unsigned char **,int, int, int, int);
void ALColourMask(unsigned char **,unsigned char **,int,int,int,int,unsigned char);
void ALSurroundMask(unsigned char **,struct seqdat *,unsigned char **,int,int,int,int,int);
int save_pir(struct seqdat *,int,FILE *);


int alps(char *command_file,int silent, int pipe)
{  
  FILE *cf,*inf,*outf,*pirf;
  char *buff;
  char *token;
  char *charstring;
  struct seqdat *bloc;
  struct seqdat *temp;
  int orientation;
  extern int MAXtlen,MAXnbloc;
  extern int MAXilen,precis,MAXslen;
  int nseq;
  int start_aa;
  int start_seq;
  int i,j,k;
  int totalpage;
  char **numbers;
  char *ticks;
  unsigned char **lines;
#ifdef VAX
 char **fills;
#else
  signed char **fills;
#endif
  unsigned char **fonts;
  unsigned char *idfonts;
  unsigned char **inverse;
  char ***texts;
  int *Yindx;
  int number_seqs = 0;

  int Xtemp,Ytemp;
  int sx,sy,ex,ey;

  int fnum;
  char *fname;
  float fpoint;
  int rowsiz;

  char oldchar,newchar;
  
  struct seqdat *addbloc;
  struct addseq *adds;
  int nadds;
  int natot;
  int l;
  int do_ticks = 0;
  int number_int = 10;
  int pirsave = 0;

  int nbp,nspage;  /*counter and total for number of blocks per page */
  int lastY;
  int nbpage;

  /* colours */
  float red,green,blue;
  unsigned char **s_colour;       /*colour number for shading */
  unsigned char **c_colour;       /*colour number for characters */
  int use_colour;                 /* flag set if the define_colour command is seen */
  int res_numbers;

  int msl;

  /* masking variables */
  unsigned char **mask;
  char *illegal;
  char *legal;
  int id_cut;

  /* odds */
  int fix_inverse;  /* if set to 1 then inverse_chars is one-way */
  
  
  /* file streams */
  extern FILE *std_in, *std_out, *std_err;

  /* By default read and write to stdin and stdout */
  outf = std_out;
  inf = std_in;

  /* default is no colour */
  use_colour = 0;

  /* if set, residue numbers are output */
  res_numbers = 1;

  /* initialise the mask */
  mask = NULL;
  illegal = NULL;
  legal = NULL;

  fix_inverse = 0;

  adds = (struct addseq *) malloc(sizeof(struct addseq) * (MAX_ADD_SEQ+1));
  nadds = 0;
  

  fname = (char *) malloc(sizeof(char)*MAXtlen);

  buff = (char *) malloc(MAXtlen * sizeof(char));
  if(buff == NULL)error("No Space for buff",1);

  if((cf = fopen(command_file,"r")) == NULL){
	error("Cannot open ALSCRIPT command file",0);
	return 0;
  }
  
  /* 
    This is a little inelegant at the moment - when I get the generalized 
    parser finished this could be cleaned up substantially
  */

  /* main loop to read commands */
  if(pipe){
    /* Alscript is being used as a pipe so write preamble to std_out */
    outf = std_out;
    PSPreamble(outf);
  }
    
  if(!silent)fprintf(std_err,"Starting ALSCRIPT\n");

  while(buff = fgets(buff,MAXtlen,cf)){
    if(!silent)echo(buff);
    if(buff == NULL) break;
    token = strtok(buff,TOKENS);
    if(token == NULL) break;
    token = GJstoup(token);    
    if(*token == '#'){
      if(!silent)echo("Comment\n");
    }else if(strcmp(token,"SILENT_MODE")==0){
        if(!silent){
	  silent=1;
	}else{
	  silent=0;
	}
    }else if(strcmp(token,"MAX_INPUT_LEN")==0){
      if(!silent) fprintf(std_err,"Max input len was:\t%d\n",MAXtlen);
      token = strtok(NULL,TOKENS);
      MAXtlen = atoi(token);
      if(!silent)fprintf(std_err,"Max input len now:\t%d\n",MAXtlen);
    }else if(strcmp(token,"MAX_NSEQ")==0){
      if(!silent)fprintf(std_err,"Max No. of sequences was:\t%d\n",MAXnbloc);
      token = strtok(NULL,TOKENS);
      MAXnbloc = atoi(token);
      MAXtlen = MAXnbloc;
      fprintf(std_err,"Max No. of sequences now:\t%d\n",MAXnbloc);
    }else if(strcmp(token,"MAX_ILEN")==0){
      if(!silent)fprintf(std_err,"Max Identifier length was:\t%d\n",MAXilen);
      token = strtok(NULL,TOKENS);
      MAXilen = atoi(token);
      if(!silent)fprintf(std_err,"Max Identifier length now:\t%d\n",MAXilen);
    }else if(strcmp(token,"PRECISION")==0){
      token = strtok(NULL,TOKENS);
      precis = atoi(token);
    }else if(strcmp(token,"MAX_SEQ_LEN")==0){
      if(!silent)fprintf(std_err,"Max Sequence Length was:\t%d\n",MAXslen);
      token = strtok(NULL,TOKENS);
      MAXslen = atoi(token);
      if(!silent)fprintf(std_err,"Max Sequence Length now:\t%d\n",MAXslen);
    }else if(strcmp(token,"X_OFFSET")==0){
      if(!silent)fprintf(std_err,"X offset was:\t%d\n",xoff);
      token = strtok(NULL,TOKENS);
      xoff = atoi(token);
      if(!silent)fprintf(std_err,"X offset now:\t%d\n",xoff);
    }else if(strcmp(token,"Y_OFFSET")==0){
      if(!silent)fprintf(std_err,"Y offset was:\t%d\n",yoff);
      token = strtok(NULL,TOKENS);
      yoff = atoi(token);
      if(!silent)fprintf(std_err,"Y offset now:\t%d\n",yoff);
    }else if(strcmp(token,"MAX_SIDE")==0){
      if(!silent)fprintf(std_err,"Longest page side was:\t%.2f inches\n",MAXside/POINT);
      token = strtok(NULL,TOKENS);
      MAXside = POINT * atof(token);
      if(!silent)fprintf(std_err,"Longest page side now:\t%.2f inches\n",MAXside/POINT);
    }else if(strcmp(token,"MIN_SIDE")==0){
      if(!silent)fprintf(std_err,"Shortest page side was:\t%.2f inches\n",MINside/POINT);
      token = strtok(NULL,TOKENS);
      MINside = POINT * atof(token);
      if(!silent)fprintf(std_err,"Shortest page side now:\t%.2f inches\n",MINside/POINT);
    }else if(strcmp(token,"LINE_WIDTH_FACTOR")==0){
      if(!silent)fprintf(std_err,"Line width multiplication factor was:\t%f\n",LineWidthFactor);
      token = strtok(NULL,TOKENS);
      LineWidthFactor = atof(token);
      if(!silent)fprintf(std_err,"Line width multiplication factor now:\t%f\n",LineWidthFactor);
    }else if(strcmp(token,"BLOCK_FILE") == 0){
      token = strtok(NULL,TOKENS);
      if(!pipe){
        inf = fopen(token,"r");
        if(inf == NULL)error("Cannot open BLOCK_FILE\n",1);
      }
    }else if(strcmp(token,"OUTPUT_FILE")==0){
      token = strtok(NULL,TOKENS);
      if(!pipe){
        outf = fopen(token,"w");
        if(outf == NULL)error("Cannot open OUTPUT_FILE\n",1);
        PSPreamble(outf);
      }
    }else if(strcmp(token,"PIR_SAVE")==0){
      token = strtok(NULL,TOKENS);
      pirf = fopen(token,"w");
      if(pirf == NULL)error("Cannot open PIR_SAVE file\n",1);
      pirsave = 1;
    }else if(strcmp(token,"LANDSCAPE")==0){
      orientation = LANDSCAPE;
    }else if(strcmp(token,"PORTRAIT")==0){
      orientation = PORTRAIT;
    }else if(strcmp(token,"POINTSIZE")==0){
      token = strtok(NULL,TOKENS);
      pointsize = atof(token);
    }else if(strcmp(token,"IDENT_WIDTH")==0){
      token = strtok(NULL,TOKENS);
      Cidwidth = atoi(token);
      if(Cidwidth > MAXilen)error("INCREASE MAX_ID_LEN value",1);
    }else if(strcmp(token,"NUMBER_SEQS")==0){
      number_seqs=1;
    }else if(strcmp(token,"X_SPACE_FACTOR")==0){
      if(!silent)fprintf(std_err,"Was:\t%.2f\n",XspaceFactor);
      token = strtok(NULL,TOKENS);
      XspaceFactor = atof(token);
      if(!silent)fprintf(std_err,"Now Is:\t%.2f\n",XspaceFactor);
    }else if(strcmp(token,"Y_SPACE_FACTOR")==0){
      if(!silent)fprintf(std_err,"Was:\t%.2f\n",YspaceFactor);
      token = strtok(NULL,TOKENS);
      YspaceFactor = atof(token);
    }else if(strcmp(token,"X_SHIFT_FACTOR")==0){
      if(!silent)fprintf(std_err,"Was:\t%.2f\n",XshiftFactor);
      token = strtok(NULL,TOKENS);
      XshiftFactor = atof(token);
      if(!silent)fprintf(std_err,"Now Is:\t%.2f\n",XshiftFactor);
    }else if(strcmp(token,"Y_SHIFT_FACTOR")==0){
      if(!silent)fprintf(std_err,"Was:\t%.2f\n",YshiftFactor);
      token = strtok(NULL,TOKENS);
      YshiftFactor = atof(token);
      if(!silent)fprintf(std_err,"Now Is:\t%.2f\n",YshiftFactor);
    }else if(strcmp(token,"VERTICAL_SPACING")==0){
      if(!silent)fprintf(std_err,"Was:\t%d\n",Vspacing);
      token = strtok(NULL,TOKENS);
      Vspacing = (int) atol(token);
      if(!silent)fprintf(std_err,"Now Is:\t%d\n",Vspacing);
    }else if(strcmp(token,"DEFINE_FONT")==0){
      token = strtok(NULL,TOKENS);
      fnum = atoi(token);                      /* font number */
      token = strtok(NULL,TOKENS);           /* font name */
      fname = strcpy(fname,token);
      token = strtok(NULL,TOKENS);
      if(strcmp(token,"DEFAULT")==0){
	fpoint = pointsize;
      }else if(strcmp(token,"REL")==0){
      	token = strtok(NULL,TOKENS);
      	fpoint = pointsize * atof(token);
      }else{
	fpoint = atof(token);
      }
      ALDefineFont(fnum,fname,fpoint,outf);
    }else if(strcmp(token,"DEFINE_COLOUR")==0 || strcmp(token,"DEFINE_COLOR")==0){
      use_colour = 1;
      token = strtok(NULL,TOKENS);
      fnum = atoi(token);                      /* colour number */
      token = strtok(NULL,TOKENS);
      red = atof(token);
      token = strtok(NULL,TOKENS);
      green = atof(token);
      token = strtok(NULL,TOKENS);
      blue = atof(token);
      ALDefineColour(fnum,red,green,blue,outf);
    }else if(strcmp(token,"ADD_SEQ")==0){
      token = strtok(NULL,TOKENS);
      adds[nadds].pos = atoi(token);
      token = strtok(NULL,TOKENS);
      adds[nadds].num = atoi(token);
      ++nadds;
    }else if(strcmp(token,"DO_TICKS")==0){
      do_ticks=1;
    }else if(strcmp(token,"NUMBER_INT")==0){
      token = strtok(NULL,TOKENS);
      number_int = atoi(token);
      if(!silent)fprintf(std_err,"Sequence Numbering interval now:\t%d\n",number_int);
    }else if(strcmp(token,"NO_NUMBERS")==0){
      res_numbers = 0;
    }else if(strcmp(token,"SETUP")==0){
      if(!silent)fprintf(std_err,"Initial Commands read - Establishing basic settings\n");
      break;
    }else{
      error("Unrecognised Step 1 command",1);
      fprintf(std_err,"%s",buff);
    }
  }

  if(outf == std_out){
    if(!silent)fprintf(std_err,"Output file undefined - will write to stdout\n");
  }

  
  if(inf == std_in){
    if(!silent)fprintf(std_err,"Block file undefined - will read from stdin\n");
  }
  
  bloc = (struct seqdat *) malloc(sizeof(struct seqdat) * MAXnbloc);
  mcheck((char *) bloc,"Cannot get space for bloc");
  if(!Agetbloc(inf,bloc,&nseq))error("Cannot read bloc file",1);

  ckdd(bloc,nseq);

  if(pirsave){
    if(!save_pir(bloc,nseq,pirf))error("Cannot write to pirf file",1);
    exit(0);
  }

  if(nadds > 0){
     if(!silent)fprintf(std_err,"Making extra space for %d additions\n",nadds);
     natot = 0;
     for(i=0;i<nadds;++i){
     	natot += adds[i].num;
     }
     if(!silent)fprintf(std_err,"Total of %d extra sequences\n",natot);
     addbloc = (struct seqdat *) malloc(sizeof(struct seqdat) *(nseq+natot+1));
     i = 0;
     j = 0;
     k = 0;
     while(i < nseq+1){
     	if(k < nadds && i==adds[k].pos){
     	  /* reserve space for the new sequences and set to blank*/
     	  for(l=0;l<adds[k].num;++l){
     	     ++j;
     	     addbloc[j].id = (char *) malloc(sizeof(char) * MAXilen);
     	     addbloc[j].id = GJstrblank(addbloc[j].id,MAXilen);
     	     addbloc[j].seq = (char *) malloc(sizeof(char) * (bloc[1].slen+1));
     	     addbloc[j].seq = GJstrblank(addbloc[j].seq,bloc[1].slen+1);
	  }
	  ++k;
	}else{
	  /* just copy the pointers to the bloc sequences */
	  ++i;
	  ++j;
	  if(i<nseq+1){
	    addbloc[j].id = bloc[i].id;
	    addbloc[j].title = bloc[i].title;
	    addbloc[j].seq = bloc[i].seq;
	  }
	}
     }
     addbloc[1].slen = bloc[1].slen;
     temp = bloc;
     bloc = addbloc;
     addbloc = temp;
     nseq += natot;
     if(!silent)fprintf(std_err,"Total number of sequences now: %d\n",nseq);
  }
  msl = bloc[1].slen-2;  /* maximum sequence length defined */

  PSSetupPage(orientation,1,outf);
  ALSetPageLimits();
  CtopSpace = 2;    /* set top 2 lines for numbers */
/*  TVspace = nseq;*/
/*  ALReserveSpace(TOP,2);*/

  
  /* allocate to +1 size to allow indexing from 1...max */
  /* lines array - stored info on where to draw lines */
  lines = uchararr(bloc[1].slen+1,nseq+1);
  if(lines == NULL)error("No space for lines array\n",1);
  GJUCinit(lines,bloc[1].slen+1,nseq+1,0);

  /* array holds grey value for each character fill*/
  fills = chararr(bloc[1].slen+1,nseq+1);
  if(fills == NULL)error("No space for fills array\n",1);
  GJCinit(fills,bloc[1].slen+1,nseq+1,-1);

  /* allocate to +1 size to allow indexing from 1...max */
  /* fonts array - stores a number representing the font to use for each residue */
  /* default font is font 0 */
  fonts = uchararr(bloc[1].slen+1,nseq+1);
  if(fonts == NULL)error("No space for fonts array\n",1);
  GJUCinit(fonts,bloc[1].slen+1,nseq+1,0);

  /* inverse indicates whether the the text at i j should be printed in white */
  inverse = uchararr(bloc[1].slen+1,nseq+1);
  if(inverse == NULL)error("No space for inverse array\n",1);
  GJUCinit(inverse,bloc[1].slen+1,nseq+1,0);

  /* colour arrays - define the colour to use for shading and lettering */
  /* default colour numbers are 99 for white and 100 for black */
  s_colour = uchararr(bloc[1].slen+1,nseq+1);       /* shading colour */
  if(s_colour == NULL)error("No space for s_colour array\n",1);
  GJUCinit(s_colour,bloc[1].slen+1,nseq+1,99);
  c_colour = uchararr(bloc[1].slen+1,nseq+1);       /* character colour */
  if(c_colour == NULL)error("No space for c_colour array\n",1);
  GJUCinit(c_colour,bloc[1].slen+1,nseq+1,100);

  /* holds font information for each identifier */
  idfonts = (unsigned char *) malloc(sizeof(unsigned char) * nseq+1);
  for(i=0;i<nseq+1;++i){
    idfonts[i]= (unsigned char) 0;
  }

  /* array holds pointers to text strings at each [i][j] point*/
  texts = (char ***) malloc(sizeof(char **) * (bloc[1].slen+1));
  rowsiz = sizeof(char *) * (nseq + 1);
  for(k=0;k<bloc[1].slen+1;++k){
    texts[k] = (char **) malloc(rowsiz);
    for(i=0;i<(nseq+1);++i){
      texts[k][i]=NULL;
    }
  }
  
  Yindx = (int *) malloc(sizeof(int)*(nseq+1));
  for(i=1,j=nseq;i<nseq+1;++i,--j){
    Yindx[i] = j;
  }

  numbers = ALCreateNumbers(0,number_int,bloc[1].slen,4);
  ticks = ALCreateTicks(0,number_int,bloc[1].slen);
  CseqTop = nseq;

  while(buff = fgets(buff,MAXtlen,cf)){
    if(buff == NULL) break;
    if(!silent)echo(buff);
    token = strtok(buff,TOKENS);
    if(token == NULL) break;
    token = GJstoup(token);
    if(*token == '#'){
      if(!silent)echo("Comment\n");
    }else if(strcmp(token,"SILENT_MODE")==0){
        if(!silent){
	  silent=1;
	}else{
	  silent=0;
	}
    }else if(strcmp(token,"RESERVE_SPACE_TOP")==0){
      token = strtok(NULL,TOKENS);
      ALReserveSpace(TOP,atoi(token));
    }else if(strcmp(token,"RESERVE_SPACE_BOTTOM")==0){
      token = strtok(NULL,TOKENS);
      ALReserveSpace(BOTTOM,atoi(token));
      CseqTop = nseq + CbottomSpace;
    }else if(strcmp(token,"SURROUND_CHARS")==0){
      token = strtok(NULL,TOKENS);
      charstring = token;
      ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
      ALSurroundChars(bloc,lines,sx,sy,ex,ey,nseq,charstring);
    }else if(strcmp(token,"SHADE_CHARS")==0){
      token = strtok(NULL,TOKENS);
      charstring = token;
      ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
      token = strtok(NULL,TOKENS);
      ALShadeChars(bloc,fills,sx,sy,ex,ey,nseq,charstring,(float)atof(token));
    }else if(strcmp(token,"FONT_CHARS")==0){
      token = strtok(NULL,TOKENS);
      charstring = token;
      ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
      token = strtok(NULL,TOKENS);
      ALFontChars(bloc,fonts,sx,sy,ex,ey,charstring,atoi(token));
    }else if(strcmp(token,"CCOL_CHARS")==0){  /* colour characters */
      if(!use_colour)error("You must use the DEFINE_COLOUR command first",1);
      token = strtok(NULL,TOKENS);
      charstring = token;
      ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
      token = strtok(NULL,TOKENS);
      ALColourChars(bloc,c_colour,sx,sy,ex,ey,charstring,atoi(token));
    }else if(strcmp(token,"SCOL_CHARS")==0){  /* colour character backgrounds */
      if(!use_colour)error("You must use the DEFINE_COLOUR command first",1);
      token = strtok(NULL,TOKENS);
      charstring = token;
      ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
      token = strtok(NULL,TOKENS);
      ALSColChars(bloc,s_colour,sx,sy,ex,ey,nseq,charstring,(unsigned char) atoi(token));
    }else if(strcmp(token,"INVERSE_CHARS")==0){
      token = strtok(NULL,TOKENS);
      charstring = token;
      ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
      if(fix_inverse == 1){
        ALFInverseChars(bloc,inverse,sx,sy,ex,ey,charstring);
      }else{
        ALInverseChars(bloc,inverse,sx,sy,ex,ey,charstring);
      }
    }else if(strcmp(token,"FIX_INVERSE")==0){
      fix_inverse = 1;
    }else if(strcmp(token,"SUB_CHARS")==0){
    	token = strtok(NULL,TOKENS);
    	if(strcmp(token,"ALL")==0){
	  token = strtok(NULL,TOKENS);
	  oldchar = ALChekSpace(token);
	  token = strtok(NULL,TOKENS);
    	  newchar = ALChekSpace(token); 
    	  ALSubChars(bloc,1,1,bloc[1].slen-1,nseq,oldchar,newchar);
    	}else{
    	  sx = atoi(token);
    	  ALGetThreeInt(&sy,&ex,&ey);
  	  token = strtok(NULL,TOKENS);
    	  oldchar = ALChekSpace(token);
  	  token = strtok(NULL,TOKENS);
	  newchar = ALChekSpace(token);
    	  ALSubChars(bloc,sx,sy,ex,ey,oldchar,newchar);
    	}
    }else if(strcmp(token,"SUB_ID")==0){
    	token = strtok(NULL,"\"");
    	sx = atoi(token);
    	token = strtok(NULL,"\"");
    	bloc[sx].id = (char *) realloc(bloc[sx].id,sizeof(char)*(strlen(token)+1));
    	bloc[sx].id = strcpy(bloc[sx].id,token);
    }else if(strcmp(token,"FONT_REGION")==0){
      ALGetFourInt(&sx,&sy,&ex,&ey);
      token = strtok(NULL,TOKENS);
      ALFontRegion(fonts,sx,sy,ex,ey,atoi(token));
    }else if(strcmp(token,"FONT_RESIDUE")==0){
      token = strtok(NULL,TOKENS);
      sx = atoi(token);
      token=strtok(NULL,TOKENS);
      sy = atoi(token);
      token=strtok(NULL,TOKENS);
      ALFontRegion(fonts,sx,sy,sx,sy,atoi(token));
    }else if(strcmp(token,"LINE")==0){
      token = strtok(NULL,TOKENS);
      if(strcmp(token,"TOP")==0){
	ALGetThreeInt(&sx,&sy,&ex);
        if(ex > bloc[1].slen-1) ex = bloc[1].slen-1;
	ALDoLine(lines,sx,sy,ex,LINE_TOP,Yindx);
      }else if(strcmp(token,"BOTTOM")==0){
	ALGetThreeInt(&sx,&sy,&ex);	
        if(ex > bloc[1].slen-1) ex = bloc[1].slen-1;
	ALDoLine(lines,sx,sy,ex,LINE_BOTTOM,Yindx);
      }else if(strcmp(token,"LEFT")==0){
	ALGetThreeInt(&sx,&sy,&ey);	
        if(ey > nseq) ey = nseq;
	ALDoLine(lines,sx,sy,ey,LINE_LEFT,Yindx);
      }else if(strcmp(token,"RIGHT")==0){
	ALGetThreeInt(&sx,&sy,&ey);	
        if(ey > nseq) ey = nseq;
	ALDoLine(lines,sx,sy,ey,LINE_RIGHT,Yindx);
      }else{
	error("Illegal LINE command\n",1);
      }
    }else if(strcmp(token,"BOX_REGION")==0){
      ALGetFourInt(&sx,&sy,&ex,&ey);
      if(ex > bloc[1].slen-1) ex = bloc[1].slen-1;
      if(ey > nseq) ey = nseq;
      ALBoxRegion(lines,sx,sy,ex,ey,Yindx);
    }else if(strcmp(token,"SHADE_REGION")==0){
      ALGetFourInt(&sx,&sy,&ex,&ey);
      if(ex > bloc[1].slen-1) ex = bloc[1].slen-1;
      if(ey > nseq) ey = nseq;
      token = strtok(NULL,TOKENS);
      ALFilRegion(fills,sx,sy,ex,ey,Yindx,(float)atof(token));
    }else if(strcmp(token,"SHADE_RES")==0){
      ALGetTwoInt(&sx,&sy);
      token = strtok(NULL,TOKENS);
      ALFilRegion(fills,sx,sy,sx,sy,Yindx,(float)atof(token));
    }else if(strcmp(token,"COLOUR_REGION")==0 || strcmp(token,"COLOR_REGION")==0){
      if(!use_colour)error("You must use the DEFINE_COLOUR command first",1);
      ALGetFourInt(&sx,&sy,&ex,&ey);
      if(ex > bloc[1].slen-1) ex = bloc[1].slen-1;
      if(ey > nseq) ey = nseq;
      token = strtok(NULL,TOKENS);
      ALColRegion(s_colour,sx,sy,ex,ey,Yindx,(unsigned char) atoi(token));
    }else if(strcmp(token,"COLOUR_RES")==0 || strcmp(token,"COLOR_RES")==0){
      if(!use_colour)error("You must use the DEFINE_COLOUR command first",1);
      ALGetTwoInt(&sx,&sy);
      token = strtok(NULL,TOKENS);
      ALColRegion(s_colour,sx,sy,sx,sy,Yindx,(unsigned char) atoi(token));
    }else if(strcmp(token,"MASK")==0){
        /* mask commands */
        token = strtok(NULL,TOKENS);
        if(strcmp(token,"SETUP")==0){
            if(mask == NULL){
                mask = uchararr(bloc[1].slen+1,nseq+1);
                GJUCinit(mask,bloc[1].slen+1,nseq+1,0);
            }else{
                GJUCinit(mask,bloc[1].slen+1,nseq+1,0);
	    }
	}else if(strcmp(token,"RESET")==0){
            GJUCinit(mask,bloc[1].slen+1,nseq+1,0);
        }else if(strcmp(token,"LEGAL")==0){
            /* get the legal character list */
            token = strtok(NULL,TOKENS);
            if(legal != NULL) GJfree(legal);
            legal = GJstrdup(token);
            if(!silent)fprintf(stderr,"legal:%s\n",legal);
        }else if(strcmp(token,"ILLEGAL")==0){
            /* get the illegal character list */
            token = strtok(NULL,TOKENS);
            if(illegal != NULL) GJfree(illegal);
            illegal = GJstrdup(token);
            if(!silent)fprintf(stderr,"illegal:%s:\n",illegal);
        }else if(strcmp(token,"ID")==0){
            ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
            token = strtok(NULL,TOKENS);
            id_cut = atoi(token);
            if(id_cut < (int) ( ( ((float) (ey-sy+1) / 2.0) + 0.6))){
                error("ID cutoff must be > half the number of sequences",1);
            }
            ALid_mask(mask,bloc,sx,sy,ex,ey,id_cut,legal,illegal);
        }else if(strcmp(token,"AGREE")==0){
            ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
            token= strtok(NULL,TOKENS);
            ALagree_mask(mask,bloc,sx,sy,ex,ey,atoi(token));
        }else if(strcmp(token,"FRE")==0){
            ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
            ALfre_mask(mask,bloc,sx,sy,ex,ey,legal,illegal);
        }else if(strcmp(token,"NOT")==0){
            ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
            ALnot_mask(mask,sx,sy,ex,ey);
        }else if(strcmp(token,"SUB")==0){
            ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
            token=strtok(NULL,TOKENS);
            ALsub_mask(mask,bloc,sx,sy,ex,ey,token[0]);
        }else if(strcmp(token,"REGION")==0){
            ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
            ALMask(mask,sx,sy,ex,ey);
        }else if(strcmp(token,"BOX")==0){
            /* box the masked residues - ie surround them*/
            ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
            ALSurroundMask(mask,bloc,lines,sx,sy,ex,ey,nseq);
        }else if(strcmp(token,"SHADE")==0){
            /* shade the masked residues - */
            ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
            token = strtok(NULL,TOKENS);
            ALShadeMask(mask,fills,sx,sy,ex,ey,Yindx,(float)atof(token));
        }else if(strcmp(token,"FONT")==0){
            /* set the font of the masked residues - */
            ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
            token = strtok(NULL,TOKENS);
            ALFontMask(mask,fonts,sx,sy,ex,ey,(unsigned char) atoi(token));
        }else if(strcmp(token,"INVERSE")==0){
            /* invert the masked residues - */
            ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
            token = strtok(NULL,TOKENS);
            ALInverseMask(mask,inverse,sx,sy,ex,ey);
        }else if(strcmp(token,"SCOL")==0){
            if(!use_colour)error("You must use the DEFINE_COLOUR command first",1);
            ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
            token = strtok(NULL,TOKENS);
            ALSColMask(mask,s_colour,sx,sy,ex,ey,nseq,(unsigned char) atoi(token));
        }else if(strcmp(token,"CCOL")==0){
            if(!use_colour)error("You must use the DEFINE_COLOUR command first",1);
            ALGetAllRange(&sx,&sy,&ex,&ey,bloc[1].slen-1,nseq);
            token = strtok(NULL,TOKENS);
            ALColourMask(mask,c_colour,sx,sy,ex,ey,(unsigned char) atoi(token));
	}else{
	    error("Unrecognised MASK qualifier\n",1);
	}
    }else if(strcmp(token,"TEXT")==0){
      token=strtok(NULL,TOKENS);
      sx = atoi(token);
      token=strtok(NULL,"\"");
      sy = atoi(token);
      token=strtok(NULL,"\"");
      texts[sx][sy] = (char *) malloc(sizeof(char)*(strlen(token)+1));
      texts[sx][sy] = (char *) strcpy(texts[sx][sy],token);
    }else if(strcmp(token,"NUMBERS")==0){
        error("Numbers not implemented",1);
    }else if(strcmp(token,"ID_FONT")==0){
      token = strtok(NULL,TOKENS);
      if(strcmp(token,"ALL")==0){
	token = strtok(NULL,TOKENS);
	for(i=1;i<nseq+1;++i)idfonts[i]= (unsigned char) atoi(token);
      }else{
	sx = atoi(token);
	token = strtok(NULL,TOKENS);
	idfonts[sx] = atoi(token);
      }

    }else{
      fprintf(std_err,"Unrecognised Step 2 command\n");
      fprintf(std_err,"%s",buff);
    }
  }
  if(!silent){
    ALReportInfo(nseq,std_err);
  }
  ALCheckSinglePage(nseq,bloc[1].slen,silent);

  start_aa = 1;
  totalpage = npage * Csheet;

  /* get how many blocks of sequence we can fit per page */
  nspage = Cheight/(nseq+CtopSpace+Vspacing);
  if(nspage < 1){
    nspage = 1;
  }
  nbp = 0;

  if(!silent)fprintf(std_err,"Will write a maximum of %d blocks of sequence per page\n",nspage);

  if((Cwidth - Cidwidth > 0)){
    if(!silent)fprintf(std_err,"Saving  (%d) Pages to file\n",totalpage/nspage);
  }else{
    fprintf(std_err,"This pointsize is too large to print even a single character on the page\n");
    fprintf(std_err,"Try a smaller pointsize - or\n");
    fprintf(std_err,"Try reducing the Identifier width\n");
    fprintf(std_err,"Increase the paper size\n");
    fprintf(std_err,"You could do some PostScrip-ery to print partial characters\n");
    fprintf(std_err,"After creating an alignment on a single page - see the CookBook\n");
    exit(1);
  }
  lastY = 0;
  nbpage = 0;
  k = 0;
  start_seq = 1;
  while(k < totalpage){
    start_aa = 1;
    Ytemp = start_seq+Cheight-CtopSpace - 1;
    if(Ytemp > nseq) Ytemp = nseq;
    for(i=0;i<npage;++i){

      if(nspage > 1){        /* more than one block per page */
/*	  fprintf(stderr,"nbp nspage %d %d\n",nbp,nspage);*/
	  if(k>0 && nbp == 0){ /* if not the first page, then setup page*/
	    PSSetupPage(orientation,nbpage+1,outf);
	  }
	  if(nbp>0){
	    /* if this is not the first block on the page, then */
	    /* translate the block of alignment back to origin  */
	    fprintf(outf,"%d %d translate\n",0,-lastY);
	  }
	  /* get new position for drawing and translate to this posn*/
	  lastY = Yspace*( Cheight-(nseq+CtopSpace)*(nbp+1) - (nbp*Vspacing) );
	  fprintf(outf,"%d %d translate\n",0,lastY);
      }else{
	if(k>0)PSSetupPage(orientation,k+1,outf);
      }

      Xtemp = start_aa+Cseqwidth-1;
      if(Xtemp > bloc[1].slen) Xtemp = bloc[1].slen;

      fprintf(outf,"2 setlinecap\n");
      fprintf(outf," %.2f setlinewidth\n",LineWidthFactor*pointsize);

      ALDrawFills(fills,use_colour,s_colour,start_aa,nseq-Ytemp+1,Xtemp,nseq-start_seq+1,Cidwidth,CseqVoffset,outf);
      ALDrawLines(lines,use_colour,start_aa,nseq-Ytemp+1,Xtemp,nseq-start_seq+1,Cidwidth,CseqVoffset,outf);

      ALOutId(bloc,use_colour,idfonts,start_seq,Ytemp,CseqVoffset,number_seqs,outf);
      ALOutSeq(bloc,fonts,texts,inverse,use_colour,c_colour,start_seq,Ytemp,start_aa,CseqVoffset,outf);
      
      if(res_numbers){
        if(do_ticks){
          ALOutTicks(ticks,start_aa,Cidwidth,Ytemp-start_seq+1,bloc[1].slen,outf);
          ALOutNumbers(numbers,start_aa,Cidwidth,Ytemp-start_seq+2,bloc[1].slen,outf);
        }else{
          ALOutNumbers(numbers,start_aa,Cidwidth,Ytemp-start_seq+1,bloc[1].slen,outf);
        }
      }
      start_aa += Cseqwidth;

      if(nspage > 1){
	++nbp;
	if(nbp == nspage || k == totalpage-1){  /* put showpage command for page or last page */
	  PSShowPage(outf);
	  nbp = 0;
	  ++nbpage;
	}
      }else{
	PSShowPage(outf);
      }
      ++k;
    }
    start_seq = Ytemp + 1;

  }
  return 1;
}

void ALShadeChars(struct seqdat *bloc,
signed char **fill,
int start_x,int start_y,int end_x,int end_y,
int maxY,
char *charlist,
float grey)

/* shade residues in charlist by grey value */

{
  int i,j,jpos;
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y,jpos=(maxY-start_y+1);j<end_y+1;++j,--jpos){
       if(strchr(charlist,bloc[j].seq[i])!=NULL){
	 ALFilRes(fill,i,jpos,grey);
       }
     }
   }
}
void ALSColChars(struct seqdat *bloc,unsigned char **colour,int start_x,
int start_y,int end_x,int end_y,int maxY,char *charlist,unsigned char colnum)

/* colour residue backgrounds in charlist by colnum value */

{
  int i,j,jpos;
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y,jpos=(maxY-start_y+1);j<end_y+1;++j,--jpos){
       if(strchr(charlist,bloc[j].seq[i])!=NULL){
	 ALColRes(colour,i,jpos,colnum);
       }
     }
   }
}

void ALSColMask(unsigned char **mask,unsigned char **colour,
                int start_x,int start_y,int end_x,int end_y,int maxY,
		unsigned char colnum)

/* colour residue backgrounds in mask by colnum value */

{
  int i,j,jpos;
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y,jpos=(maxY-start_y+1);j<end_y+1;++j,--jpos){
       if(mask[i][j]==1){
	 ALColRes(colour,i,jpos,colnum);
       }
     }
   }
}
void ALInverseChars(struct seqdat *bloc,unsigned char **inverse,
		    int start_x,int start_y,int end_x,int end_y,char *charlist)

/* Invert the characters within the range */
/* 18/Jan/93:  Allow inverted chars to be reverted */

{
  int i,j;
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y;j<end_y+1;++j){
       if(strchr(charlist,bloc[j].seq[i])!=NULL){
       	 if(inverse[i][j] == 1){
       	 	inverse[i][j] = (unsigned char) 0;
       	 }else{
	 	inverse[i][j] = (unsigned char ) 1;
	 }
       }
     }
   }
}
void ALFInverseChars(struct seqdat *bloc,unsigned char **inverse,
		    int start_x,int start_y,int end_x,int end_y,char *charlist)

/* Invert the characters within the range */
/* this version does not allow reversion ie toggling*/

{
  int i,j;
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y;j<end_y+1;++j){
       if(strchr(charlist,bloc[j].seq[i])!=NULL){
	 inverse[i][j] = 1;
       }
     }
   }
}
void ALInverseMask(unsigned char **mask,unsigned char **inverse,
		   int start_x,int start_y,int end_x,int end_y)
/* Invert the characters within the range of the mask*/
/* 18/Jan/93:  Allow inverted chars to be reverted */

{
  int i,j;
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y;j<end_y+1;++j){
       if(mask[i][j]){
       	 if(inverse[i][j] == 1){
       	 	inverse[i][j] = (unsigned char) 0;
       	 }else{
	 	inverse[i][j] = (unsigned char ) 1;
	 }
       }
     }
   }
}

void ALFontChars(struct seqdat *bloc,unsigned char **fonts,
		 int start_x,int start_y,int end_x,int end_y,char *charlist,unsigned char fontnum)

/* set the characters within the range to the fontnum */

{
  int i,j;
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y;j<end_y+1;++j){
       if(strchr(charlist,bloc[j].seq[i])!=NULL){
	 fonts[i][j] = fontnum;
       }
     }
   }
}
void ALFontRegion(unsigned char **fonts,
		  int start_x,int start_y,int end_x,int end_y,
		  unsigned char fontnum)
/* set the characters within the range to the fontnum */

{
  int i,j;
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y;j<end_y+1;++j){
	 fonts[i][j] = fontnum;
     }
   }
}
void ALFontMask(unsigned char **mask,unsigned char **fonts,
		int start_x,int start_y,int end_x,int end_y,
		unsigned char fontnum)

/* set the characters within the range to the fontnum */

{
  int i,j;
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y;j<end_y+1;++j){
        if(mask[i][j]==1){
	 fonts[i][j] = fontnum;
	}
     }
   }
}

void ALColourChars(struct seqdat *bloc,unsigned char **colour,
		   int start_x,int start_y,int end_x,int end_y,
		   char *charlist,unsigned char colournum)

/* set the characters within the range to the colour defined by colournum */

{
  int i,j;
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y;j<end_y+1;++j){
       if(strchr(charlist,bloc[j].seq[i])!=NULL){
	 colour[i][j] = colournum;
       }
     }
   }
}
void ALColourRegion(unsigned char **colour,
		    int start_x,int start_y,int end_x,int end_y,
		    unsigned char colournum)

/* set the characters within the range to the colournum */

{
  int i,j;
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y;j<end_y+1;++j){
	 colour[i][j] = colournum;
     }
   }
}

void ALColourMask(unsigned char **mask,unsigned char **colour,
		  int start_x,int start_y,int end_x,int end_y,
		  unsigned char colournum)

/* set the characters within the range to the colournum */

{
  int i,j;
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y;j<end_y+1;++j){
        if(mask[i][j] == 1){
	 colour[i][j] = colournum;
	}
     }
   }
}

  
void ALSubChars(struct seqdat *bloc,
		int start_x,int start_y,int end_x,int end_y,
		char oldchar,char newchar)

/* substitute oldchar for newchar at each position in bloc */

{
  int i,j;
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y;j<end_y+1;++j){
	 if(bloc[j].seq[i] == oldchar) bloc[j].seq[i] = newchar;
     }
   }
}


void ALSurroundChars(struct seqdat *bloc,unsigned char **lines,
		     int start_x,int start_y,int end_x,int end_y,
		     int maxY,char *charlist)

/* 
within the range of start_x etc.  Draw lines such that every character that
is present within charlist is surrounded by a box.
*/
{
  int i,j,jpos;
  char above,below,left,right;
  
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y,jpos=(maxY-start_y+1);j<end_y+1;++j,--jpos){
      if(strchr(charlist,bloc[j].seq[i]) != NULL){
/*	fprintf(std_err,"%d %d %c %c\n",i,j,bloc[j].seq[i]*/
	if(j == end_y){
	  below = '!';
	}else{
	  below = bloc[j+1].seq[i];
	}
	if(j == start_y){
	  above = '!';
	}else{
	  above = bloc[j-1].seq[i];
	}
	if(i == start_x){
	  left = '!';
	}else{
	  left  = bloc[j].seq[i-1];
	}
	if(i == end_x){
	  right = '!';
	}else{
	  right = bloc[j].seq[i+1];
	}

	if(strchr(charlist,above)== NULL){
	  ALLineRes(lines,i,jpos,LINE_TOP);
	}
	if(strchr(charlist,below)== NULL){
	  ALLineRes(lines,i,jpos,LINE_BOTTOM);
	}
	if(strchr(charlist,left)==NULL){
	  ALLineRes(lines,i,jpos,LINE_LEFT);
	}
	if(strchr(charlist,right)==NULL){
	  ALLineRes(lines,i,jpos,LINE_RIGHT);
	}
      }
    }
  }
}
void ALSurroundMask(unsigned char **mask,struct seqdat *bloc,unsigned char **lines,
		    int start_x,int start_y,int end_x,int end_y,
		    int maxY)
/* 
within the range of start_x etc.  Draw lines such that every
masked position is isolated from every non-masked position
*/
{
  int i,j,jpos;
  char above,below,left,right;

/*  fprintf(stdout,"ALSurroundMask: %d %d %d %d\n",start_x,start_y,end_x,end_y);*/
  
  for(i=start_x;i<end_x+1;++i){
     for(j=start_y,jpos=(maxY-start_y+1);j<end_y+1;++j,--jpos){
        if(mask[i][j] == 1){
/*	fprintf(std_err,"%d %d %c %c\n",i,j,bloc[j].seq[i]*/
	if(j == end_y){
	  below = 0;
	}else{
	  below = mask[i][j+1];
	}
	if(j == start_y){
	  above = 0;
	}else{
	  above = mask[i][j-1];
	}
	if(i == start_x){
	  left = 0;
	}else{
	  left  = mask[i-1][j];
	}
	if(i == end_x){
	  right = 0;
	}else{
	  right = mask[i+1][j];
	}
	/*
	fprintf(stdout,"IN ALSurroundMask\n");
	fprintf(stdout,"%d %d (%d %d %d %d)\n",i,j,above,below,left,right);
	*/

	if(mask[i][j] != above){
	  ALLineRes(lines,i,jpos,LINE_TOP);
	}
	if(mask[i][j] != below){
	  ALLineRes(lines,i,jpos,LINE_BOTTOM);
	}
	if(mask[i][j] != left){
	  ALLineRes(lines,i,jpos,LINE_LEFT);
	}
	if(mask[i][j] != right){
	  ALLineRes(lines,i,jpos,LINE_RIGHT);
	}
	}
     }
  }
}

void ALFilRegion(signed char **fill,
		 int start_x,int start_y,int end_x,int end_y,
		 int *Yindx,float grey)
{
  int i,j;
  for(i=start_x;i<end_x+1;++i){
    for(j=start_y;j<end_y+1;++j){
      ALFilRes(fill,i,Yindx[j],grey);
    }
  }
}

void ALShadeMask(unsigned char **mask,signed char **fill,
		 int start_x,int start_y,int end_x,int end_y,int *Yindx,
		 float grey)

{
  int i,j;
  for(i=start_x;i<end_x+1;++i){
    for(j=start_y;j<end_y+1;++j){
        if(mask[i][j]==1){
            ALFilRes(fill,i,Yindx[j],grey);
        }
    }
  }
}


void ALColRegion(unsigned char **colour,
		 int start_x,int start_y,int end_x,int end_y,
		 int *Yindx,unsigned char colnum)
{
  int i,j;
  for(i=start_x;i<end_x+1;++i){
    for(j=start_y;j<end_y+1;++j){
      ALColRes(colour,i,Yindx[j],colnum);
    }
  }
}

void ALBoxRegion(unsigned char **lines,
		 int start_x,int start_y,int end_x,int end_y,
		 int *Yindx)
{
  int i;
  for(i=start_x;i<end_x+1;++i){
    ALLineRes(lines,i,Yindx[end_y],LINE_BOTTOM);
    ALLineRes(lines,i,Yindx[start_y],LINE_TOP);
  }
  for(i=start_y;i<end_y+1;++i){
    ALLineRes(lines,start_x,Yindx[i],LINE_LEFT);
    ALLineRes(lines,end_x,Yindx[i],LINE_RIGHT);
  }
}
void ALDoLine(unsigned char **lines,
	      int start_x,int start_y,int end,
	      int type,int *Yindx)
{
  int i;
  if(type == LINE_TOP || type == LINE_BOTTOM){
    for(i=start_x;i<end+1;++i){
      ALLineRes(lines,i,Yindx[start_y],type);
    }
  }else{
    for(i=start_y;i<end+1;++i){
      ALLineRes(lines,start_x,Yindx[i],type);


    }
  }
}

void ALDrawFills(signed char **fill,int use_colour,unsigned char **colour,
		 int start_x,int start_y,int end_x,int end_y,int XOffset,int YOffset,FILE *outf)

/* for values of fill >= 0  draw box and fill in with grey value */
/* 6/12/92:  add colour option */
{
  int i,j;
  extern int precis;

  XOffset *= Xspace;
  YOffset *= Yspace;
  
  for(i=end_x;i>(start_x-1);--i){
    for(j=end_y;j>(start_y-1);--j){
      if((int) fill[i][j] >=  0 || (use_colour && colour[i][j] != 99)){
        /* only do this if a non white fill, or a non white colour */
/*
Revise to use ML function 27th Oct 1992
	fprintf(outf," newpath\n");
	fprintf(outf,"%.2f %.2f moveto \n",(float)(i-start_x)*Xspace-Xshift+XOffset,
		                         (float)(j-start_y+1)*Yspace-Yshift+YOffset);
	fprintf(outf,"%.2f %.2f lineto \n",(float)(i-start_x)*Xspace-Xshift+XOffset,
		                         (float)(j-start_y+0)*Yspace-Yshift+YOffset);
	fprintf(outf,"%.2f %.2f lineto \n",(float)(i-start_x+1)*Xspace-Xshift+XOffset,
		                         (float)(j-start_y+0)*Yspace-Yshift+YOffset);
        fprintf(outf,"%.2f %.2f lineto \n",(float)(i-start_x+1)*Xspace-Xshift+XOffset,
		                         (float)(j-start_y+1)*Yspace-Yshift+YOffset);
	fprintf(outf,"%.2f %.2f lineto \n",(float)(i-start_x)*Xspace-Xshift+XOffset,
		                         (float)(j-start_y+1)*Yspace-Yshift+YOffset);
	fprintf(outf,"closepath %.2f setgray fill 0 setgray \n",(float)fill[i][j]/(float)precis);
*/
	fprintf(outf," newpath\n");
	fprintf(outf,"%.2f %.2f %.2f %.2f %.2f %.2f %.2f %.2f %.2f %.2f ML\n",
               (float)(i-start_x)*Xspace-Xshift+XOffset,
	       (float)(j-start_y+1)*Yspace-Yshift+YOffset,
	       (float)(i-start_x)*Xspace-Xshift+XOffset,
	       (float)(j-start_y+0)*Yspace-Yshift+YOffset,
	       (float)(i-start_x+1)*Xspace-Xshift+XOffset,
	       (float)(j-start_y+0)*Yspace-Yshift+YOffset,
               (float)(i-start_x+1)*Xspace-Xshift+XOffset,
	       (float)(j-start_y+1)*Yspace-Yshift+YOffset,
	       (float)(i-start_x)*Xspace-Xshift+XOffset,
	       (float)(j-start_y+1)*Yspace-Yshift+YOffset);
	if(use_colour && colour[i][j] != 99){
	    /* set to colour number then back to white */
	    fprintf(outf,"closepath \n C%-d \n fill C99\n",colour[i][j]);
	}else{
	    fprintf(outf,"closepath %.2f setgray fill 0 setgray \n",(float)fill[i][j]/(float)precis);
	}
      }
    }
  }
}

void ALFilRes(signed char **fill,int i,int j,float grey)

{
  extern int precis;
/*  ALBoxRes(lines,i,j); 
  fill[i+Cidwidth][j+CseqVoffset] = grey * precis;
*/
  fill[i][j] = grey * precis;
}

void ALColRes(unsigned char **colour,int i,int j,int colnum)

{
  colour[i][j] = colnum;
}
  

void ALBoxRes(unsigned char **lines,int i,int j)

{
  ALLineRes(lines,i,j,LINE_TOP);
  ALLineRes(lines,i,j,LINE_BOTTOM);
  ALLineRes(lines,i,j,LINE_LEFT);
  ALLineRes(lines,i,j,LINE_RIGHT);
}


void ALLineRes(
unsigned char **lines,
int i,int j,
unsigned char value)
{
/*
  i = i + Cidwidth;
  j = j + CseqVoffset;
*/
  ALLineChar(lines,i,j,value);
}

void ALLineChar(unsigned char **lines,int i,int j,unsigned char value)

{
  lines[i][j] = lines[i][j] | value;
}

void ALDrawLines(unsigned char **lines,int use_colour,
		 int start_x,int start_y,int end_x,int end_y,
		 int XOffset,int YOffset,FILE *outf)

/* draw the lines that are indicated by the lines array in the range of the
array indicated by start_x etc
*/
{
  int i,j;
  XOffset *= Xspace;
  YOffset *= Yspace;
  
  if(use_colour){
    /* set line colour to black */
    fprintf(outf,"C100\n");
  }
  
  for(i=end_x;i>(start_x-1);--i){
    for(j=end_y;j>(start_y-1);--j){
      if(lines[i][j] != 0){
	if((LINE_LEFT & lines[i][j]) == LINE_LEFT){
	  PSline((float)(i-start_x)*Xspace-Xshift+XOffset,
		 (float)(j-start_y+1)*Yspace-Yshift+YOffset,
		 (float)(i-start_x)*Xspace-Xshift+XOffset,
		 (float)(j-start_y+0)*Yspace-Yshift+YOffset,outf);
	}
	if((LINE_RIGHT & lines[i][j]) == LINE_RIGHT){
	  PSline((float)(i-start_x+1)*Xspace-Xshift+XOffset,
		 (float)(j-start_y+1)*Yspace-Yshift+YOffset,
		 (float)(i-start_x+1)*Xspace-Xshift+XOffset,
		 (float)(j-start_y+0)*Yspace-Yshift+YOffset,outf);
	}
	if((LINE_TOP & lines[i][j]) == LINE_TOP){
	  PSline((float)(i-start_x)*Xspace-Xshift+XOffset,
		 (float)(j-start_y+1)*Yspace-Yshift+YOffset,
		 (float)(i-start_x+1)*Xspace-Xshift+XOffset,
		 (float)(j-start_y+1)*Yspace-Yshift+YOffset,outf);
	}
	if((LINE_BOTTOM & lines[i][j]) == LINE_BOTTOM){
	  PSline((float)(i-start_x)*Xspace-Xshift+XOffset,
		 (float)(j-start_y+0)*Yspace-Yshift+YOffset,
		 (float)(i-start_x+1)*Xspace-Xshift+XOffset,
		 (float)(j-start_y+0)*Yspace-Yshift+YOffset,outf);
	}
      }
    }
  }
  if(use_colour){
    /* set colour back to white */
    fprintf(outf,"C99\n");
  }
}

void PSline(float x1,float y1,float x2,float y2,FILE *outf)

{
  fprintf(outf,"%.2f %.2f %.2f %.2f L\n ",x1,y1,x2,y2);
}



void ALSimpleBox(int x1,int y1,int x2,int y2,float pfactor,    /* multiply pointsize by this to get linewidth */         
		 FILE *outf)

/* draw a box to include character coordinates x1,y1 to x2,y2 */
/* x1 < x2 y1< y2 */
{
  float lshift = Xspace/6;
  float rshift = Xspace - lshift;
  
  x1 *= Xspace; 
  y1 *= Yspace;
  x2 *= Xspace;
  y2 *= Yspace;

  x1 -= lshift;
  x2 += rshift;
  y1 -= lshift;

  PSDrawBox((float)x1,(float)y1,(float)x2,(float)y2,pfactor,outf);
}

void PSDrawBox(float x1,float y1,float x2,float y2,float pfactor,FILE *out)

{
  fprintf(out,"%f setlinewidth\n ",pointsize*pfactor);
  fprintf(out,"%f %f moveto %f %f lineto %f %f lineto %f %f lineto %f %f lineto closepath stroke\n",
               x1,y1,x2,y1,x2,y2,x1,y2,x1,y1);
}

void ALOutTicks(char *array,int start_aa,int xpos,int ypos,int len,FILE *out)

{
  int right;
  int j;
  
  xpos *= Xspace;
  ypos *= Yspace;

  right = start_aa + Cseqwidth;
  if(right > len)right = len;
  for(j=start_aa;j<right;j++){
    if(array[j] != '\0'){
      PSPutChar(xpos,ypos,array[j],out);
    }
    xpos +=  Xspace;
  }
}

void ALOutNumbers(char **array,int start_aa,int xpos,int ypos,int len,FILE *out)

{
  int right;
  int j;
  
  xpos *= Xspace;
  ypos *= Yspace;

  right = start_aa + Cseqwidth;
  if(right > len)right = len;
  for(j=start_aa;j<right;j++){
    if(array[j][0] != '\0'){
      PSPutText(xpos,ypos,array[j],out);
    }
    xpos +=  Xspace;
  }
}

char *ALCreateTicks(int start, int interval, int len)

/* create ticks at the desired interval starting with residue start */
{
  int i;
  char *temp;
  temp = (char *) malloc(sizeof(char) * len);
  /* first set all ticks to '\0' */
  for(i=0;i<len;++i){
    temp[i]='\0';
  }
  for(i=0;i<len;i+=interval){
    temp[i] = '|';
  }
  temp[0]='\0';
  temp[1]='|';
  return temp;
}



char **ALCreateNumbers(int start,int interval,int nres,int nwidth)

/* create character array of numbers from start to nres at interval*/
{
  int i;
  char **temp;
  temp = (char **) malloc(nres*sizeof(char *));
  for(i=0;i<nres;++i){
    temp[i] = (char *) malloc(sizeof(char) *nwidth);
    temp[i][0]='\0';
  }
  for(i=start;i<nres;i+=interval){
    if(i==0){
      sprintf(temp[i+1],"%-d",i+1);      
    }else{
      sprintf(temp[i],"%-d",i);
    }
  }

  return temp;
}
  
char **ALCreateNum2(
int nstart,     /* starting number */
int ninterval,  /* interval for numbers */
int nend,       /* end of number range */
int nres,       /* total length of sequence */
int xstart,     /* starting location for writing numbers */
int nwidth)     /* width of the number string */

/* create character array of numbers from start to nres at interval*/
{
  int i,xp;
  char **temp;
  temp = (char **) malloc(nres*sizeof(char *));
  for(i=0;i<nres;++i){
    temp[i] = (char *) malloc(sizeof(char) *nwidth);
    temp[i][0]='\0';
  }
  xp = xstart;
  for(i=nstart;i<nend;i+=ninterval){
    if(i==0){
      sprintf(temp[xp+1],"%-d",i+1);      
    }else{
      sprintf(temp[xp],"%-d",i);
    }
    xp += ninterval;
  }

  return temp;
}

void ALReportInfo(int nseq,FILE *out)

{
  fprintf(out,"Number of Sequences: %d\n",nseq);
  fprintf(out,"Total Vertical Space per page: %d\n",Cheight);
}

void ALReserveSpace(int location,int nspace)

{
  extern int TVspace,CseqVoffset;

  switch(location){
  case TOP:
    TVspace += nspace;
    CtopSpace +=nspace;
    break;
  case BOTTOM:
    TVspace += nspace;
    CseqVoffset += nspace;
    CbottomSpace += nspace;
    break;
  }
}

void PSSetupPage(int orientation,int npage,FILE *outf)

{
  fprintf(outf,"%%%%Page: ? %d\n",npage);
  PSSetOrientation(orientation,outf);
  PSSetFont(0,outf);    /* set default font for document */
}

void ALOutSeq(struct seqdat *bloc,unsigned char **fonts,char ***texts,unsigned char **inverse,
	      int use_colour,unsigned char **colour,
	      int start_seq,int end_seq,int start_aa,int ypos,FILE *outf)

{
  extern int Xspace,Yspace;
  extern int Cidwidth;

  int i,j;
  int xpos;
  unsigned char current_col;

  int right;

  ypos *= Yspace;

  current_col = 99;

  right = start_aa + Cseqwidth;
  if(right > bloc[1].slen-1)right = bloc[1].slen;
  
  for(i=end_seq;i>(start_seq-1);--i){
    xpos = Cidwidth * Xspace;
    for(j=start_aa;j<right;++j){
      if(texts[j][i] != NULL || bloc[i].seq[j] != ' '){
	if(fonts[j][i]!=0)PSSetFont(fonts[j][i],outf);
	if(inverse[j][i] != 0)PSSetGrey(1,outf);
	if(use_colour && colour[j][i] != current_col){
	    PSSetColour(colour[j][i],outf);
	    current_col = colour[j][i];
	} 
	PSPutChar(xpos,ypos,bloc[i].seq[j],outf);
	if(texts[j][i] != NULL){
	  PSPutText(xpos,ypos,texts[j][i],outf);
	}
/*
	if(use_colour && colour[j][i] != 100){
	    PSSetColour(100,outf);
	} 
*/
	if(inverse[j][i] != 0)PSSetGrey(0,outf);
	if(fonts[j][i]!=0)PSSetFont(0,outf);
      }
      xpos += Xspace;
    }
    ypos += Yspace;
  }
  if(use_colour){
	PSSetColour(100,outf);
  }
}


void ALOutId(struct seqdat *bloc,int use_colour,unsigned char *fonts,
	     int start_seq,int end_seq,int ypos,int numbers,
             FILE *outf)

{
  extern int MAXilen;
  extern int Xspace,Yspace;
  int i;
  int xpos;
  char *temp;

  xpos = 0;
  ypos *= Yspace;
  temp = (char *) malloc(sizeof(char) * (MAXilen + 5));

  if(use_colour){
    /* set line colour to black */
    fprintf(outf,"C100\n");
  }

  for(i=end_seq;i>(start_seq-1);--i){
    if(fonts[i] != 0)PSSetFont(fonts[i],outf);
    if(numbers){
      sprintf(temp,"%4d:%s",i,bloc[i].id);
    }else{
      sprintf(temp,"%s",bloc[i].id);
    }
    PSPutText(xpos,ypos,temp,outf);
    ypos += Yspace;
    if(fonts[i] != 0)PSSetFont(0,outf);
  }
  free((char *) temp);
  if(use_colour){
    /* set colour back to white */
    fprintf(outf,"C99\n");
  }
}

void PSPutText(int x,int y,char *text,FILE *outf)

{
  fprintf(outf," %d %d moveto (%s) show \n",x,y,text);
}

void PSPutChar(int x,int y,char c,FILE *outf)

{
  /* fprintf(outf," %d %d moveto (%c) show \n",x,y,c);*/
  fprintf(outf,"(%c) %d %d P\n",c,x,y);
}

void PSShowPage(FILE *outf)

{
  fprintf(outf,"showpage\n");
}

void ALSetFont(char *font,int point,FILE *outf)

{
  fprintf(outf,"/%s findfont %d scalefont setfont\n",font,point);
}

void PSSetFont(unsigned char number,FILE *outf)

{
  fprintf(outf,"F%-d\n ",number);
}

void PSSetColour(unsigned char number,FILE *outf)

{
  fprintf(outf,"C%-d\n ",number);
}

void ALDefineFont(int number,char *font,float point,FILE *outf)

/* define a PostScript function called /Fnumber which selects the 
font at point pointsize */
{
  fprintf(outf,"/F%-d {/%s findfont %.2f scalefont setfont } def\n",
	  number,font,point);
}

void ALDefineColour(int number,float red,float green,float blue,FILE *outf)

/* define a PostScript function called /Cnumber which selects the 
colour defined by red,green, blue intensities */
{
  fprintf(outf,"/C%-d {%f %f %f setrgbcolor } def\n", number,red,green,blue);
}

void ALCheckSinglePage(int nseq,int len,int silent)

{
  extern FILE *std_err;
  Csheet = 1;


  if(Cheight-CtopSpace < nseq){
    if(!silent)fprintf(std_err,"Not enough height to print all sequences on a single page at this pointsize\n");
    if(!silent)fprintf(std_err,"Pointsize: %.2f\n",pointsize);
    if(!silent)fprintf(std_err,"Need a pointsize of < : %.2f\n",(float) height/(nseq+ CtopSpace));
    Csheet = nseq/(Cheight-CtopSpace) + 1;
    if(!silent)fprintf(std_err,"Will split alignment into %d segments\n",Csheet);
  }
  
  npage = len/(Cwidth - Cidwidth)  + 1;
}


void ALSetPageLimits(void)
/* Use the current width height and pointsize values to calculate:
   Cwidth
   Cheight
   Xspace
   Yspace
*/
{

  Xspace = pointsize + pointsize * XspaceFactor;
  Yspace = pointsize + pointsize * YspaceFactor;

  Cwidth = width/Xspace;
  Cheight = height/Yspace;

  Cseqwidth = Cwidth - Cidwidth;

  Xshift = Xspace * XshiftFactor;
  Yshift = Yspace * YshiftFactor;
}

void PSSetOrientation(int orientation,FILE *outf)

{
  switch(orientation){
  case LANDSCAPE:
    PSLandscape(outf);
    break;
  case PORTRAIT:
    PSPortrait(outf);
    break;
  default:
    PSPortrait(outf);
  }
}

void PSPreamble(FILE *out)
{
  fprintf(out,"%%!PS-Adobe\n");
  fprintf(out,"%%%%Creator: ALSCRIPT Version 1.4.4\n");
  fprintf(out,"%%%%Title: Alscript Output\n");
  fprintf(out,"%% Barton, G. J. (1993)\n");
  fprintf(out,"%% ALSCRIPT - A Tool to format multiple sequence alignments\n");
  fprintf(out,"%% Protein Engineering, 6, 37-40\n");
  fprintf(out,"%%\n");
  fprintf(out,"/P {moveto show} def\n");
  fprintf(out,"/L {moveto lineto stroke} def\n");
  fprintf(out,
  "/ML {moveto lineto lineto lineto lineto} def\n");
  fprintf(out,"2 setlinecap\n");                 /* projecting line caps */
  ALDefineColour(99,1.0,1.0,1.0,out);  /* white */
  ALDefineColour(100,0.0,0.0,0.0,out);  /* black */

}

void PSLandscape(FILE *out)

{
  width = MAXside;
  height = MINside;

  fprintf(out,"%d %d translate\n",xoff+height,yoff);
  fprintf(out,"90 rotate\n");  

}

void PSPortrait(FILE *out)

{
  width = MINside;
  height = MAXside;
  fprintf(out,"%d %d translate\n",xoff,yoff);
}


void echo(char *s)

{
  extern FILE *std_err;
  fprintf(std_err,"%s",s);
}

void ALGetFourInt(int *sx,int *sy,int *ex,int *ey)

{
char *token;

	token = strtok(NULL,TOKENS);
	*sx = atoi(token);
	token = strtok(NULL,TOKENS);
	*sy = atoi(token);
	token = strtok(NULL,TOKENS);
	*ex = atoi(token);
	token = strtok(NULL,TOKENS);
	*ey = atoi(token);
}
void ALGetThreeInt(int *sy,int *ex,int *ey)

{
char *token;

	token = strtok(NULL,TOKENS);
	*sy = atoi(token);
	token = strtok(NULL,TOKENS);
	*ex = atoi(token);
	token = strtok(NULL,TOKENS);
	*ey = atoi(token);
}
void ALGetTwoInt(int *sy,int *ex)

{
char *token;

	token = strtok(NULL,TOKENS);
	*sy = atoi(token);
	token = strtok(NULL,TOKENS);
	*ex = atoi(token);
}

char ALChekSpace(const char *token)
{
  if(strcmp(token,"SPACE") == 0){
    return ' ';
  }else{
    return token[0];
  }
}

void PSSetGrey(float grey,FILE *outf)
{
  fprintf(outf," %.2f setgray\n",grey);
}

int save_pir(struct seqdat *bloc,int nseq,FILE *pirf)

{
  int i,j;
  int count;


   for(i=1;i<(nseq+1);++i){
    fprintf(pirf,">%s",bloc[i].id);
    fprintf(pirf,"Derived from block file\n");
    count=0;
    for(j=1;j<bloc[i].slen;++j){
      ++count;
      fputc(bloc[i].seq[j],pirf);
      if(count==50){
	count=0;
	fprintf(pirf,"\n");
      }
    }
    fprintf(pirf,"*\n");
  }
  return 1;
}
  

void ALGetAllRange(int *sx, int *sy, int *ex, int *ey,int dex,int dey)
/* checks the next token if it is ALL then return box around whole alignment
   else return box according to the integers found
*/
{
    char *token;
    
    token = strtok(NULL,TOKENS);
    
    if(strcmp(token,"ALL")==0){
                *sx = 1;
                *sy = 1;
                *ex = dex;
                *ey = dey;
    }else{
                *sx = atoi(token);
               	ALGetThreeInt(sy,ex,ey);
    }
    if(*ex > dex) *ex = dex;
    if(*ey > dey) *ey = dey;
}

void ALid_mask(
unsigned char **mask,
struct seqdat *bloc,
int sx,int sy,int ex,int ey,
int id_cut,
char *legal,
char *illegal)

{
    int i,j,k,iseen;
    char fchar;
    char *seen;   /* array of characters that are seen at this position */
    int *freq;    /* array of frequencies of the characters */
    int nc;       /* number of different character types at this position */
    int ns;
    int mc;       /* maximum character present */
    int imc;      /* location of max character */

    ns = ey - sy + 1;
    nc = 0;

    seen = (char *) GJmalloc(sizeof(char) *ns);
    freq = (int *)  GJmalloc(sizeof(int) *ns);

    for(i=0;i<ns;++i){
        freq[i] = 0;
        seen[i] = '\0';
    }

/*    fprintf(stdout,"In ALidmask\n");*/
    
    for(i=sx;i<(ex+1);++i){
        /* for each position get the list of characters seen and their frequencies */
        for(j=sy;j<(ey+1);++j){
            iseen = Ifound(bloc,i,j,seen,nc);
            if(iseen >= 0){
                ++freq[iseen];
            }else{
                seen[nc] = bloc[j].seq[i];
                ++freq[nc];
                ++nc;
            }
        }
        /*
        for(k=0;k<nc;++k)fprintf(stdout,"%c ",seen[k]);
        for(k=0;k<nc;++k)fprintf(stdout,"%d ",freq[k]);
        fprintf(stdout,"\n");
        */
        /* find the most frequent character that is legal and not illegal*/
        mc = 0;
        for(k=0;k<nc;++k){
            if(legal != NULL && illegal != NULL){
                if(  strchr(legal,seen[k]) != NULL
                  && strchr(illegal,seen[k]) == NULL
                  && freq[k] > mc){
                  mc = freq[k];
                  imc = k;
                  }
            }else if(legal != NULL){
                if(  strchr(legal,seen[k]) != NULL
                  && freq[k] > mc){ 
                  mc = freq[k];
                  imc = k;
                  }
            }else if(illegal != NULL){
                if(  strchr(illegal,seen[k]) == NULL
                  && freq[k] > mc){
                  mc = freq[k];
                  imc = k;
                  }
            }else{
                if(freq[k] > mc){
                     mc = freq[k];
                     imc = k;
                }
            }
	}
	if(mc >= id_cut){
	    /* only select the character if it is >= id_cut */
	    fchar = seen[imc];
/*	    fprintf(stdout,"At %d %c\n",i,fchar);*/
            for(j=sy;j<(ey+1);++j){
                if(bloc[j].seq[i] == fchar){
                    mask[i][j] = 1;
/*                    fprintf(stdout,"%d %d \n",i,j);*/
		}
	    }
        }
        for(j=0;j<ns;++j){
           freq[j] = 0;
           seen[j] = '\0';
	}
	nc = 0;
    }
    GJfree(seen);
    GJfree(freq);
}

void ALfre_mask(
unsigned char **mask,
struct seqdat *bloc,
int sx,int sy,int ex,int ey,
char *legal,
char *illegal)

{
    int i,j,k,iseen;
    char fchar;
    char *seen;   /* array of characters that are seen at this position */
    int *freq;    /* array of frequencies of the characters */
    int nc;       /* number of different character types at this position */
    int ns;
    int mc;       /* maximum character present */
    int imc;      /* location of max character */

    ns = ey - sy + 1;
    nc = 0;

    seen = (char *) GJmalloc(sizeof(char) *ns);
    freq = (int *)  GJmalloc(sizeof(int) *ns);

    for(i=0;i<ns;++i){
        freq[i] = 0;
        seen[i] = '\0';
    }

/*    fprintf(stdout,"In ALidmask\n");*/
    
    for(i=sx;i<(ex+1);++i){
        /* for each position get the list of characters seen and their frequencies */
        for(j=sy;j<(ey+1);++j){
            iseen = Ifound(bloc,i,j,seen,nc);
            if(iseen >= 0){
                ++freq[iseen];
            }else{
                seen[nc] = bloc[j].seq[i];
                ++freq[nc];
                ++nc;
            }
        }
/*        for(k=0;k<nc;++k)fprintf(stdout,"%c ",seen[k]);
        for(k=0;k<nc;++k)fprintf(stdout,"%d ",freq[k]);
        fprintf(stdout,"\n");
*/
        /* find the most frequent character that is legal and not illegal*/
        mc = 0;
        for(k=0;k<nc;++k){
            if(legal != NULL && illegal != NULL){
                if(  strchr(legal,seen[k]) != NULL
                  && strchr(illegal,seen[k]) == NULL
                  && freq[k] > mc){
                  mc = freq[k];
                  imc = k;
                  }
            }else if(legal != NULL){
                if(  strchr(legal,seen[k]) != NULL
                  && freq[k] > mc){ 
                  mc = freq[k];
                  imc = k;
                  }
            }else if(illegal != NULL){
                if(  strchr(illegal,seen[k]) == NULL
                  && freq[k] > mc){
                  mc = freq[k];
                  imc = k;
                  }
            }else{
                if(freq[k] > mc){
                     mc = freq[k];
                     imc = k;
                }
            }
	}
        fchar = seen[imc];
/*	    fprintf(stdout,"At %d %c\n",i,fchar);*/
            for(j=sy;j<(ey+1);++j){
                if(bloc[j].seq[i] == fchar){
                    mask[i][j] = 1;
/*                    fprintf(stdout,"%d %d \n",i,j);*/
		}
	    }

        for(j=0;j<ns;++j){
           freq[j] = 0;
           seen[j] = '\0';
	}
	nc = 0;
    }
    GJfree(seen);
    GJfree(freq);
}
int Ifound(struct seqdat *bloc,
int i,int j,
char *seen,
int nc)
{
    int k;
    
    if(nc > 0){
       for(k=0;k<nc;++k){
             if(bloc[j].seq[i] == seen[k]){
                return k;
             }
       }
    }
    return -1;
}
void ALagree_mask(
unsigned char **mask,
struct seqdat *bloc,
int sx,int sy,int ex,int ey,
int ns)

{
    int i,j;

    for(i=sx;i<(ex+1);++i){
        /* for each position mask the characters that are identical to sequence ns */
        for(j=sy;j<(ey+1);++j){
            if(bloc[j].seq[i] == bloc[ns].seq[i]){
                mask[i][j] = 1;
            }
        }
    }
}

void ALnot_mask(
unsigned char **mask,
int sx,int sy,int ex,int ey)

{
    int i,j;

    for(i=sx;i<(ex+1);++i){
        /* NOT on each mask element ie 1 goes to 0, 0 to 1 */
        for(j=sy;j<(ey+1);++j){
            if(mask[i][j]==1){
                mask[i][j]=0;
            }else if(mask[i][j]==0){
                mask[i][j]=1;
            }
        }
    }
}
                    

void ALsub_mask(unsigned char **mask,struct seqdat *bloc,
		int sx,int sy,int ex,int ey,char cchar)

{
    int i,j;

    for(i=sx;i<(ex+1);++i){
        /* substitute all characters in the mask for cchar */
        for(j=sy;j<(ey+1);++j){
            if(mask[i][j]==1){
                bloc[j].seq[i] = cchar;
            }
        }
    }
}

void ALMask(unsigned char **mask,
int sx,int sy,int ex,int ey)

{
    int i,j;

    for(i=sx;i<(ex+1);++i){
        /* substitute all characters in the mask for cchar */
        for(j=sy;j<(ey+1);++j){
            mask[i][j] = 1;
        }
    }
}


