/******************************************************************************
 The computer software and associated documentation called DOMAK hereinafter
 referred to as the WORK which is more particularly identified and described in
 Appendix A of the file LICENSE.  Conditions and restrictions for use of
 this package are also in this file.

 This routine was written by Asim S. Siddiqui

 The WORK is Copyright (1995) A. S. Siddiqui and G. J. Barton

 All use of the WORK must cite:
 Siddiqui, A. S. and Barton, G. J., "Continuous and Discontinuous Domains: An
 Algorithm for the Automatic Generation of Reliable Protein Domain Definitions" 
 PROTEIN SCIENCE, 4:872-884 (1995).
*****************************************************************************/

/*
 * Name
 *    asd_focus_utils.c
 * Purpose
 *    to provide a set of routines that focus in on a range
 * Author
 *    Asim Siddiqui
 * SccsId
 *    %W%   %U%   %E%
 */

#include <rdssp.h>
#include <stdio.h>
#include <local.h>
#include <asd_structs.h>
#include <asm_mop.h>
#include <ase_error.h>
#include <asd_make_domains.h>
#include <asd_mds_utils1.h>
#include <asd_mds_utils2.h>
#include <asd_mds_utils3.h>
#include <asd_utils.h>
#include <asd_focus_utils.h>
#include <asd_check_utils.h>
#include <asd_value.h>

extern Asd_Parameters params;

void
asd_focus_multiple_domains(Asd_Domain *d_to_focus, Asd_Domain *d_to_split,
                           Asd_Contact_Info **contact_info,
                           Asd_Contact_Info **contact_rev_info, 
                           struct brookn *bn, int start_chain, int end_chain,
                           char *ss)
{
    int increment;            /* testing increment            */
    int i;                    /* loop counter                 */
    int s1_s;                 /* start of segment 1           */
    int s1_e;                 /* end   of segment 1           */
    int s2_s;                 /* start of segment 2           */
    int s2_e;                 /* end   of segment 2           */
    int s1_sn;                /* start of segment 1           */
    int s1_en;                /* end   of segment 1           */
    int s2_sn;                /* start of segment 2           */
    int s2_en;                /* end   of segment 2           */
    int s1_s_s_lim;           /* s1 start start limit         */
    int s1_e_s_lim;           /* s1 end start limit           */
    int s2_s_s_lim;           /* s2 start start limit         */
    int s2_e_s_lim;           /* s2 end start limit           */
    int s1_s_e_lim;           /* s1 start end limit           */
    int s1_e_e_lim;           /* s1 end end limit             */
    int s2_s_e_lim;           /* s2 start end limit           */
    int s2_e_e_lim;           /* s2 end end limit             */
    int flag;                 /* boolean flag                 */
    int level_f;              /* boolean flag                 */
    int level_f2;             /* boolean flag                 */
    int total_contacts;       /* total number of contacts     */
    bool_t ss_only;           /* whether to use ss only       */
    Asd_Seg_Cont s_conts;     /* segment contacts             */
    Asd_Domain *max_domain_m; /* most likely multi domain     */
    Asd_Contact_List *c_list; /* list of contacts             */

    ss_only = FALSE;
    increment = 1;

    c_list = asd_total_contacts(d_to_split, ss, ss_only, contact_info,
                                contact_rev_info, bn, &total_contacts);

    s1_s_e_lim = d_to_focus->range.s1_s_max;
    s1_e_e_lim = d_to_focus->range.s1_e_max;
    s2_s_e_lim = d_to_focus->range.s2_s_max;
    s2_e_e_lim = d_to_focus->range.s2_e_max;

    s1_s_s_lim = d_to_focus->range.s1_s_min;
    s1_e_s_lim = d_to_focus->range.s1_e_min;
    s2_s_s_lim = d_to_focus->range.s2_s_min;
    s2_e_s_lim = d_to_focus->range.s2_e_min;

    if (s1_s_s_lim != d_to_split->start1) {
        if (d_to_split->start1 == start_chain) {
            if ((s1_s_s_lim - d_to_split->start1) < params.MIN_SEGMENT_SIZE_END) {
                s1_s_s_lim = d_to_split->start1 + params.MIN_SEGMENT_SIZE_END;
            } /*if*/
        } else {
            if ((s1_s_s_lim - d_to_split->start1) < params.MIN_SEGMENT_SIZE_MID) {
                s1_s_s_lim = d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID;
            } /*if*/
        } /*if*/
    } /*if*/

    if (s1_s_s_lim == start_chain) {
        if ((s1_e_s_lim - s1_s_s_lim + 1) < params.MIN_SEGMENT_SIZE_END) {
            s1_e_s_lim = s1_s_s_lim + params.MIN_SEGMENT_SIZE_END - 1;
        } /*if*/
    } else {
        if ((s1_e_s_lim - s1_s_s_lim + 1) < params.MIN_SEGMENT_SIZE_MID) {
            s1_e_s_lim = s1_s_s_lim + params.MIN_SEGMENT_SIZE_MID - 1;
        } /*if*/
    } /*if*/

    if ((s2_s_s_lim - s1_e_s_lim) < params.MIN_SEGMENT_SIZE_MID) {
        s2_s_s_lim = s1_e_s_lim + params.MIN_SEGMENT_SIZE_MID;
    } /*if*/

    if ((s2_e_s_lim - s2_s_s_lim + 1) < params.MIN_SEGMENT_SIZE_MID) {
        s2_e_s_lim = s2_s_s_lim + params.MIN_SEGMENT_SIZE_MID - 1;
    } /*if*/

    if (s2_e_s_lim > d_to_split->end1) {
        s2_e_s_lim = d_to_split->end1;
    } /*if*/

    s1_s = s1_s_s_lim;
    s1_e = s1_e_s_lim;
    s2_s = s2_s_s_lim;
    s2_e = s2_e_s_lim;

    s1_sn = s1_s;
    s1_en = s1_e;
    s2_sn = s2_s;
    s2_en = s2_e;
    max_domain_m = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
    max_domain_m->value = 0.0;
    max_domain_m->type = 2;

    while (s1_sn <= s1_s_e_lim) {
        level_f = asd_check_seg_a(c_list, ss_only, ss, d_to_split, s1_sn,
                                  params.MIN_SEGMENT_SIZE_MID, start_chain, 2);
        while (s1_en <= s1_e_e_lim && level_f == 0) {
            level_f2 = asd_check_seg_1(c_list, ss_only, ss, d_to_split,
                                      s1_sn, s1_en);
            while (s2_sn <= s2_s_e_lim && level_f2 == 0) {
                flag = asd_calc_res_from_scratch_m(contact_info,
                                        contact_rev_info,
                                        d_to_split, max_domain_m,
                                        &s_conts, ss_only, ss,
                                        start_chain, end_chain,
                                        s1_sn, s1_en, s2_sn, s2_en, bn, c_list);
                s1_s = s1_sn;
                s1_e = s1_en;
                s2_s = s2_sn;
                s2_e = s2_en;
                while (s2_en <= s2_e_e_lim && flag !=2) {
                    flag = asd_calc_res_given_m(contact_info, contact_rev_info,
                                       d_to_split, max_domain_m, &s_conts,
                                       ss_only, ss, start_chain, end_chain,
                                       s1_sn,  s1_en,  s2_sn,  s2_en,
                                       s1_s,   s1_e,   s2_s,   s2_e,
                                       total_contacts, bn, c_list, increment);
                    if (flag == 0) {
                        s2_e = s2_en;
                    } /*if*/
                    if (s2_en == d_to_split->end1) {
                        s2_en = s2_en + increment;
                    } else {
                        s2_en = s2_en + increment;
                        if (s2_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_END) &&
                            d_to_split->end1 == end_chain) {
                            s2_en = d_to_split->end1;
                        } else if (s2_en >
                                  (d_to_split->end1 - params.MIN_SEGMENT_SIZE_MID) &&
                                   d_to_split->end1 != end_chain) {
                            s2_en = d_to_split->end1;
                        } /*if*/
                    } /*if*/
                } /*while*/
                s2_s = s2_sn;
                s2_sn = s2_sn + increment;
                if ((s2_e_s_lim - s2_sn + 1) < params.MIN_SEGMENT_SIZE_MID) {
                    s2_e_s_lim = s2_sn + params.MIN_SEGMENT_SIZE_MID - 1;
                } /*if*/

                s2_en = s2_e_s_lim;
                if (s2_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_END) &&
                             d_to_split->end1 == end_chain) {
                    s2_en = d_to_split->end1;
                } else if (s2_en >
                          (d_to_split->end1 - params.MIN_SEGMENT_SIZE_MID) &&
                           d_to_split->end1 != end_chain) {
                    s2_en = d_to_split->end1;
                } /*if*/
            } /*while*/
            s1_e = s1_en;
            s1_en = s1_en + increment;
            if ((s2_s_s_lim - s1_en) < params.MIN_SEGMENT_SIZE_MID) {
                s2_s_s_lim = s1_en + params.MIN_SEGMENT_SIZE_MID;
            } /*if*/
            s2_sn = s2_s_s_lim;

            if ((s2_e_s_lim - s2_s_s_lim + 1) < params.MIN_SEGMENT_SIZE_MID) {
                s2_e_s_lim = s2_s_s_lim + params.MIN_SEGMENT_SIZE_MID - 1;
            } /*if*/

            s2_en = s2_e_s_lim;
            if (s2_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_END) &&
                         d_to_split->end1 == end_chain) {
                s2_en = d_to_split->end1;
            } else if (s2_en >
                      (d_to_split->end1 - params.MIN_SEGMENT_SIZE_MID) &&
                       d_to_split->end1 != end_chain) {
                s2_en = d_to_split->end1;
            } /*if*/
        } /*while*/
        s1_s = s1_sn;
        s1_sn = s1_sn + increment;
        if (d_to_split->start1 == start_chain &&
            s1_sn < (start_chain + params.MIN_SEGMENT_SIZE_END)) {
            s1_sn = start_chain + params.MIN_SEGMENT_SIZE_END;
        } else if (d_to_split->start1 != start_chain &&
                   s1_sn < (d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID)) {
            s1_sn = d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID;
        } /*if*/
        if (s1_sn == start_chain) {
            if ((s1_e_s_lim - s1_sn + 1) < params.MIN_SEGMENT_SIZE_END) {
                s1_e_s_lim = s1_sn + params.MIN_SEGMENT_SIZE_END - 1;
            } /*if*/
        } else {
            if ((s1_e_s_lim - s1_sn + 1) < params.MIN_SEGMENT_SIZE_MID) {
                s1_e_s_lim = s1_sn + params.MIN_SEGMENT_SIZE_MID - 1;
            } /*if*/
        } /*if*/
        s1_en = s1_e_s_lim;

        if ((s2_s_s_lim - s1_e_s_lim) < params.MIN_SEGMENT_SIZE_MID) {
            s2_s_s_lim = s1_e_s_lim + params.MIN_SEGMENT_SIZE_MID;
        } /*if*/
        s2_sn = s2_s_s_lim;

        if ((s2_e_s_lim - s2_s_s_lim + 1) < params.MIN_SEGMENT_SIZE_MID) {
            s2_e_s_lim = s2_s_s_lim + params.MIN_SEGMENT_SIZE_MID - 1;
        } /*if*/
        s2_en = s2_e_s_lim;

        if (s2_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_END) &&
                     d_to_split->end1 == end_chain) {
            s2_en = d_to_split->end1;
        } else if (s2_en >
                  (d_to_split->end1 - params.MIN_SEGMENT_SIZE_MID) &&
                   d_to_split->end1 != end_chain) {
            s2_en = d_to_split->end1;
        } /*if*/
    } /*while*/

    asd_free_c_list(d_to_split, c_list);
    if (max_domain_m->range.s1_s_min == max_domain_m->range.s1_s_max &&
        max_domain_m->range.s1_e_min == max_domain_m->range.s1_e_max &&
        max_domain_m->range.s2_s_min == max_domain_m->range.s2_s_max &&
        max_domain_m->range.s2_e_min == max_domain_m->range.s2_e_max) {
        d_to_focus->start1 = max_domain_m->range.s1_s_min;
        d_to_focus->end1 = max_domain_m->range.s1_e_min;
        d_to_focus->start2 = max_domain_m->range.s2_s_min;
        d_to_focus->end2 = max_domain_m->range.s2_e_min;
    } else if (!d_to_focus->ok) {
        d_to_focus->start1 = (d_to_focus->range.s1_s_min +
                              d_to_focus->range.s1_s_max) / 2;
        d_to_focus->end1 = (d_to_focus->range.s1_e_min +
                            d_to_focus->range.s1_e_max) / 2;
        d_to_focus->start2 = (d_to_focus->range.s2_s_min +
                              d_to_focus->range.s2_s_max) / 2;
        d_to_focus->end2 = (d_to_focus->range.s2_e_min +
                            d_to_focus->range.s2_e_max) / 2;
    } else {
        fprintf(stdout, "An ERROR condition has been detected\n");
        fprintf(stdout, "This is probably the result of either,\n");
        fprintf(stdout, "a) Choosing the ss only split values, normal split\n");
        fprintf(stdout, "   values unwisely\n");
       fprintf(stdout, "b) if a) is ok setting minimum ss percentage to 1.0\n");
        fprintf(stdout, "   should solve the problem\n");
        ase_error_fatal("asd_focus_multiple_domains", "could not focus");
    } /*if*/
} /*asd_focus_multiple_domains*/

/*
 * name
 *    asd_focus_single
 * purpose
 *    to focus a single domain file
 */
void
asd_focus_single(Asd_Domain *d_to_focus, Asd_Domain *d_to_split,
                 Asd_Contact_Info **contact_info,
                 Asd_Contact_Info **contact_rev_info,
                 struct brookn *bn, int start_chain, int end_chain, char *ss)
{
    int increment;            /* testing increment            */
    int i;                    /* loop counter                 */
    int s1_s;                 /* start of segment 1           */
    int s1_e;                 /* end   of segment 1           */
    int s1_sn;                /* start of segment 1           */
    int s1_en;                /* end   of segment 1           */
    int s1_s_s_lim;           /* s1 start start limit         */
    int s1_e_s_lim;           /* s1 end start limit           */
    int s1_s_e_lim;           /* s1 start end limit           */
    int s1_e_e_lim;           /* s1 end end limit             */
    int flag;                 /* boolean flag                 */
    int total_contacts;       /* total number of contacts     */
    bool_t ss_only;           /* whether to use ss only       */
    Asd_Seg_Cont s_conts;     /* segment contacts             */
    Asd_Domain *max_domain_s; /* most likely multi domain     */
    Asd_Contact_List *c_list; /* list of contacts             */

    ss_only = FALSE;
    increment = 1;

    c_list = asd_total_contacts(d_to_split, ss, ss_only, contact_info,
                                contact_rev_info, bn, &total_contacts);

    s1_s_e_lim = d_to_focus->range.s1_s_max;
    s1_e_e_lim = d_to_focus->range.s1_e_max;

    s1_s_s_lim = d_to_focus->range.s1_s_min;
    s1_e_s_lim = d_to_focus->range.s1_e_min;
/* set up start and end positions */
    s1_s = s1_s_s_lim;
    if ((s1_e_s_lim - s1_s_s_lim + 1) < params.MIN_DOMAIN_SIZE) {
        s1_e_s_lim = s1_s_s_lim + params.MIN_DOMAIN_SIZE - 1;
    } /*if*/
    s1_e = s1_e_s_lim;
    s1_sn = s1_s;
    s1_en = s1_e;

    max_domain_s = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
    max_domain_s->value = 0.0;
    max_domain_s->type = 1;

    while (s1_sn <= s1_s_e_lim &&
           s1_sn <= (d_to_split->end1 + 1 - params.MIN_DOMAIN_SIZE)) {
        flag = asd_calc_res_from_scratch_s(contact_info, contact_rev_info,
                                    d_to_split,
                                    max_domain_s, &s_conts, ss_only, ss,
                                    start_chain, end_chain,
                                    s1_sn, s1_en, c_list);
        s1_s = s1_sn;
        s1_e = s1_en;
           
        while (s1_en <= s1_e_e_lim && flag !=2) {
            flag = asd_calc_res_given_s(contact_info, contact_rev_info,
                                       d_to_split, max_domain_s, &s_conts,
                                       ss_only, ss, start_chain, end_chain,
                                       s1_sn,  s1_en, s1_s, s1_e,
                                       total_contacts, c_list, increment, bn);
            if (flag == 0) {
                s1_e = s1_en;
            } /*if*/
            if (s1_en == d_to_split->end1) {
                s1_en = s1_en + increment;
            } else {
                s1_en = s1_en + increment;
                if (s1_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_END) &&
                             d_to_split->end1 == end_chain) {
                    s1_en = d_to_split->end1;
                } else if (s1_en > (d_to_split->end1 -
                           params.MIN_SEGMENT_SIZE_MID) &&
                           d_to_split->end1 != end_chain) {
                    s1_en = d_to_split->end1;
                } /*if*/
            } /*if*/
            if (s1_sn == d_to_split->start1 && s1_en == d_to_split->end1) {
                flag = 2;
            } /*if*/
        } /*while*/
        s1_s = s1_sn;
        s1_sn = s1_sn + increment;
        if (d_to_split->start1 == start_chain &&
                   s1_sn < (start_chain + params.MIN_SEGMENT_SIZE_END)) {
            s1_sn = start_chain + params.MIN_SEGMENT_SIZE_END;
        } else if (d_to_split->start1 != start_chain &&
                   s1_sn < (d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID)) {
            s1_sn = d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID;
        } /*if*/

        if ((s1_e_s_lim - s1_sn + 1) < params.MIN_DOMAIN_SIZE) {
            s1_e_s_lim = s1_sn + params.MIN_DOMAIN_SIZE - 1;
        } /*if*/
        s1_en = s1_e_s_lim;
        if ((s1_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_END) &&
             d_to_split->end1 == end_chain) ||
            (s1_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_MID) &&
             d_to_split->end1 != end_chain)) {
            s1_en = d_to_split->end1;
        } /*if*/

    } /*while*/

    asd_free_c_list(d_to_split, c_list);
    if (max_domain_s->range.s1_s_min == max_domain_s->range.s1_s_max &&
        max_domain_s->range.s1_e_min == max_domain_s->range.s1_e_max) {
        d_to_focus->start1 = max_domain_s->range.s1_s_min;
        d_to_focus->end1 = max_domain_s->range.s1_e_min;
    } else if (!d_to_focus->ok) {
        d_to_focus->start1 = (d_to_focus->range.s1_s_min +
                              d_to_focus->range.s1_s_max) / 2;
        d_to_focus->end1 = (d_to_focus->range.s1_e_min +
                            d_to_focus->range.s1_e_max) / 2;
    } else {
        fprintf(stdout, "An ERROR condition has been detected\n");
        fprintf(stdout, "This is probably the result of either,\n");
        fprintf(stdout, "a) Choosing the ss only split values, normal split\n");
        fprintf(stdout, "   values unwisely\n");
       fprintf(stdout, "b) if a) is ok setting minimum ss percentage to 1.0\n");
        fprintf(stdout, "   should solve the problem\n");
        ase_error_fatal("asd_focus_single", "could not focus");
    } /*if*/
} /*asd_focus_single*/

/*
 * name
 *    asd_focus_multi_sep
 * purpose
 *    to focus a multiple split
 */
void
asd_focus_multi_sep(Asd_Domain *d_to_focus, Asd_Domain *d_to_split,
                    Asd_Contact_Info **contact_info,
                    Asd_Contact_Info **contact_rev_info,
                    struct brookn *bn, int start_chain, int end_chain, char *ss)
{
    int increment;            /* testing increment            */
    int i;                    /* loop counter                 */
    int s1_s;                 /* start of segment 1           */
    int s1_e;                 /* end   of segment 1           */
    int s2_s;                 /* start of segment 2           */
    int s2_e;                 /* end   of segment 2           */
    int s1_sn;                /* start of segment 1           */
    int s1_en;                /* end   of segment 1           */
    int s2_sn;                /* start of segment 2           */
    int s2_en;                /* end   of segment 2           */
    int s1_s_s_lim;           /* s1 start start limit         */
    int s1_e_s_lim;           /* s1 end start limit           */
    int s2_s_s_lim;           /* s2 start start limit         */
    int s2_e_s_lim;           /* s2 end start limit           */
    int s1_s_e_lim;           /* s1 start end limit           */
    int s1_e_e_lim;           /* s1 end end limit             */
    int s2_s_e_lim;           /* s2 start end limit           */
    int s2_e_e_lim;           /* s2 end end limit             */
    int flag;                 /* boolean flag                 */
    int level_f;              /* boolean flag                 */
    int level_e;              /* boolean flag                 */
    int level_f2;             /* boolean flag                 */
    int split1;               /* whether can split segment1   */
    int split2;               /* whether can split segment1   */
    int total_contacts;       /* total number of contacts     */
    bool_t ss_only;           /* whether to use ss only       */
    Asd_Seg_Cont s_conts;     /* segment contacts             */
    Asd_Domain *max_domain_m; /* most likely multi domain     */
    Asd_Contact_List *c_list; /* list of contacts             */

    ss_only = FALSE;
    increment = 1;

    c_list = asd_total_contacts(d_to_split, ss, ss_only, contact_info,
                                contact_rev_info, bn, &total_contacts);

    s1_s_e_lim = d_to_focus->range.s1_s_max;
    s1_e_e_lim = d_to_focus->range.s1_e_max;
    s2_s_e_lim = d_to_focus->range.s2_s_max;
    s2_e_e_lim = d_to_focus->range.s2_e_max;

    s1_s_s_lim = d_to_focus->range.s1_s_min;
    s1_e_s_lim = d_to_focus->range.s1_e_min;
    s2_s_s_lim = d_to_focus->range.s2_s_min;
    s2_e_s_lim = d_to_focus->range.s2_e_min;

    split1 = 1;
    split2 = 1;
    if (d_to_split->start1 == start_chain) {
        if ((d_to_split->end1 - d_to_split->start1 + 1) < (params.MIN_SEGMENT_SIZE_MID
             + params.MIN_SEGMENT_SIZE_END)) {
            split1 = 0;
        } /*if*/
    } else {
        if ((d_to_split->end1 - d_to_split->start1 + 1) < (params.MIN_SEGMENT_SIZE_MID
             * 2)) {
            split1 = 0;
        } /*if*/
    } /*if*/
    if (d_to_split->end2 == end_chain) {
        if ((d_to_split->end2 - d_to_split->start2 + 1) < (params.MIN_SEGMENT_SIZE_MID
            + params.MIN_SEGMENT_SIZE_END)) {
            split2 = 0;
        } /*if*/
    } else {
        if ((d_to_split->end2 - d_to_split->start2 + 1) < (params.MIN_SEGMENT_SIZE_MID
            * 2)) {
            split2 = 0;
        } /*if*/
    } /*if*/

    max_domain_m = (Asd_Domain *) asm_malloc(sizeof(Asd_Domain));
    max_domain_m->value = 0.0;
    max_domain_m->type = 2;

    if (split1 == 0) {
        s1_s = s1_s_s_lim;
        s1_e = s1_e_s_lim;
        s2_s = s2_s_s_lim;
        if (s2_e_s_lim == end_chain) {
            if ((s2_e_s_lim - s2_s_s_lim + 1) < params.MIN_SEGMENT_SIZE_END) {
                s2_e_s_lim = s2_s_s_lim + params.MIN_SEGMENT_SIZE_END - 1;
            } /*if*/
        } else {
            if ((s2_e_s_lim - s2_s_s_lim + 1) < params.MIN_SEGMENT_SIZE_MID) {
                s2_e_s_lim = s2_s_s_lim + params.MIN_SEGMENT_SIZE_MID - 1;
            } /*if*/
        } /*if*/
        s2_e = s2_e_s_lim;

        s1_sn = s1_s;
        s1_en = s1_e;
        s2_sn = s2_s;
        s2_en = s2_e;

        while (s2_sn <= s2_s_e_lim &&
              (s2_sn <= (d_to_split->end2 + 1 - params.MIN_SEGMENT_SIZE_END) &&
               d_to_split->end2 == end_chain) ||
              (s2_sn <= (d_to_split->end2 + 1 - params.MIN_SEGMENT_SIZE_MID) &&
               d_to_split->end2 != end_chain)) {

            flag = asd_calc_res_from_scratch_sep_m(contact_info,
                                            contact_rev_info,
                                            d_to_split, max_domain_m,
                                            &s_conts, ss_only, ss,
                                            start_chain, end_chain,
                                            s1_sn, s1_en, s2_sn, s2_en, 1, bn,
                                            c_list);
   
            s1_s = s1_sn;
            s1_e = s1_en;
            s2_s = s2_sn;
            s2_e = s2_en;
            while (s2_en <= s2_e_e_lim && flag !=2) {
                flag = asd_calc_res_given_sep_m(contact_info, contact_rev_info,
                                       d_to_split, max_domain_m, &s_conts,
                                       ss_only, ss, start_chain, end_chain,
                                       s1_sn,  s1_en,  s2_sn,  s2_en,
                                       s1_s,   s1_e,   s2_s,   s2_e, 1,
                                       total_contacts, bn, c_list, increment);
                if (flag == 0) {
                    s2_e = s2_en;
                } /*if*/
                if (s2_en == d_to_split->end2) {
                    s2_en = s2_en + increment;
                } else {
                    s2_en = s2_en + increment;
                    if (s2_en > (d_to_split->end2 -
                        params.MIN_NO_CONTACT_CUTOFF_END) &&
                        d_to_split->end2 == end_chain) {
                        s2_en = d_to_split->end2;
                    } else if (s2_en >
                              (d_to_split->end2 - params.MIN_NO_CONTACT_CUTOFF_MID) &&
                               d_to_split->end2 != end_chain) {
                        s2_en = d_to_split->end2;
                    } /*if*/
                } /*if*/
                if (s2_sn == d_to_split->start2 && s2_en == d_to_split->end2) {
                    flag = 2;
                } /*if*/
            } /*while*/
            s2_s = s2_sn;
            if (s2_sn == d_to_split->start2) {
                s2_sn = d_to_split->start2 + params.MIN_NO_CONTACT_CUTOFF_MID;
            } else {
                s2_sn = s2_sn + increment;
            } /*if*/
            s2_en = d_to_split->end2;
        } /*while*/

    } else if (split2 == 0) {
        s1_s = s1_s_s_lim;
        if (s1_s == start_chain) {
            if ((s1_e_s_lim - s1_s_s_lim + 1) < params.MIN_SEGMENT_SIZE_END) {
                s1_e_s_lim = s1_s_s_lim + params.MIN_SEGMENT_SIZE_END - 1;
            } /*if*/
        } else {
            if ((s1_e_s_lim - s1_s_s_lim + 1) < params.MIN_SEGMENT_SIZE_MID) {
                s1_e_s_lim = s1_s_s_lim + params.MIN_SEGMENT_SIZE_MID - 1;
            } /*if*/
        } /*if*/
        s1_e = s1_e_s_lim;
        s2_s = s2_s_s_lim;
        s2_e = s2_e_s_lim;

        s1_sn = s1_s;
        s1_en = s1_e;
        s2_sn = s2_s;
        s2_en = s2_e;

        while (s1_sn <= s1_s_e_lim &&
               s1_sn <= (d_to_split->end1 + 1 - params.MIN_SEGMENT_SIZE_MID)) {
            flag = asd_calc_res_from_scratch_sep_m(contact_info,
                                            contact_rev_info,
                                            d_to_split, max_domain_m,
                                            &s_conts, ss_only, ss,
                                            start_chain, end_chain,
                                            s1_sn, s1_en, s2_sn, s2_en, 2, bn,
                                            c_list);

            s1_s = s1_sn;
            s1_e = s1_en;
            s2_s = s2_sn;
            s2_e = s2_en;
            while (s1_en <= s1_e_e_lim && flag !=2) {
                flag = asd_calc_res_given_sep_2m(contact_info, contact_rev_info,                                       d_to_split, max_domain_m, &s_conts,
                                       ss_only, ss, start_chain, end_chain,
                                       s1_sn,  s1_en,  s2_sn,  s2_en,
                                       s1_s,   s1_e,   s2_s,   s2_e, 2,
                                       total_contacts, bn, c_list, increment);
                if (flag == 0) {
                    s1_e = s1_en;
                } /*if*/
                if (s1_en == d_to_split->end1) {
                    s1_en = s1_en + increment;
                } else {
                    s1_en = s1_en + increment;
                    if (s1_en > (d_to_split->end1 -
                        params.MIN_NO_CONTACT_CUTOFF_MID)) {
                        s1_en = d_to_split->end1;
                    } /*if*/
                } /*if*/
                if (s1_sn == d_to_split->start1 && s1_en == d_to_split->end1) {
                    flag = 2;
                } /*if*/
            } /*while*/
            if (s1_sn == d_to_split->start1 &&
                        d_to_split->start1 == start_chain) {
                s1_sn = d_to_split->start1 + params.MIN_NO_CONTACT_CUTOFF_END;
            } else if (s1_sn == d_to_split->start1 &&
                        d_to_split->start1 != start_chain) {
                s1_sn = d_to_split->start1 + params.MIN_NO_CONTACT_CUTOFF_MID;
            } else {
                s1_sn = s1_sn + increment;
            } /*if*/
            s1_en = d_to_split->end1;
        } /*while*/

    } else {
        s1_s = s1_s_s_lim;
        if (s1_s == start_chain) {
            if ((s1_e_s_lim - s1_s_s_lim + 1) < params.MIN_SEGMENT_SIZE_END) {
                s1_e_s_lim = s1_s_s_lim + params.MIN_SEGMENT_SIZE_END - 1;
            } /*if*/
        } else {
            if ((s1_e_s_lim - s1_s_s_lim + 1) < params.MIN_SEGMENT_SIZE_MID) {
                s1_e_s_lim = s1_s_s_lim + params.MIN_SEGMENT_SIZE_MID - 1;
            } /*if*/
        } /*if*/
        s1_e = s1_e_s_lim;

        s2_s = s2_s_s_lim;
        if (s2_e_s_lim == end_chain) {
            if ((s2_e_s_lim - s2_s_s_lim + 1) < params.MIN_SEGMENT_SIZE_END) {
                s2_e_s_lim = s2_s_s_lim + params.MIN_SEGMENT_SIZE_END - 1;
            } /*if*/
        } else {
            if ((s2_e_s_lim - s2_s_s_lim + 1) < params.MIN_SEGMENT_SIZE_MID) {
                s2_e_s_lim = s2_s_s_lim + params.MIN_SEGMENT_SIZE_MID - 1;
            } /*if*/
        } /*if*/
        s2_e = s2_e_s_lim;

        s1_sn = s1_s;
        s1_en = s1_e;
        s2_sn = s2_s;
        s2_en = s2_e;
        level_e = 0;
        while (s1_sn <= s1_s_e_lim &&
               s1_sn <= (d_to_split->end1 + 1 - params.MIN_SEGMENT_SIZE_MID)) {
            level_f = asd_check_seg_a(c_list, ss_only, ss, d_to_split, s1_sn,
                                      params.MIN_SEGMENT_SIZE_MID, start_chain, 1);
            while (s1_en <= s1_e_e_lim &&
                   s1_en <= d_to_split->end1 && level_f == 0 && level_e == 0) {
                while (s2_sn <= s2_s_e_lim &&
                      (s2_sn <= (d_to_split->end2 + 1 - params.MIN_SEGMENT_SIZE_END)
                       && d_to_split->end2 == end_chain) ||
                      (s2_sn <= (d_to_split->end2 + 1 - params.MIN_SEGMENT_SIZE_MID) &&                       d_to_split->end2 != end_chain)) {
                    flag = asd_calc_res_from_scratch_sep_m(contact_info,
                                        contact_rev_info, d_to_split,
                                        max_domain_m,
                                        &s_conts, ss_only, ss,
                                        start_chain, end_chain,
                                        s1_sn, s1_en, s2_sn, s2_en, 3, bn,
                                        c_list);

                    s1_s = s1_sn;
                    s1_e = s1_en;
                    s2_s = s2_sn;
                    s2_e = s2_en;
                    while (s2_en <= s2_e_e_lim && flag !=2) {
                        flag = asd_calc_res_given_sep_m(contact_info,
                                       contact_rev_info, d_to_split,
                                       max_domain_m, &s_conts, ss_only, ss,
                                       start_chain, end_chain,
                                       s1_sn,  s1_en,  s2_sn,  s2_en,
                                       s1_s,   s1_e,   s2_s,   s2_e, 3,
                                       total_contacts, bn, c_list, increment);
                        if (flag == 0) {
                            s2_e = s2_en;
                        } /*if*/
                        if (s2_en == d_to_split->end2) {
                            s2_en = s2_en + increment;
                        } else {
                            s2_en = s2_en + increment;
                            if (s1_sn == d_to_split->start1 &&
                                s1_en == d_to_split->end1) {
                                if (s2_en > (d_to_split->end2 -
                                    params.MIN_NO_CONTACT_CUTOFF_END) &&
                                    d_to_split->end2 == end_chain) {
                                    s2_en = d_to_split->end2;
                                } else if (s2_en >
                                          (d_to_split->end2 -
                                           params.MIN_NO_CONTACT_CUTOFF_MID) &&
                                           d_to_split->end2 != end_chain) {
                                    s2_en = d_to_split->end2;
                                } /*if*/
                            } else {
                                if (s2_en > (d_to_split->end2 -
                                    params.MIN_SEGMENT_SIZE_END) &&
                                    d_to_split->end2 == end_chain) {
                                    s2_en = d_to_split->end2;
                                } else if (s2_en >
                                          (d_to_split->end2 -
                                           params.MIN_SEGMENT_SIZE_MID) &&
                                           d_to_split->end2 != end_chain) {
                                    s2_en = d_to_split->end2;
                                } /*if*/
                            } /*if*/
                        } /*if*/
                        if (s2_sn == d_to_split->start2 &&
                            s2_en == d_to_split->end2 &&
                            s1_sn == d_to_split->start1 &&
                            s1_en == d_to_split->end1) {
                            flag = 2;
                        } /*if*/
                    } /*while*/
                    s2_s = s2_sn;
                    s2_sn = s2_sn + increment;
                    if (s1_sn == d_to_split->start1 &&
                        s1_en == d_to_split->end1) {
                        if (s2_sn < (d_to_split->start2 +
                            params.MIN_NO_CONTACT_CUTOFF_MID)) {
                            s2_sn = d_to_split->start2 +
                                    params.MIN_NO_CONTACT_CUTOFF_MID;
                        } /*if*/
                    } else {
                        if (s2_sn < (d_to_split->start2 +
                                    params.MIN_SEGMENT_SIZE_MID)) {
                            s2_sn = d_to_split->start2 +
                                    params.MIN_SEGMENT_SIZE_MID;
                        } /*if*/
                    } /*if*/
                    s2_en = d_to_split->end2;
                } /*while*/
                s1_e = s1_en;
                if (s1_en == d_to_split->end1) {
                    level_e = 1;
                } /*if*/
                s1_en = s1_en + increment;
                if (s1_en > (d_to_split->end1 - params.MIN_SEGMENT_SIZE_MID)) {
                    s1_en = d_to_split->end1;
                } /*if*/
                s2_sn = s2_s_s_lim;
                s2_en = s2_e_s_lim;
            } /*while*/
            s1_s = s1_sn;
            level_e = 0;
            s1_sn = s1_sn + increment;
            if (d_to_split->start1 == start_chain &&
                s1_sn < (start_chain + params.MIN_SEGMENT_SIZE_END)) {
                s1_sn = start_chain + params.MIN_SEGMENT_SIZE_END;
            } else if (d_to_split->start1 != start_chain &&
                       s1_sn < (d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID)) {
                s1_sn = d_to_split->start1 + params.MIN_SEGMENT_SIZE_MID;
            } /*if*/
            s1_en = d_to_split->end1;
            s2_sn = s2_s_s_lim;
            s2_en = s2_e_s_lim;
        } /*while*/
    } /*if*/

    asd_free_c_list(d_to_split, c_list);
    if (max_domain_m->range.s1_s_min == max_domain_m->range.s1_s_max &&
        max_domain_m->range.s1_e_min == max_domain_m->range.s1_e_max &&
        max_domain_m->range.s2_s_min == max_domain_m->range.s2_s_max &&
        max_domain_m->range.s2_e_min == max_domain_m->range.s2_e_max) {
        d_to_focus->start1 = max_domain_m->range.s1_s_min;
        d_to_focus->end1 = max_domain_m->range.s1_e_min;
        d_to_focus->start2 = max_domain_m->range.s2_s_min;
        d_to_focus->end2 = max_domain_m->range.s2_e_min;
    } else if (!d_to_focus->ok) {
        d_to_focus->start1 = (d_to_focus->range.s1_s_min +
                              d_to_focus->range.s1_s_max) / 2;
        d_to_focus->end1 = (d_to_focus->range.s1_e_min +
                            d_to_focus->range.s1_e_max) / 2;
        d_to_focus->start2 = (d_to_focus->range.s2_s_min +
                              d_to_focus->range.s2_s_max) / 2;
        d_to_focus->end2 = (d_to_focus->range.s2_e_min +
                            d_to_focus->range.s2_e_max) / 2;
    } else {
        fprintf(stdout, "An ERROR condition has been detected\n");
        fprintf(stdout, "This is probably the result of either,\n");
        fprintf(stdout, "a) Choosing the ss only split values, normal split\n");
        fprintf(stdout, "   values unwisely\n");
       fprintf(stdout, "b) if a) is ok setting minimum ss percentage to 1.0\n");
        fprintf(stdout, "   should solve the problem\n");
        ase_error_fatal("asd_focus_multi_sep", "could not focus");
    } /*if*/
} /*asd_focus_multi_sep*/
 
/*
 * name
 *    asd_narrow_range
 * purpose
 *    top level routine for focussing rseults
 */
void
asd_narrow_range(Asd_Domain *d_to_focus, Asd_Domain *d_to_split,
                 Asd_Contact_Info **contact_info,
                 Asd_Contact_Info **contact_rev_info,
                 struct brookn *bn, int start_chain, int end_chain,
                 char *ss)
{
    if (d_to_focus->type == 1) {
        asd_focus_single(d_to_focus, d_to_split, contact_info, contact_rev_info,
                         bn, start_chain, end_chain, ss);
    } else if (d_to_focus->type == 2) {
        if (d_to_split->type == 1) {
            asd_focus_multiple_domains(d_to_focus, d_to_split, contact_info,
                                       contact_rev_info, bn, start_chain,
                                       end_chain, ss);
        } else if (d_to_split->type == 2) {
            asd_focus_multi_sep(d_to_focus, d_to_split, contact_info,
                                contact_rev_info, bn, start_chain,
                                end_chain, ss);
        } else {
            ase_error_fatal("asd_narrow_range", "could not get type");
        } /*if*/
    } else {
        ase_error_fatal("asd_narrow_range", "could not get type");
    } /*if*/
} /*asd_narrow_range*/ 
