/******************************************************************************
 The computer software and associated documentation called DOMAK hereinafter
 referred to as the WORK which is more particularly identified and described in
 Appendix A of the file LICENSE.  Conditions and restrictions for use of
 this package are also in this file.

 This routine was written by Asim S. Siddiqui

 The WORK is Copyright (1995) A. S. Siddiqui and G. J. Barton

 All use of the WORK must cite:
 Siddiqui, A. S. and Barton, G. J., "Continuous and Discontinuous Domains: An
 Algorithm for the Automatic Generation of Reliable Protein Domain Definitions" 
 PROTEIN SCIENCE, 4:872-884 (1995).
*****************************************************************************/

/*
 * Name
 *   asd_mds_utils3.c
 * Purpose
 *   to provide a set of utility routines for the make_domans routines
 * Author
 *   Asim Siddiqui
 * SccsId
 *   %W%   %U%   %E%
 */

#include <rdssp.h>
#include <asd_structs.h>
#include <asm_mop.h>
#include <local.h>
#include <ase_error.h>
#include <asd_value.h>
#include <asd_make_domains.h>
#include <asd_make_domains_2.h>
#include <asd_mds_utils3.h>
#include <asd_utils.h>

extern Asd_Parameters params;

int sc_corr3;                   /* whether segment c is correlated to another
                                * left over part                             */
Asd_Domain d_sc_corr3;     /* sc correlated domain                       */
Asd_Seg_Cont sc_seg_info3; /* segement c and correlated part info        */

/*
 * Name
 *   asd_calc_res_from_scratch_sep_m
 * Purpose
 *   calculates the contacts from scratch for seperated double join case
 */
int
asd_calc_res_from_scratch_sep_m(Asd_Contact_Info **c1,
                            Asd_Contact_Info **c2,
                            Asd_Domain *d_to_split, Asd_Domain *max_domain_m,
                            Asd_Seg_Cont *s_conts, bool_t ss_only, char *ss,
                            int start_chain, int end_chain,
                            int s1_s, int s1_e, int s2_s, int s2_e, int type,
                            struct brookn *bn, Asd_Contact_List *c_list)
{
    int i;                      /* loop counter                          */
    int j;                      /* loop counter                          */
    int size1;                  /* size of segment                       */
    int size2;                  /* size of segment                       */
    int num_contacts;           /* number of contacts for that item      */
    Asd_Contact_Info **cf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info *cf2_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info **cr1_ptr; /* temporary pointer to reverse contacts */
    Asd_Contact_Info *cr2_ptr; /* temporary pointer to reverse contacts */
    Asd_Domain *d_left_over;    /* domain left over                      */
    Asd_Domain d_l_over[2];     /* domain left over                      */
    Asd_Contact_Info **cbf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info **cbr1_ptr; /* temporary pointer to r b     contacts */
    Asd_Contact_Info **contact_info;  /* beta sheet list                 */
    Asd_Contact_Info **contact_rev_info;  /* reverse beta sheet list     */
    Asd_Contact_Info **beta_l;  /* beta sheet list                       */
    Asd_Contact_Info **rev_beta_l;  /* reverse beta sheet list           */

    contact_info = c_list->cl;
    contact_rev_info = c_list->rcl;
    beta_l = c_list->bl;
    rev_beta_l = c_list->rbl;

    s_conts->s1_int = 0;
    s_conts->s2_int = 0;
    s_conts->s1_ext = 0;
    s_conts->s2_ext = 0;
    s_conts->s1_s2_cont = 0;
    s_conts->s1_sht_int = 0;
    s_conts->s2_sht_int = 0;
    s_conts->s1_sht_ext = 0;
    s_conts->s2_sht_ext = 0;
    s_conts->s1_s2_sht_cont = 0;
    sc_corr3 = 0;

/* check everything ok if doing a type 3 ie splitting both */
    if (type == 3 && d_to_split->start1 == s1_s && d_to_split->end1 == s1_e) {
        type = 1;
    } else if (type == 3 && d_to_split->start2 == s2_s &&
               d_to_split->end2 == s2_e) {
        type = 2;
        if (d_to_split->start1 == s1_s) {
            if ((d_to_split->end1 - s1_e) < params.MIN_NO_CONTACT_CUTOFF_MID) {
                return(2);
            } /*if*/
        } else {
            size1 = s1_s - d_to_split->start1;
            if (d_to_split->start1 != start_chain && 
                size1 < params.MIN_NO_CONTACT_CUTOFF_MID) {
                return(2);
            } else if (size1 < params.MIN_NO_CONTACT_CUTOFF_END) {
                return(2);
            } /*if*/
        } /*if*/
    } /*if*/

    if (type == 3) {
        d_left_over = d_l_over;
        if (d_to_split->start1 == s1_s) {
            d_left_over->start1 = s1_e + 1;
            d_left_over->end1 = d_to_split->end1;
        } else {
            d_left_over->start1 = d_to_split->start1;
            d_left_over->end1 =  s1_s - 1;
        } /*if*/
        if (d_to_split->start2 == s2_s) {
            d_left_over[1].start1 = s2_e + 1;
            d_left_over[1].end1 = d_to_split->end2;
        } else {
            d_left_over[1].start1 = d_to_split->start2;
            d_left_over[1].end1 =  s2_s - 1;
        } /*if*/
        size1 = d_left_over->end1 - d_left_over->start1 + 1;
        size2 = d_left_over[1].end1 - d_left_over[1].start1 + 1;
        if ((d_left_over->start1 != d_to_split->start1 &&
             size1 < params.MIN_NO_CONTACT_CUTOFF_MID) ||
            ((d_left_over->start1 == d_to_split->start1 &&
              d_left_over->start1 == start_chain) &&
              size1 < params.MIN_NO_CONTACT_CUTOFF_END) ||
            (d_left_over[1].end1 != d_to_split->end2 &&
             size2 < params.MIN_NO_CONTACT_CUTOFF_MID) ||
            ((d_left_over[1].end1 == d_to_split->end2 &&
              d_left_over[1].end1 == end_chain) &&
              size2 < params.MIN_NO_CONTACT_CUTOFF_END)) {
            if (!asd_check_sc_correlation(contact_info, contact_rev_info,
                              bn, d_left_over, 2, &sc_seg_info3, ss, ss_only,
                              c_list)) {
                return(2);
            } else {
                sc_corr3 = 1;
                d_sc_corr3.start1 = d_left_over->start1;
                d_sc_corr3.end1 = d_left_over->end1;
            } /*if*/
        } /*if*/
    } /*if*/

/* first scan s1 to work out s1_int s1_ext s1_sht_ext s1_sht_int */
    i = s1_s;
    cf1_ptr = (contact_info + s1_s);
    cr1_ptr = (contact_rev_info + s1_s);
    cbf1_ptr = (beta_l + s1_s);
    cbr1_ptr = (rev_beta_l + s1_s);

    while (i <= s1_e) {
        if (!ss_only ||
            (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
/* first analyse forward contacts */
            cf2_ptr = *cf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s1_s && (*cf2_ptr).res_num <= s1_e) {
                    (s_conts->s1_int)++;
                } else {
                    (s_conts->s1_ext)++;
                    if ((*cf2_ptr).res_num >= s2_s && (*cf2_ptr).res_num
                                                                      <= s2_e) {
                        (s_conts->s1_s2_cont)++;
                    } /*if*/
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* first analyse forward beta contacts */
            cf2_ptr = *cbf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s1_s && (*cf2_ptr).res_num <= s1_e) {
                    (s_conts->s1_sht_int)++;
                } else {
                    (s_conts->s1_sht_ext)++;
                    if ((*cf2_ptr).res_num >= s2_s && (*cf2_ptr).res_num
                                                                      <= s2_e) {
                        (s_conts->s1_s2_sht_cont)++;
                    } /*if*/
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* now do reverse contacts - no need to count internal ones again */
            cr2_ptr = *cr1_ptr;
            num_contacts = (*cr2_ptr).res_num;
            cr2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cr2_ptr).res_num < s1_s || (*cr2_ptr).res_num > s1_e) {
                    (s_conts->s1_ext)++;
                } /*if*/
                j++;
                cr2_ptr++;
            } /*while*/
/* now do reverse beta contacts - no need to count internal ones again */
            cr2_ptr = *cbr1_ptr;
            num_contacts = (*cr2_ptr).res_num;
            cr2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cr2_ptr).res_num < s1_s || (*cr2_ptr).res_num > s1_e) {
                    (s_conts->s1_sht_ext)++;
                } /*if*/
                j++;
                cr2_ptr++;
            } /*while*/
        } /*if*/
        i++;
        cf1_ptr = (cf1_ptr + 1);
        cr1_ptr = (cr1_ptr + 1);
        cbf1_ptr = (cbf1_ptr + 1);
        cbr1_ptr = (cbr1_ptr + 1);
    } /*while*/

/* now scan s2 to work out s2_int s2_ext s2_sht_ext s2_sht_int */
    i = s2_s;
    cf1_ptr = (contact_info + s2_s);
    cr1_ptr = (contact_rev_info + s2_s);
    cbf1_ptr = (beta_l + s2_s);
    cbr1_ptr = (rev_beta_l + s2_s);
    while (i <= s2_e) {
        if (!ss_only ||
            (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
/* first analyse forward contacts */
            cf2_ptr = *cf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s2_s && (*cf2_ptr).res_num <= s2_e) {
                    (s_conts->s2_int)++;
                } else {
                    (s_conts->s2_ext)++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* first analyse forward beta contacts */
            cf2_ptr = *cbf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s2_s && (*cf2_ptr).res_num <= s2_e) {
                    (s_conts->s2_sht_int)++;
                } else {
                    (s_conts->s2_sht_ext)++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* now do reverse contacts - no need to count internal ones again */
            cr2_ptr = *cr1_ptr;
            num_contacts = (*cr2_ptr).res_num;
            cr2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cr2_ptr).res_num < s2_s || (*cr2_ptr).res_num > s2_e) {
                    (s_conts->s2_ext)++;
                } /*if*/
                j++;
                cr2_ptr++;
            } /*while*/
/* now do reverse beta contacts - no need to count internal ones again */
            cr2_ptr = *cbr1_ptr;
            num_contacts = (*cr2_ptr).res_num;
            cr2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cr2_ptr).res_num < s2_s || (*cr2_ptr).res_num > s2_e) {
                    (s_conts->s2_sht_ext)++;
                } /*if*/
                j++;
                cr2_ptr++;
            } /*while*/
        } /*if*/
        cf1_ptr = (cf1_ptr + 1);
        cr1_ptr = (cr1_ptr + 1);
        cbf1_ptr = (cbf1_ptr + 1);
        cbr1_ptr = (cbr1_ptr + 1);
        i++;
    } /*while*/

    return(0);
} /*asd_calc_res_from_scratch_sep_m*/

/* 
 * Name
 *    asd_calc_res_given_sep_m
 * Purpose
 *    to calculate the residues given change in s2_e ONLY
 *    ASSUMES s1 is before s2
 *    must have called asd_calc_res_from_scratch_sep_m before calling
 */
int
asd_calc_res_given_sep_m(Asd_Contact_Info **c1,
                         Asd_Contact_Info **c2,
                         Asd_Domain *d_to_split, Asd_Domain *max_domain_m,
                         Asd_Seg_Cont *s_conts, bool_t ss_only, char *ss,
                         int start_chain, int end_chain,
                         int s1_sn, int s1_en, int s2_sn, int s2_en,
                         int s1_s,  int s1_e,  int s2_s,  int s2_e, int type,
                         int total_contacts, struct brookn *bn,
                         Asd_Contact_List *c_list, int increment)
{
    int i;                      /* loop counter                          */
    int j;                      /* loop counter                          */
    int size1;                  /* size of segment                       */
    int size2;                  /* size of segment                       */
    int num_contacts;           /* number of contacts for that item      */
    int int_1;                  /* number of internal contacts           */
    int int_2;                  /* number of internal contacts           */
    float ext_no;               /* number of external contacts           */
    int ext_sht;                /* external sheet contacts               */
    float value;                /* temporary value                       */
    Asd_Domain *d_left_over;    /* domain left over                      */
    Asd_Domain d_l_over[2];     /* domain left over                      */
    Asd_Contact_Info **cf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info *cf2_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info **cr1_ptr; /* temporary pointer to reverse contacts */
    Asd_Contact_Info *cr2_ptr; /* temporary pointer to reverse contacts */
    Asd_Contact_Info **cbf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info **cbr1_ptr; /* temporary pointer to r b     contacts */
    Asd_Contact_Info **contact_info;  /* beta sheet list                 */
    Asd_Contact_Info **contact_rev_info;  /* reverse beta sheet list     */
    Asd_Contact_Info **beta_l;  /* beta sheet list                       */
    Asd_Contact_Info **rev_beta_l;  /* reverse beta sheet list           */

    contact_info = c_list->cl;
    contact_rev_info = c_list->rcl;
    beta_l = c_list->bl;
    rev_beta_l = c_list->rbl;

    if (s1_sn != s1_s || s1_en != s1_e || s2_sn != s2_s || s2_en < s2_e) {
        ase_error_fatal("asd_calc_res_given_m", "function called incorrectly");
    } /*if*/

    if ((s2_en - s2_sn + s1_en - s1_sn + 2) < params.MIN_DOMAIN_SIZE) {
        return (1);
    } /*if*/

    if (type == 3 && s2_en == d_to_split->end2 && s2_sn == d_to_split->start2) {
        if (s1_sn == d_to_split->start1) {
            if ((d_to_split->end1 - s1_en) < params.MIN_NO_CONTACT_CUTOFF_MID) {
                return(1); 
            } /*if*/
        } else {
            if (d_to_split->start1 == start_chain) {
                if ((s1_sn - d_to_split->start1) < params.MIN_NO_CONTACT_CUTOFF_END) {
                    return(1);
                } /*if*/
            } else {
                if ((s1_sn - d_to_split->start1) < params.MIN_NO_CONTACT_CUTOFF_MID) {
                    return(1);
                } /*if*/
            } /*if*/
        } /*if*/
    } /*if*/ 

/* obviously s1_int and s1_ext s1_int_sht s1_sht_ext stay the same */
/* only need to modify the corresponding s2 variables */
    i = s2_e + 1;
    cf1_ptr = (contact_info + (s2_e + 1));
    cr1_ptr = (contact_rev_info + (s2_e + 1));;
    cbf1_ptr = (beta_l + (s2_e + 1));
    cbr1_ptr = (rev_beta_l + (s2_e + 1));
    while (i <= s2_en) {
        if (!ss_only ||
            (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
/* first analyse forward contacts */
            cf2_ptr = *cf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s2_sn && (*cf2_ptr).res_num <=
                                                                        s2_en) {
                    (s_conts->s2_int)++;
                } else {
                    (s_conts->s2_ext)++;
                } /*if*/
                if (sc_corr3) {
                    if ((*cf2_ptr).res_num > s2_e &&
                        (*cf2_ptr).res_num <= d_to_split->end2) {
                        (sc_seg_info3.s2_int)--;
                    } /*if*/
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* first analyse forward beta contacts */
            cf2_ptr = *cbf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s2_sn && (*cf2_ptr).res_num <=
                                                                        s2_en) {
                    (s_conts->s2_sht_int)++;
                } else {
                    (s_conts->s2_sht_ext)++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* now do reverse contacts - add externals also remove add extra internals
   and remove them from externals */
            cr2_ptr = *cr1_ptr;
            num_contacts = (*cr2_ptr).res_num;
            cr2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cr2_ptr).res_num >= s1_sn && (*cr2_ptr).res_num
                                                                     <= s1_en) {
                    (s_conts->s1_s2_cont)++;
                } /*if*/
                if ((*cr2_ptr).res_num < s2_sn || (*cr2_ptr).res_num > s2_en) {
                    (s_conts->s2_ext)++;
                } else if ((*cr2_ptr).res_num >= s2_sn  &&
                           (*cr2_ptr).res_num <= s2_e) {
                    (s_conts->s2_int)++;
                    (s_conts->s2_ext)--;
                } /*if*/
                if (sc_corr3) {
                    if ((*cr2_ptr).res_num >= d_sc_corr3.start1 &&
                        (*cr2_ptr).res_num <= d_sc_corr3.end1) {
                        (sc_seg_info3.s1_s2_cont)--;
                    } /*if*/
                } /*if*/
                j++;
                cr2_ptr++;
            } /*while*/
/* now do reverse beta contacts - add externals also remove add extra internals
   and remove them from externals */
            cr2_ptr = *cbr1_ptr;
            num_contacts = (*cr2_ptr).res_num;
            cr2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cr2_ptr).res_num >= s1_sn && (*cr2_ptr).res_num <=
                                                                        s1_en) {
                    (s_conts->s1_s2_sht_cont)++;
                } /*if*/
                if ((*cr2_ptr).res_num < s2_sn || (*cr2_ptr).res_num > s2_en) {
                    (s_conts->s2_sht_ext)++;
                } else if ((*cr2_ptr).res_num >= s2_sn  &&
                           (*cr2_ptr).res_num <= s2_e) {
                    (s_conts->s2_sht_int)++;
                    (s_conts->s2_sht_ext)--;
                } /*if*/
                if (sc_corr3) {
                    if ((*cr2_ptr).res_num >= d_sc_corr3.start1 &&
                        (*cr2_ptr).res_num <= d_sc_corr3.end1) {
                        (sc_seg_info3.s1_s2_sht_cont)--;
                    } /*if*/
                } /*if*/
                j++;
                cr2_ptr++;
            } /*while*/
        } /*if*/
        cf1_ptr = (cf1_ptr + 1);
        cr1_ptr = (cr1_ptr + 1);
        cbf1_ptr = (cbf1_ptr + 1);
        cbr1_ptr = (cbr1_ptr + 1);
        i++;
    } /*while*/
/* now analyse rest*/
/* if no intersegment contacts then parts uncorrelated */
    if (!s_conts->s1_s2_cont) {
        return(0);
    } /*if*/
/* check parts are correlated if not return as false */
    value = asd_calc_c_value(s_conts->s1_int, s_conts->s2_int,
                             s_conts->s1_s2_cont, s_conts->s1_s2_sht_cont,
                             s1_sn, s1_en, 0, 0, s2_sn, s2_en, 0, 0);
    if (! ((ss_only && value < params.MIN_PEAK_SS_ONLY_DC) ||
          (!ss_only && value < params.MIN_PEAK_DC))) {
        return(0);
    } /*if*/
    if (sc_corr3) {
        if (((d_to_split->end2 - s2_en) < params.MIN_SEGMENT_SIZE_END &&
              d_to_split->end2 == end_chain) ||
            ((d_to_split->end2 - s2_en) < params.MIN_SEGMENT_SIZE_MID &&
              d_to_split->end2 != end_chain)) {
            return(0);
        } /*if*/
        value = asd_calc_c_value(sc_seg_info3.s1_int, sc_seg_info3.s2_int,
                                 sc_seg_info3.s1_s2_cont,
                                 sc_seg_info3.s1_s2_sht_cont,
                                 d_sc_corr3.start1, d_sc_corr3.end1, 0, 0,
                                 (s2_en + 1), d_to_split->end2, 0, 0);
        if ((ss_only && value >= params.MIN_PEAK_SS_ONLY_DC) ||
           (!ss_only && value >= params.MIN_PEAK_DC)) {
            return(0);
        } /*if*/
/* check case where sc has become smaller than params.MIN_NO_CONTACT_CUTOFF */
    } else if ((((d_to_split->end2 - s2_en) < params.MIN_NO_CONTACT_CUTOFF_END &&
                 d_to_split->end2 == end_chain) ||
               ((d_to_split->end2 - s2_en) < params.MIN_NO_CONTACT_CUTOFF_MID &&
                 d_to_split->end2 != end_chain)) && s2_en != d_to_split->end2 &&
                 type == 3 && !(s1_s == d_to_split->start1 &&
                                s1_e == d_to_split->end1)) {
        d_left_over = d_l_over;
        if (d_to_split->start1 == s1_sn) {
            d_left_over->start1 = s1_en + 1;
            d_left_over->end1 = d_to_split->end1;
        } else {
            d_left_over->start1 = d_to_split->start1;
            d_left_over->end1 = s1_sn - 1;
        } /*if*/
        d_left_over[1].start1 = s2_sn + 1;
        d_left_over[1].end1 = d_to_split->end2;
        if (asd_check_sc_correlation(contact_info, contact_rev_info,
                          bn, d_left_over, 2, &sc_seg_info3, ss, ss_only,
                          c_list)) {    
            sc_corr3 = 1;
            d_sc_corr3.start1 = d_left_over->start1;
            d_sc_corr3.end1 = d_left_over->end1;
        } else {
            return(0);
        } /*if*/
    } /*if*/

/* now work out value of split */
    int_1 = s_conts->s1_int + s_conts->s2_int + s_conts->s1_s2_sht_cont;
    ext_no = s_conts->s2_ext + s_conts->s1_ext - (2 * s_conts->s1_s2_cont);
    int_2 = total_contacts - int_1 - ext_no;
    ext_sht = s_conts->s2_sht_ext + s_conts->s1_sht_ext -
                              (2 * s_conts->s1_s2_sht_cont);

    value = asd_calc_d_value(int_1, int_2, ext_no, ext_sht, d_to_split,
                             s1_sn, s1_en, s2_sn, s2_en);

    if (value > max_domain_m->value) {
        if ((ss_only && value >= params.MIN_PEAK_SS_ONLY_DC) ||
           (!ss_only && value >= params.MIN_PEAK_DC) || 1) {
            max_domain_m->value = value;
            if (increment == 1) {
                max_domain_m->range.s1_s_min = s1_sn;
                max_domain_m->range.s1_e_min = s1_en;
                max_domain_m->range.s2_s_min = s2_sn;
                max_domain_m->range.s2_e_min = s2_en;
                max_domain_m->range.s1_s_max = s1_sn;
                max_domain_m->range.s1_e_max = s1_en;
                max_domain_m->range.s2_s_max = s2_sn;
                max_domain_m->range.s2_e_max = s2_en;
            } else {
                max_domain_m->range.s1_s_min = s1_sn - increment + 1;
                max_domain_m->range.s1_s_max = s1_sn + increment - 1;
                if (max_domain_m->range.s1_s_min < d_to_split->start1 ||
                    s1_sn == d_to_split->start1) {
                    max_domain_m->range.s1_s_min = d_to_split->start1;
                    max_domain_m->range.s1_s_max = d_to_split->start1;
                } /*if*/

                max_domain_m->range.s1_e_min = s1_en - increment + 1;
                max_domain_m->range.s1_e_max = s1_en + increment - 1;
                if (max_domain_m->range.s1_e_max > d_to_split->end1 ||
                    s1_en == d_to_split->end1) {
                    max_domain_m->range.s1_e_min = d_to_split->end1;
                    max_domain_m->range.s1_e_max = d_to_split->end1;
                } /*if*/

                max_domain_m->range.s2_s_min = s2_sn - increment + 1;
                max_domain_m->range.s2_s_max = s2_sn + increment - 1;
                if (max_domain_m->range.s2_s_min < d_to_split->start2 ||
                    s2_sn == d_to_split->start2) {
                    max_domain_m->range.s2_s_min = d_to_split->start2;
                    max_domain_m->range.s2_s_max = d_to_split->start2;
                } /*if*/

                max_domain_m->range.s2_e_min = s2_en - increment + 1;
                max_domain_m->range.s2_e_max = s2_en + increment - 1;
                if (max_domain_m->range.s2_e_max > d_to_split->end2 ||
                    s2_en == d_to_split->end2) {
                    max_domain_m->range.s2_e_max = d_to_split->end2;
                    max_domain_m->range.s2_e_min = d_to_split->end2;
                } /*if*/
            } /*if*/

            if (ss_only) {
                if (s1_sn != d_to_split->start1) {
                    i = max_domain_m->range.s1_s_max - 1;
                    i++;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    max_domain_m->range.s1_s_max = i;

                    i = max_domain_m->range.s1_s_min - 1;
                    i--;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i > (d_to_split->start1 - 1)) {
                        i--;
                    } /*while*/
                    i++;
                    max_domain_m->range.s1_s_min = i;
                } /*if*/

                if (s1_en != d_to_split->end1) {
                    i = max_domain_m->range.s1_e_max - 1;
                    i++;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    max_domain_m->range.s1_e_max = i;

                    i = max_domain_m->range.s1_e_min - 1;
                    i--;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i > (d_to_split->start1 - 1)) {
                        i--;
                    } /*while*/
                    i++;
                    max_domain_m->range.s1_e_min = i;
                } /*if*/


                if (s2_sn != d_to_split->start2) {
                    i = max_domain_m->range.s2_s_max - 1;
                    i++;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    max_domain_m->range.s2_s_max = i;

                    i = max_domain_m->range.s2_s_min - 1;
                    i--;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i > (d_to_split->start1 - 1)) {
                        i--;
                    } /*while*/
                    i++;
                    max_domain_m->range.s2_s_min = i;
                } /*if*/

                if (s2_en != d_to_split->end2) {
                    i = max_domain_m->range.s2_e_max - 1;
                    i++;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    max_domain_m->range.s2_e_max = i;

                    i = max_domain_m->range.s2_e_min - 1;
                    i--;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i > (d_to_split->start1 - 1)) {
                        i--;
                    } /*while*/
                    i++;
                    max_domain_m->range.s2_e_min = i;
               } /*if*/
            } /*if*/
        } /*if*/
    } /*if*/

    return(0);
} /*asd_calc_res_given_sep_m*/

/* 
 * Name
 *    asd_calc_res_given_sep_2m
 * Purpose
 *    to calculate the residues given change in s1_e ONLY
 *    ASSUMES s1 is before s2
 *    must have called asd_calc_res_from_scratch_sep_m before calling
 *    for type 2 only
 */
int
asd_calc_res_given_sep_2m(Asd_Contact_Info **c1,
                        Asd_Contact_Info **c2,
                        Asd_Domain *d_to_split, Asd_Domain *max_domain_m,
                        Asd_Seg_Cont *s_conts, bool_t ss_only, char *ss,
                        int start_chain, int end_chain,
                        int s1_sn, int s1_en, int s2_sn, int s2_en,
                        int s1_s,  int s1_e,  int s2_s,  int s2_e, int type,
                        int total_contacts, struct brookn *bn,
                        Asd_Contact_List *c_list, int increment)
{
    int i;                      /* loop counter                          */
    int j;                      /* loop counter                          */
    int size1;                  /* size of segment                       */
    int size2;                  /* size of segment                       */
    int num_contacts;           /* number of contacts for that item      */
    int int_1;                  /* number of internal contacts           */
    int int_2;                  /* number of internal contacts           */
    float ext_no;               /* number of external contacts           */
    int ext_sht;                /* external sheet contacts               */
    float value;                /* temporary value                       */
    Asd_Domain *d_left_over;    /* domain left over                      */
    Asd_Contact_Info **cf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info *cf2_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info **cr1_ptr; /* temporary pointer to reverse contacts */
    Asd_Contact_Info *cr2_ptr; /* temporary pointer to reverse contacts */
    Asd_Contact_Info **cbf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info **cbr1_ptr; /* temporary pointer to r b     contacts */
    Asd_Contact_Info **contact_info;  /* beta sheet list                 */
    Asd_Contact_Info **contact_rev_info;  /* reverse beta sheet list     */
    Asd_Contact_Info **beta_l;  /* beta sheet list                       */
    Asd_Contact_Info **rev_beta_l;  /* reverse beta sheet list           */

    contact_info = c_list->cl;
    contact_rev_info = c_list->rcl;
    beta_l = c_list->bl;
    rev_beta_l = c_list->rbl;

    if (s1_sn != s1_s || s2_en != s2_e || s2_sn != s2_s || s1_en < s1_e) {
        ase_error_fatal("asd_calc_res_given_m", "function called incorrectly");
    } /*if*/

    if ((s2_en - s2_sn + s1_en - s1_sn + 2) < params.MIN_DOMAIN_SIZE) {
        return (1);
    } /*if*/

/* obviously s2_int and s2_ext s2_int_sht s2_sht_ext stay the same */
/* only need to modify the corresponding s1 variables */
    i = s1_e + 1;
    cf1_ptr = (contact_info + (s1_e + 1));
    cr1_ptr = (contact_rev_info + (s1_e + 1));;
    cbf1_ptr = (beta_l + (s1_e + 1));
    cbr1_ptr = (rev_beta_l + (s1_e + 1));
    while (i <= s1_en) {
        if (!ss_only ||
            (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
/* first analyse forward contacts */
            cf2_ptr = *cf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s1_sn && (*cf2_ptr).res_num <=
                                                                        s1_en) {
                    (s_conts->s1_int)++;
                } else {
                    (s_conts->s1_ext)++;
                } /*if*/
                if ((*cf2_ptr).res_num >= s2_sn && (*cf2_ptr).res_num
                                                                 <= s2_en) {
                    (s_conts->s1_s2_cont)++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* first analyse forward beta contacts */
            cf2_ptr = *cbf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s1_sn && (*cf2_ptr).res_num <=
                                                                        s1_en) {
                    (s_conts->s1_sht_int)++;
                } else {
                    (s_conts->s1_sht_ext)++;
                } /*if*/
                if ((*cf2_ptr).res_num >= s2_sn && (*cf2_ptr).res_num
                                                                 <= s2_en) {
                    (s_conts->s1_s2_sht_cont)++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* now do reverse contacts - add externals also remove add extra internals
   and remove them from externals */
            cr2_ptr = *cr1_ptr;
            num_contacts = (*cr2_ptr).res_num;
            cr2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cr2_ptr).res_num < s1_sn || (*cr2_ptr).res_num > s1_en) {
                    (s_conts->s1_ext)++;
                } else if ((*cr2_ptr).res_num >= s1_sn  &&
                           (*cr2_ptr).res_num <= s1_e) {
                    (s_conts->s1_int)++;
                    (s_conts->s1_ext)--;
                } /*if*/
                j++;
                cr2_ptr++;
            } /*while*/
/* now do reverse beta contacts - add externals also remove add extra internals
   and remove them from externals */
            cr2_ptr = *cbr1_ptr;
            num_contacts = (*cr2_ptr).res_num;
            cr2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cr2_ptr).res_num < s1_sn || (*cr2_ptr).res_num > s1_en) {
                    (s_conts->s1_sht_ext)++;
                } else if ((*cr2_ptr).res_num >= s1_sn  &&
                           (*cr2_ptr).res_num <= s1_e) {
                    (s_conts->s1_sht_int)++;
                    (s_conts->s1_sht_ext)--;
                } /*if*/
                j++;
                cr2_ptr++;
            } /*while*/
        } /*if*/
        cf1_ptr = (cf1_ptr + 1);
        cr1_ptr = (cr1_ptr + 1);
        cbf1_ptr = (cbf1_ptr + 1);
        cbr1_ptr = (cbr1_ptr + 1);
        i++;
    } /*while*/
/* now analyse rest*/
/* if no intersegment contacts then parts uncorrelated */
    if (!s_conts->s1_s2_cont) {
        return(0);
    } /*if*/
/* check parts are correlated if not return as false */
    value = asd_calc_c_value(s_conts->s1_int, s_conts->s2_int,
                             s_conts->s1_s2_cont, s_conts->s1_s2_sht_cont,
                             s1_sn, s1_en, 0, 0, s2_sn, s2_en, 0, 0);

    if (! ((ss_only && value < params.MIN_PEAK_SS_ONLY_DC) ||
          (!ss_only && value < params.MIN_PEAK_DC))) {
        return(0);
    } /*if*/

/* now work out value of split */
    int_1 = s_conts->s1_int + s_conts->s2_int + s_conts->s1_s2_sht_cont;
    ext_no = s_conts->s2_ext + s_conts->s1_ext - (2 * s_conts->s1_s2_cont);
    int_2 = total_contacts - int_1 - ext_no;
    ext_sht = s_conts->s2_sht_ext + s_conts->s1_sht_ext -
                              (2 * s_conts->s1_s2_sht_cont);

    value = asd_calc_d_value(int_1, int_2, ext_no, ext_sht, d_to_split,
                             s1_sn, s1_en, s2_sn, s2_en);
    if (value > max_domain_m->value) {
        if ((ss_only && value >= params.MIN_PEAK_SS_ONLY_DC) ||
           (!ss_only && value >= params.MIN_PEAK_DC) || 1) {
            max_domain_m->value = value;
            if (increment == 1) {
                max_domain_m->range.s1_s_min = s1_sn;
                max_domain_m->range.s1_e_min = s1_en;
                max_domain_m->range.s2_s_min = s2_sn;
                max_domain_m->range.s2_e_min = s2_en;
                max_domain_m->range.s1_s_max = s1_sn;
                max_domain_m->range.s1_e_max = s1_en;
                max_domain_m->range.s2_s_max = s2_sn;
                max_domain_m->range.s2_e_max = s2_en;
            } else {
                max_domain_m->range.s1_s_min = s1_sn - increment + 1;
                max_domain_m->range.s1_s_max = s1_sn + increment - 1;
                if (max_domain_m->range.s1_s_min < d_to_split->start1 ||
                    s1_sn == d_to_split->start1) {
                    max_domain_m->range.s1_s_min = d_to_split->start1;
                    max_domain_m->range.s1_s_max = d_to_split->start1;
                } /*if*/

                max_domain_m->range.s1_e_min = s1_en - increment + 1;
                max_domain_m->range.s1_e_max = s1_en + increment - 1;
                if (max_domain_m->range.s1_e_max > d_to_split->end1 ||
                    s1_en == d_to_split->end1) {
                    max_domain_m->range.s1_e_min = d_to_split->end1;
                    max_domain_m->range.s1_e_max = d_to_split->end1;
                } /*if*/

                max_domain_m->range.s2_s_min = s2_sn - increment + 1;
                max_domain_m->range.s2_s_max = s2_sn + increment - 1;
                if (max_domain_m->range.s2_s_min < d_to_split->start2 ||
                    s2_sn == d_to_split->start2) {
                    max_domain_m->range.s2_s_min = d_to_split->start2;
                    max_domain_m->range.s2_s_max = d_to_split->start2;
                } /*if*/

                max_domain_m->range.s2_e_min = s2_en - increment + 1;
                max_domain_m->range.s2_e_max = s2_en + increment - 1;
                if (max_domain_m->range.s2_e_max > d_to_split->end2 ||
                    s2_en == d_to_split->end2) {
                    max_domain_m->range.s2_e_max = d_to_split->end2;
                    max_domain_m->range.s2_e_min = d_to_split->end2;
                } /*if*/
            } /*if*/

            if (ss_only) {
                if (s1_sn != d_to_split->start1) {
                    i = max_domain_m->range.s1_s_max - 1;
                    i++;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    max_domain_m->range.s1_s_max = i;

                    i = max_domain_m->range.s1_s_min - 1;
                    i--;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i > (d_to_split->start1 - 1)) {
                        i--;
                    } /*while*/
                    i++;
                    max_domain_m->range.s1_s_min = i;
                } /*if*/

                if (s1_en != d_to_split->end1) {
                    i = max_domain_m->range.s1_e_max - 1;
                    i++;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    max_domain_m->range.s1_e_max = i;

                    i = max_domain_m->range.s1_e_min - 1;
                    i--;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i > (d_to_split->start1 - 1)) {
                        i--;
                    } /*while*/
                    i++;
                    max_domain_m->range.s1_e_min = i;
                } /*if*/


                if (s2_sn != d_to_split->start2) {
                    i = max_domain_m->range.s2_s_max - 1;
                    i++;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    max_domain_m->range.s2_s_max = i;

                    i = max_domain_m->range.s2_s_min - 1;
                    i--;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i > (d_to_split->start1 - 1)) {
                        i--;
                    } /*while*/
                    i++;
                    max_domain_m->range.s2_s_min = i;
                } /*if*/

                if (s2_en != d_to_split->end2) {
                    i = max_domain_m->range.s2_e_max - 1;
                    i++;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    max_domain_m->range.s2_e_max = i;

                    i = max_domain_m->range.s2_e_min - 1;
                    i--;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i > (d_to_split->start1 - 1)) {
                        i--;
                    } /*while*/
                    i++;
                    max_domain_m->range.s2_e_min = i;
               } /*if*/
            } /*if*/
        } /*if*/
    } /*if*/

    return(0);
} /*asd_calc_res_given_sep_2m*/
