/******************************************************************************
 The computer software and associated documentation called DOMAK hereinafter
 referred to as the WORK which is more particularly identified and described in
 Appendix A of the file LICENSE.  Conditions and restrictions for use of
 this package are also in this file.

 This routine was written by Asim S. Siddiqui

 The WORK was developed by:
        Asim S. Siddiqui and Geoffrey J. Barton
        Laboratory of Molecular Biophysics
        University of Oxford
        Rex Richards Building
        South Parks Road
        Oxford OX1 3QU U.K.
        Tel:  (+44) 865-275379
        FAX:  (+44) 865-510454
        INTERNET: as@bioch.ox.ac.uk
        JANET:    as@uk.ac.ox.bioch

 The WORK is Copyright (1995) University of Oxford
        Administrative Offices
        Wellington Square
        Oxford OX1 2JD U.K.

 All use of the WORK must cite:
 Siddiqui, A. S. and Barton, G. J., "Continuous and Discontinuous Domains: An
 Algorithm for the Automatic Generation of Reliable Protein Domain Definitions" 
 PROTEIN SCIENCE, 4:872-884 (1995).
*****************************************************************************/

/*
 * Name
 *   asd_mds_utils.c
 * Purpose
 *   to provide a set of utility routines for the make_domains routines
 * Author
 *   Asim Siddiqui
 * SccsId
 *   %W%   %U%   %E%
 */

#include <rdssp.h>
#include <asd_structs.h>
#include <asm_mop.h>
#include <asd_utils.h>
#include <local.h>
#include <ase_error.h>
#include <asd_value.h>
#include <asd_make_domains.h>
#include <asd_make_domains_2.h>
#include <asd_mds_utils2.h>

extern Asd_Parameters params;

/*
 * Name
 *   asd_calc_res_from_scratch_s
 * Purpose
 *   calculate the contacts form scratch for a single cut senario
 */
int
asd_calc_res_from_scratch_s(Asd_Contact_Info **c1,
                            Asd_Contact_Info **c2,
                            Asd_Domain *d_to_split, Asd_Domain *max_domain_s,
                            Asd_Seg_Cont *s_conts, bool_t ss_only, char *ss,
                            int start_chain, int end_chain,
                            int s1_s, int s1_e, Asd_Contact_List *c_list)
{
    int i;                      /* loop counter                     */
    int j;                      /* loop counter                     */
    int num_contacts;           /* number of contacts for that item */
    Asd_Contact_Info **cf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info *cf2_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info **cr1_ptr; /* temporary pointer to reverse contacts */
    Asd_Contact_Info *cr2_ptr; /* temporary pointer to reverse contacts */
    Asd_Contact_Info **cbf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info **cbr1_ptr; /* temporary pointer to r b     contacts */
    Asd_Contact_Info **contact_info;  /* beta sheet list                 */
    Asd_Contact_Info **contact_rev_info;  /* reverse beta sheet list     */
    Asd_Contact_Info **beta_l;  /* beta sheet list                       */
    Asd_Contact_Info **rev_beta_l;  /* reverse beta sheet list           */

    contact_info = c_list->cl;
    contact_rev_info = c_list->rcl;
    beta_l = c_list->bl;
    rev_beta_l = c_list->rbl;

    s_conts->s1_int = 0;
    s_conts->s1_ext = 0;
    s_conts->s1_sht_int = 0;
    s_conts->s1_sht_ext = 0;

/* scan s1 to work out the 4 variables above */
    i = s1_s;
    cf1_ptr = (contact_info + s1_s);
    cr1_ptr = (contact_rev_info + s1_s);
    cbf1_ptr = (beta_l + s1_s);
    cbr1_ptr = (rev_beta_l + s1_s);

    while (i <= s1_e) {
        if (!ss_only ||
            (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
/* first analyse forward contacts */
            cf2_ptr = *cf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s1_s && (*cf2_ptr).res_num <= s1_e) {
                    (s_conts->s1_int)++;
                } else {
                    (s_conts->s1_ext)++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* first analyse forward beta contacts */
            cf2_ptr = *cbf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s1_s && (*cf2_ptr).res_num <= s1_e) {
                    (s_conts->s1_sht_int)++;
                } else {
                    (s_conts->s1_sht_ext)++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* now do reverse contacts - no need to count internal ones again */
            cr2_ptr = *cr1_ptr;
            num_contacts = (*cr2_ptr).res_num;
            cr2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cr2_ptr).res_num < s1_s || (*cr2_ptr).res_num > s1_e) {
                    (s_conts->s1_ext)++;
                } /*if*/
                j++;
                cr2_ptr++;
            } /*while*/
/* now do reverse beta contacts - no need to count internal ones again */
            cr2_ptr = *cbr1_ptr;
            num_contacts = (*cr2_ptr).res_num;
            cr2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cr2_ptr).res_num < s1_s || (*cr2_ptr).res_num > s1_e) {
                    (s_conts->s1_sht_ext)++;
                } /*if*/
                j++;
                cr2_ptr++;
            } /*while*/
        } /*if*/
        i++;
        cf1_ptr = (cf1_ptr + 1);
        cr1_ptr = (cr1_ptr + 1);
        cbf1_ptr = (cbf1_ptr + 1);
        cbr1_ptr = (cbr1_ptr + 1);
    } /*while*/

    return(0);
} /*asd_calc_res_from_scratch_s*/

/*
 * Name
 *   asd_calc_res_given_s
 * Purpose
 *   given an a change in s2_e calculate the new contacts for single cut senario */
int
asd_calc_res_given_s(Asd_Contact_Info **c1,
                     Asd_Contact_Info **c2,
                     Asd_Domain *d_to_split, Asd_Domain *max_domain_s,
                     Asd_Seg_Cont *s_conts, bool_t ss_only, char *ss,
                     int start_chain, int end_chain,
                     int s1_sn, int s1_en, int s1_s,  int s1_e,
                     int total_contacts, Asd_Contact_List *c_list,
                     int increment, struct brookn *bn)
{
    int i;                      /* loop counter                          */
    int j;                      /* loop counter                          */
    int num_contacts;           /* number of contacts for that item      */
    int int_1;                  /* number of internal contacts           */
    int int_2;                  /* number of internal contacts           */
    int size1;                  /* size of segment                       */
    int size2;                  /* size of segment                       */
    float ext_no;               /* number of external contacts           */
    int ext_sht;                /* external sheet contacts               */
    float value;                /* temporary value                       */
    Asd_Domain d_left_over[2];  /* domain left over                      */
    Asd_Contact_Info **cf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info *cf2_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info **cr1_ptr; /* temporary pointer to reverse contacts */
    Asd_Contact_Info *cr2_ptr; /* temporary pointer to reverse contacts */
    Asd_Contact_Info **cbf1_ptr; /* temporary pointer to forward contacts */
    Asd_Contact_Info **cbr1_ptr; /* temporary pointer to r b     contacts */
    Asd_Contact_Info **contact_info;  /* beta sheet list                 */
    Asd_Contact_Info **contact_rev_info;  /* reverse beta sheet list     */
    Asd_Contact_Info **beta_l;  /* beta sheet list                       */
    Asd_Contact_Info **rev_beta_l;  /* reverse beta sheet list           */

    contact_info = c_list->cl;
    contact_rev_info = c_list->rcl;
    beta_l = c_list->bl;
    rev_beta_l = c_list->rbl;

    if (s1_sn != s1_s || s1_en < s1_e) {
        ase_error_fatal("asd_calc_res_given_s", "function called incorrectly");
    } /*if*/

    i = s1_e + 1;
    cf1_ptr = (contact_info + (s1_e + 1));
    cr1_ptr = (contact_rev_info + (s1_e + 1));;
    cbf1_ptr = (beta_l + (s1_e + 1));
    cbr1_ptr = (rev_beta_l + (s1_e + 1));
    while (i <= s1_en) {
        if (!ss_only ||
            (ss_only && (ss[i - 1] == 'H' || ss[i - 1] == 'E'))) {
/* first analyse forward contacts */
            cf2_ptr = *cf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s1_sn && (*cf2_ptr).res_num <=
                                                                        s1_en) {
                    (s_conts->s1_int)++;
                } else {
                    (s_conts->s1_ext)++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* first analyse forward beta contacts */
            cf2_ptr = *cbf1_ptr;
            num_contacts = (*cf2_ptr).res_num;
            cf2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cf2_ptr).res_num >= s1_sn && (*cf2_ptr).res_num <=
                                                                        s1_en) {
                    (s_conts->s1_sht_int)++;
                } else {
                    (s_conts->s1_sht_ext)++;
                } /*if*/
                j++;
                cf2_ptr++;
            } /*while*/
/* now do reverse contacts - add externals also remove add extra internals
   and remove them from externals */
            cr2_ptr = *cr1_ptr;
            num_contacts = (*cr2_ptr).res_num;
            cr2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cr2_ptr).res_num < s1_sn || (*cr2_ptr).res_num > s1_en) {
                    (s_conts->s1_ext)++;
                } else if ((*cr2_ptr).res_num >= s1_sn  &&
                           (*cr2_ptr).res_num <= s1_e) {
                    (s_conts->s1_int)++;
                    (s_conts->s1_ext)--;
                } /*if*/
                j++;
                cr2_ptr++;
            } /*while*/
/* now do reverse contacts - add externals also remove add extra internals
   and remove them from externals */
            cr2_ptr = *cbr1_ptr;
            num_contacts = (*cr2_ptr).res_num;
            cr2_ptr++;
            j = 0;
            while (j < num_contacts) {
                if ((*cr2_ptr).res_num < s1_sn || (*cr2_ptr).res_num > s1_en) {
                    (s_conts->s1_sht_ext)++;
                } else if ((*cr2_ptr).res_num >= s1_sn  &&
                           (*cr2_ptr).res_num <= s1_e) {
                    (s_conts->s1_sht_int)++;
                    (s_conts->s1_sht_ext)--;
                } /*if*/
                j++;
                cr2_ptr++;
            } /*while*/
        } /*if*/
        cf1_ptr = (cf1_ptr + 1);
        cr1_ptr = (cr1_ptr + 1);
        cbf1_ptr = (cbf1_ptr + 1);
        cbr1_ptr = (cbr1_ptr + 1);
        i++;
    } /*while*/

/* now analyse info */
    int_1 = s_conts->s1_int;
    ext_no = s_conts->s1_ext;
    int_2 = total_contacts - int_1 - ext_no;

    ext_sht = s_conts->s1_sht_ext;

    value = asd_calc_d_value(int_1, int_2, ext_no, ext_sht, d_to_split,
                             s1_sn, s1_en, 0, 0);

    size1 = s1_sn - d_to_split->start1;
    size2 = d_to_split->end1 - s1_en;

    if (size1 == 0) {
        if ((size2 < params.MIN_NO_CONTACT_CUTOFF_END && d_to_split->end1 == end_chain)
            ||
            (size2 < params.MIN_NO_CONTACT_CUTOFF_MID && d_to_split->end1 != end_chain)
            ) {
            return(0);
        } /*if*/
    } else if (size2 == 0) {
        if ((size1 < params.MIN_NO_CONTACT_CUTOFF_END && d_to_split->start1
                                                               == start_chain)
            ||
            (size1 < params.MIN_NO_CONTACT_CUTOFF_MID && d_to_split->start1
                                                               != start_chain))
            { 
            return(0);
        } /*if*/
    } else {
        if ((size1 < params.MIN_NO_CONTACT_CUTOFF_END && d_to_split->start1
                                                               == start_chain)
            ||
            (size1 < params.MIN_NO_CONTACT_CUTOFF_MID && d_to_split->start1
                                                               != start_chain)
            ||
            (size2 < params.MIN_NO_CONTACT_CUTOFF_END && d_to_split->end1 == end_chain)
            ||
            (size2 < params.MIN_NO_CONTACT_CUTOFF_MID && d_to_split->end1 != end_chain)
            ) {
            if ((size1 + size2) < params.MIN_DOMAIN_SIZE) {
                return(0);
            } else {
                d_left_over[0].start1 = d_to_split->start1;
                d_left_over[0].end1 = s1_sn - 1;
                d_left_over[1].start1 = s1_en + 1;
                d_left_over[1].end1 = d_to_split->end1;
                if (!asd_check_correlation(contact_info, contact_rev_info, bn,
                                          d_left_over, ss, 2, c_list)) {
                    return(0);
                } /*if*/
            } /*if*/
        } /*if*/
    } /*if*/

    if (value > max_domain_s->value) {
        if ((ss_only && value >= params.MIN_PEAK_SS_ONLY_C) ||
           (!ss_only && value >= params.MIN_PEAK_C) || 1) {
            max_domain_s->value = value;
            if (increment == 1) {
                max_domain_s->range.s1_s_min = s1_sn;
                max_domain_s->range.s1_e_min = s1_en;
                max_domain_s->range.s1_s_max = s1_sn;
                max_domain_s->range.s1_e_max = s1_en;
            } else {
                max_domain_s->range.s1_s_min = s1_sn - increment + 1;
                max_domain_s->range.s1_s_max = s1_sn + increment - 1;
                if (max_domain_s->range.s1_s_min < d_to_split->start1 ||
                    s1_sn == d_to_split->start1) {
                    max_domain_s->range.s1_s_min = d_to_split->start1;
                    max_domain_s->range.s1_s_max = d_to_split->start1;
                } /*if*/

                max_domain_s->range.s1_e_min = s1_en - increment + 1;
                max_domain_s->range.s1_e_max = s1_en + increment - 1;
                if (max_domain_s->range.s1_e_max > d_to_split->end1 ||
                    s1_en == d_to_split->end1) {
                    max_domain_s->range.s1_e_max = d_to_split->end1;
                    max_domain_s->range.s1_e_min = d_to_split->end1;
                } /*if*/
            } /*if*/

            if (ss_only) {
                if (s1_sn != d_to_split->start1) {
                    i = max_domain_s->range.s1_s_max - 1;
                    i++;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    max_domain_s->range.s1_s_max = i;

                    i = max_domain_s->range.s1_s_min - 1;
                    i--;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i > (d_to_split->start1 - 1)) {
                        i--;
                    } /*while*/
                    i++;
                    max_domain_s->range.s1_s_min = i;
                } /*if*/

                if (s1_en != d_to_split->end1) {
                    i = max_domain_s->range.s1_e_max - 1;
                    i++;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i < d_to_split->end1) {
                        i++;
                    } /*while*/
                    max_domain_s->range.s1_e_max = i;

                    i = max_domain_s->range.s1_e_min - 1;
                    i--;
                    while (ss[i] != 'H' &&
                           ss[i] != 'E' && i > (d_to_split->start1 - 1)) {
                       i--;
                    } /*while*/
                    i++;
                    max_domain_s->range.s1_e_min = i;
                } /*if*/
            } /*if*/
        } /*if*/
    } /*if*/
    return(0);
} /*asd_calc_res_given_s*/
