/******************************************************************************
 The computer software and associated documentation called DOMAK hereinafter
 referred to as the WORK which is more particularly identified and described in
 Appendix A of the file LICENSE.  Conditions and restrictions for use of
 this package are also in this file.

 This routine was developed by Asim S. Siddiqui

 The WORK was developed by:
        Asim S. Siddiqui and Geoffrey J. Barton
        Laboratory of Molecular Biophysics
        University of Oxford
        Rex Richards Building
        South Parks Road
        Oxford OX1 3QU U.K.
        Tel:  (+44) 865-275379
        FAX:  (+44) 865-510454
        INTERNET: as@bioch.ox.ac.uk
        JANET:    as@uk.ac.ox.bioch

 The WORK is Copyright (1995) University of Oxford
        Administrative Offices
        Wellington Square
        Oxford OX1 2JD U.K.

 All use of the WORK must cite:
 Siddiqui, A. S. and Barton, G. J., "Continuous and Discontinuous Domains: An
 Algorithm for the Automatic Generation of Reliable Protein Domain Definitions" 
 PROTEIN SCIENCE, 4:872-884 (1995).
*****************************************************************************/

/*
 * Title
 *    ass_stamp_utils.c
 * Purpose
 *    utilities to inyeract with stamp file
 * SccsId
 *    %W%  %U%   %E%
 */

#include <stdio.h>
#include <gjutil.h>
#include <ase_error.h>
#include <asm_mop.h>
#include <ass_stamp_utils.h>
#include <stdlib.h>
#include <string.h>
#include <rdssp.h>
#include <asstr_util.h>

extern Asd_Parameters params;

static char env[] = "$STAMPDIR/stamp.defaults";

struct domain_loc *
ass_read_stamp(char *stamp_file_name, int *num_domains)
{
    FILE *fptr;                /* file pointer                */
    int i;                     /* loop counter                */
    int j;                     /* loop counter                */
    int flag;                  /* flag                        */
    struct domain_loc *domain; /* pointer to domain structure */

    fptr = GJfopen(stamp_file_name, "r", 1);

    *num_domains = count_domain(fptr);
    rewind(fptr);
    domain = (struct domain_loc *)
              asm_malloc(*num_domains * sizeof(struct domain_loc));

    RBR_getdomain(fptr, domain, &i, *num_domains, &flag, env, 0, stdout);

    if (i != *num_domains) {
        ase_error_fatal("main", "error in getdomain");
    } /*if*/

/* check all domains */
    flag = 0;
    while (i < *num_domains) {
        j = 0;
        while (j < domain[i].nobj) {
            if (domain[i].type[j] == 0) {
                flag = 1;
            } /*if*/
            j++;
        } /*while*/
        i++;
        if (flag == 1) {
            ase_error_fatal("main", "error in reading domains");
        } /*if*/
    } /*while*/

    return (domain);
} /*ass_read_stamp*/

char *code;     /* store pdb code */
char *pdb_file; /* store pdb file */

/*
 * name
 *    ass_set_code
 * purpose
 *    set pdb code variable
 */
void
ass_set_code(char *code_i)
{
    code = asstr_save(code_i);
} /*ass_set_code*/

/*
 * name
 *    ass_set_pdb_file
 * purpose
 *    set pdb file variable
 */
void
ass_set_pdb_file(char *pdb_file_i)
{
    pdb_file = asstr_save(pdb_file_i);
} /*ass_set_pdb_file*/

#define BUFFSIZE 200

/*
 * name
 *    ass_write_domain_defn
 * purpose
 *    to write the domain description to a string
 */
char *
ass_write_domain_defn(Asd_Domain *domain, struct brookn *bn)
{
    char cid_s, cid_e, in_s, in_e;
    int start, end;
    char *d_string;
    char buff[BUFFSIZE];

    start = domain->start1 - 1;
    end = domain->end1 - 1;

    if (bn[start].cid == ' ') {
        cid_s = '_';
    } else {
        cid_s = bn[start].cid;
    } /*if*/
    if (bn[end].cid == ' ') {
        cid_e = '_';
    } else {
        cid_e = bn[end].cid;
    } /*if*/
    if (bn[start].in == ' ') {
        in_s = '_';
    } else {
        in_s = bn[start].in;
    } /*if*/
    if (bn[end].in == ' ') {
        in_e = '_';
    } else {
        in_e = bn[end].in;
    } /*if*/

    if (sprintf(buff, "%c %d %c to %c %d %c ",
                cid_s, bn[start].n, in_s,
                cid_e, bn[end].n, in_e) == -1) {
        ase_error_fatal("ass_write_domain_string", "overflow");
    } /*if*/

    d_string = asstr_save(buff);

    if (domain->type == 2) {
        start = domain->start2 - 1;
        end = domain->end2 - 1;
        if (bn[start].cid == ' ') {
            cid_s = '_';
        } else {
            cid_s = bn[start].cid;
        } /*if*/
        if (bn[end].cid == ' ') {
            cid_e = '_';
        } else {
            cid_e = bn[end].cid;
        } /*if*/
        if (bn[start].in == ' ') {
            in_s = '_';
        } else {
            in_s = bn[start].in;
        } /*if*/
        if (bn[end].in == ' ') {
            in_e = '_';
        } else {
            in_e = bn[end].in;
        } /*if*/

        if (sprintf(buff, "%c %d %c to %c %d %c ",
                    cid_s, bn[start].n, in_s,
                    cid_e, bn[end].n, in_e) == -1) {
            ase_error_fatal("ass_write_domain_string", "overflow");
        } /*if*/

        d_string = asstr_cat_safe(d_string, buff);
    } /*if*/

    return(d_string);
} /*ass_write_domain_defn*/

/*
 * name
 *    asd_d_comments
 * purpose
 *    to write a comment out to user about what is happening
 */
void
asd_d_comments(int comm_num, Asd_Domain *domain, struct brookn *bn)
{

    if (!(domain->ok) && comm_num != 3) {
        return;
    } /*if*/

    fprintf(stdout, "\n");
    fprintf(stdout, "%s", ass_write_domain_string(domain, bn));
    if (comm_num == 1) {
        fprintf(stdout,"# %s This part is larger than MIN_NO_CONTACT_CUTOFF,\n\
# %s but smaller than MIN_DOMAIN_SIZE\n", domain->id, domain->id);
        fprintf(stdout, "# %s It will be considered as a \"bit left over\"\n",
                domain->id);
    } else if (comm_num == 2) {
        fprintf(stdout,
                "# %s This domain is too small to be subdivided further\n",
                 domain->id);
    } else if (comm_num == 3) {
        if (domain->ok) {
            fprintf(stdout,
                    "# %s Split value too small to allow futher splitting\n",
                    domain->id);
         } else {
            fprintf(stdout, "# %s No suitable sub-domains were found\n",
                    domain->id);
         } /*if*/
    } else if (comm_num == 4) {
        fprintf(stdout,
                "# %s This is the domain that would have been extracted\n",
                domain->id);
        fprintf(stdout, "# %s had the previous domain been split\n",
                domain->id);
        fprintf(stdout, "# %s with split value of %f\n", domain->id,
                domain->value);
    } else if (comm_num == 5) {
        fprintf(stdout,"# %s The above domain has been extracted\n\
# %s with split value %f\n", domain->id, domain->id, domain->value);
    } else if (comm_num == 6) {
        fprintf(stdout, "# %s Domain currently being analysed\n", domain->id);
    } else {
        ase_error_warn("Unrecognised comment type");
    } /*if*/

    fflush(stdout);
} /*asd_d_comments*/

/*
 * name
 *    ass_bit_comments
 * purpose
 *    to write how bits left over have been assigned
 */
void
ass_bit_comments(Asd_Domain *bit, Asd_Domain *domain, struct brookn *bn)
{
    fprintf(stdout, "\n");
    fprintf(stdout, "%s", ass_write_domain_string(bit, bn));
    fprintf(stdout, "# %s This \"bit left over\" has been assigned to domain\n",
                     bit->id);
    fprintf(stdout, "%s", ass_write_domain_string(domain, bn));
    fflush(stdout);
} /*ass_bit_comments*/

/*
 * name
 *    ass_glob_comments
 * purpose
 *    to write about globularity of domains
 */
void
ass_glob_comments(int comm_num, Asd_Domain *domain, struct brookn *bn)
{
    if (comm_num == 1) {
        fprintf(stdout, "\n");
    } /*if*/
    fprintf(stdout, "%s", ass_write_domain_string(domain, bn));

    if (comm_num == 1) {
        fprintf(stdout,
                "# %s Analysing compactness of sub-domains of above domain\n",
                domain->id);
    } else if (comm_num == 2) {
        fprintf(stdout, "# %s Compactness of domain %f\n", domain->id,
                domain->glob);
    } else if (comm_num == 3) {
        fprintf(stdout, "# %s Average compactness > MAX_ALLOWABLE_GLOB\n",
                domain->id);
        fprintf(stdout, "# %s Reverting to parent domain above\n",domain->id);
    } else if (comm_num == 4) {
        fprintf(stdout,
           "# %s One of the sub-domains has compactness > MAX_ALLOWABLE_GLOB\n",
           domain->id);
        fprintf(stdout, "# %s Reverting to parent domain above\n", domain->id);
    } else if (comm_num == 5) {
        fprintf(stdout, "# %s Average compactness > MAX_ALLOWABLE_GLOB\n",
                domain->id);
    } else if (comm_num == 6) {
        fprintf(stdout, "# Combined with domain %s\n", domain->id);
    } else {
        ase_error_warn("Unrecognised comment type");
    } /*if*/

    fflush(stdout);
} /*ass_glob_comments*/
        

/*
 * name
 *    ass_write_domain_string
 * purpose
 *    write domain string out in STAMP format
 */
char *
ass_write_domain_string(Asd_Domain *domain, struct brookn *bn)
{
    char cid_s, cid_e, in_s, in_e;
    int start, end;
    char *d_string;
    char buff[BUFFSIZE];

    if (sprintf(buff, "%s %s%s { ", pdb_file, code, domain->id) == -1) {
        ase_error_fatal("ass_write_domain_string", "overflow");
    } /*if*/

    d_string = asstr_save(buff);

    d_string = asstr_cat_safe(d_string, ass_write_domain_defn(domain, bn));

    sprintf(buff, "}\n");
    d_string = asstr_cat_safe(d_string, buff);

    return(d_string);
} /*ass_write_domain_string*/

/*
 * name
 *    ass_write_domain_defn_with_all
 * purpose
 *    to write out domain defns
 */
char *
ass_write_domain_defn_with_all(Asd_Domain_List *d_list, int d_num,
                               struct brookn *bn)
{
    char *d_string;
    int i; /* loop counter */
    char buff[BUFFSIZE];

    if (sprintf(buff, "%s %s%s { ", pdb_file, code, d_list->domains[d_num].id)
                                                                        == -1) {
        ase_error_fatal("ass_write_domain_string", "overflow");
    } /*if*/

    d_string = asstr_save(buff);

    d_string = asstr_cat_safe(d_string,
                              ass_write_domain_defn(&(d_list->domains[d_num]),
                                                                      bn));

    i = 0;
    while (i < d_list->n_d_list) {
        if (d_list->domains[i].d_assigned == d_num) {
            d_string = asstr_cat_safe(d_string,
                       ass_write_domain_defn(&(d_list->domains[i]), bn));
        } /*if*/
        i++;
    } /*while*/

    i = 0;
    while (i < d_list->n_b_left_over) {
        if (d_list->bits_left_over[i].d_assigned == d_num) {
            d_string = asstr_cat_safe(d_string,
                       ass_write_domain_defn(&(d_list->bits_left_over[i]), bn));
        } /*if*/
        i++;
    } /*while*/

    sprintf(buff, "}\n");
    d_string = asstr_cat_safe(d_string, buff);

    return(d_string);
} /*ass_write_domain_defn_with_all*/


/*
 * name
 *    ass_write_domain_string_with_bit
 * purpose
 *    write domain string out in STAMP format
 */
char *
ass_write_domain_string_with_bit(Asd_Domain *domain, Asd_Domain *bit,
                                 struct brookn *bn)
{
    char cid_s, cid_e, in_s, in_e;
    int start, end;
    char *d_string;
    char buff[BUFFSIZE];

    if (sprintf(buff, "%s %s%s { ", pdb_file, code, domain->id) == -1) {
        ase_error_fatal("ass_write_domain_string", "overflow");
    } /*if*/

    d_string = asstr_save(buff);

    d_string = asstr_cat_safe(d_string, ass_write_domain_defn(domain, bn));
    d_string = asstr_cat_safe(d_string, ass_write_domain_defn(bit, bn));

    sprintf(buff, "}\n");
    d_string = asstr_cat_safe(d_string, buff);

    return(d_string);
} /*ass_write_domain_string_with_bit*/

#define MAX_COLOURS 10

/*
 * name
 *    asd_print_domains
 * purpose
 *    to write out domains in STAMP format to standard output
 */
void
asd_print_domains(Asd_Domain_List *d_list, struct brookn *bn, FILE *fptr,
                  FILE *rasmol_fptr, char cid)
{
    int i; /* loop counter */
    int j; /* loop counter */
    int err; /* whether error has occured */
    Asd_Domain *domain; /* current domain */
    char cid_s, cid_e, in_s, in_e;
    int start, end;
    static char *colours[MAX_COLOURS] = { "red", "green", "blue", "yellow",
                                 "purple", "cyan", "magenta", "greenblue",
                                 "violet", "redorange"};

    fprintf(fptr, "\nDOMAK DOMAINS\n");
    if (rasmol_fptr != NULL) {
        fprintf(rasmol_fptr, "#\n# Domains of %s\n#\n", code);
        if (d_list->n_d_list > 10) {
            fprintf(rasmol_fptr, "# WARNING, too many domains, some overlap\
 in colours\n");
        } /*if*/
        fprintf(rasmol_fptr, "load pdb%c%c%c%c.ent\n", code[0], code[1],
                                                       code[2], code[3]);
        if (cid != '*') {
            fprintf(rasmol_fptr, "restrict *:%c.*\n", cid);
        } /*if*/
        fprintf(rasmol_fptr, "wireframe off\n");
        fprintf(rasmol_fptr, "backbone 80\n");
    } /*if*/
    i = 0;
    while (i < d_list->n_d_list) {
        if (d_list->domains[i].d_assigned == -2) {
            err = 0;
            domain = &(d_list->domains[i]);
            start = domain->start1 - 1;
            end = domain->end1 - 1;

            if (fprintf(fptr, "%s %s%s { ", pdb_file, code, domain->id) == -1) {
                ase_error_fatal("asd_print_domains", "overflow");
            } /*if*/

            if (bn[start].cid == ' ') {
                cid_s = '_';
            } else {
                cid_s = bn[start].cid;
            } /*if*/
            if (bn[end].cid == ' ') {
                cid_e = '_';
            } else {
                cid_e = bn[end].cid;
            } /*if*/
            if (bn[start].in == ' ') {
                in_s = '_';
            } else {
                in_s = bn[start].in;
            } /*if*/
            if (bn[end].in == ' ') {
                in_e = '_';
            } else {
                in_e = bn[end].in;
            } /*if*/

            if (cid_s != cid_e || bn[start].n >= bn[end].n) {
                err = 1;
            } /*if*/

            if (fprintf(fptr, "%c %d %c to %c %d %c ",
                    cid_s, bn[start].n, in_s,
                    cid_e, bn[end].n, in_e) == -1) {
                ase_error_fatal("asd_print_domains", "overflow");
            } /*if*/

            if (rasmol_fptr != NULL) {
                fprintf(rasmol_fptr, "# domain %d\n", (i + 1));
                fprintf(rasmol_fptr, "restrict %d-%d\n", bn[start].n,
                                                         bn[end].n);
                fprintf(rasmol_fptr, "colour %s\n", colours[i % MAX_COLOURS]);
            } /*if*/

            if (domain->type == 2) {
                start = domain->start2 - 1;
                end = domain->end2 - 1;

                if (bn[start].cid == ' ') {
                    cid_s = '_';
                } else {
                    cid_s = bn[start].cid;
                } /*if*/
                if (bn[end].cid == ' ') {
                    cid_e = '_';
                } else {
                    cid_e = bn[end].cid;
                } /*if*/
                if (bn[start].in == ' ') {
                    in_s = '_';
                } else {
                    in_s = bn[start].in;
                } /*if*/
                if (bn[end].in == ' ') {
                    in_e = '_';
                } else {
                    in_e = bn[end].in;
                } /*if*/

                if (fprintf(fptr, "%c %d %c to %c %d %c ",
                            cid_s, bn[start].n, in_s,
                            cid_e, bn[end].n, in_e) == -1) {
                    ase_error_fatal("asd_print_domains", "overflow");
                } /*if*/

                if (rasmol_fptr != NULL) {
                    fprintf(rasmol_fptr, "restrict %d-%d\n", bn[start].n,
                                                             bn[end].n);
                    fprintf(rasmol_fptr, "colour %s\n",
                            colours[i % MAX_COLOURS]);
                } /*if*/

                if (cid_s != cid_e || bn[start].n >= bn[end].n) {
                    err = 1;
                } /*if*/

            } /*if*/

/* now do assigned domains */
            j = 0;
            while (j < d_list->n_d_list) {
                domain = &(d_list->domains[j]);

                if (domain->d_assigned == i) {
                    start = domain->start1 - 1;
                    end = domain->end1 - 1;
                    if (bn[start].cid == ' ') {
                        cid_s = '_';
                    } else {
                        cid_s = bn[start].cid;
                    } /*if*/
                    if (bn[end].cid == ' ') {
                        cid_e = '_';
                    } else {
                        cid_e = bn[end].cid;
                    } /*if*/
                    if (bn[start].in == ' ') {
                        in_s = '_';
                    } else {
                        in_s = bn[start].in;
                    } /*if*/
                    if (bn[end].in == ' ') {
                        in_e = '_';
                    } else {
                        in_e = bn[end].in;
                    } /*if*/

                    if (fprintf(fptr, "%c %d %c to %c %d %c ",
                            cid_s, bn[start].n, in_s,
                            cid_e, bn[end].n, in_e) == -1) {
                        ase_error_fatal("asd_print_domains", "overflow");
                    } /*if*/

                    if (rasmol_fptr != NULL) {
                        fprintf(rasmol_fptr, "restrict %d-%d\n", bn[start].n,
                                                                 bn[end].n);
                        fprintf(rasmol_fptr, "colour %s\n",
                                colours[i % MAX_COLOURS]);
                    } /*if*/

                    if (cid_s != cid_e || bn[start].n >= bn[end].n) {
                        err = 1;
                    } /*if*/

                    if (domain->type == 2) {
                        start = domain->start2 - 1;
                        end = domain->end2 - 1;

                        if (bn[start].cid == ' ') {
                            cid_s = '_';
                        } else {
                            cid_s = bn[start].cid;
                        } /*if*/
                        if (bn[end].cid == ' ') {
                            cid_e = '_';
                        } else {
                            cid_e = bn[end].cid;
                        } /*if*/
                        if (bn[start].in == ' ') {
                            in_s = '_';
                        } else {
                            in_s = bn[start].in;
                        } /*if*/
                        if (bn[end].in == ' ') {
                            in_e = '_';
                        } else {
                            in_e = bn[end].in;
                        } /*if*/

                        if (fprintf(fptr, "%c %d %c to %c %d %c ",
                                    cid_s, bn[start].n, in_s,
                                    cid_e, bn[end].n, in_e) == -1) {
                            ase_error_fatal("asd_print_domains", "overflow");
                        } /*if*/

                        if (rasmol_fptr != NULL) {
                            fprintf(rasmol_fptr, "restrict %d-%d\n",
                                                 bn[start].n, bn[end].n);
                            fprintf(rasmol_fptr, "colour %s\n",
                                    colours[i % MAX_COLOURS]);
                        } /*if*/

                        if (cid_s != cid_e || bn[start].n >= bn[end].n) {
                            err = 1;
                        } /*if*/

                    } /*if*/

                } /*if*/
                j++;
            } /*while*/

/* now do bits left over */
            j = 0;
            while (j < d_list->n_b_left_over) {
                domain = &(d_list->bits_left_over[j]);

                if (domain->d_assigned == i) {
                    start = domain->start1 - 1;
                    end = domain->end1 - 1;
                    if (bn[start].cid == ' ') {
                        cid_s = '_';
                    } else {
                        cid_s = bn[start].cid;
                    } /*if*/
                    if (bn[end].cid == ' ') {
                        cid_e = '_';
                    } else {
                        cid_e = bn[end].cid;
                    } /*if*/
                    if (bn[start].in == ' ') {
                        in_s = '_';
                    } else {
                        in_s = bn[start].in;
                    } /*if*/
                    if (bn[end].in == ' ') {
                        in_e = '_';
                    } else {
                        in_e = bn[end].in;
                    } /*if*/

                    if (fprintf(fptr, "%c %d %c to %c %d %c ",
                            cid_s, bn[start].n, in_s,
                            cid_e, bn[end].n, in_e) == -1) {
                        ase_error_fatal("asd_print_domains", "overflow");
                    } /*if*/

                    if (rasmol_fptr != NULL) {
                        fprintf(rasmol_fptr, "restrict %d-%d\n", bn[start].n,
                                                                 bn[end].n);
                        fprintf(rasmol_fptr, "colour %s\n",
                                colours[i % MAX_COLOURS]);
                    } /*if*/

                    if (cid_s != cid_e || bn[start].n >= bn[end].n) {
                        err = 1;
                    } /*if*/

                } /*if*/
                j++;
            } /*while*/

            fprintf(fptr, "}\n");

            if (err == 1) {
                ase_error_warn("domain has a strange format check pdb file");
            } /*if*/
        } /*if*/
        i++;
    } /*while*/ 
} /*asd_print_domains*/
